"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphQLApi = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
const data_source_1 = require("./data-source");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: this.formatAdditionalAuthenticationProviders(props),
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (defaultAuthorizationType === AuthorizationType.API_KEY || ((_h = (_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) === null || _h === void 0 ? void 0 : _h.some((authMode) => authMode.authorizationType === AuthorizationType.API_KEY))) {
            const apiKeyConfig = ((_k = (_j = props.authorizationConfig) === null || _j === void 0 ? void 0 : _j.defaultAuthorization) === null || _k === void 0 ? void 0 : _k.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this.createAPIKey(apiKeyConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new data_source_1.NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new data_source_1.DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new data_source_1.HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new data_source_1.LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
    formatAdditionalAuthenticationProviders(props) {
        var _a;
        const authModes = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.additionalAuthorizationModes;
        return authModes ? this.formatAdditionalAuthorizationModes(authModes) : undefined;
    }
}
exports.GraphQLApi = GraphQLApi;
//# sourceMappingURL=data:application/json;base64,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