import { IUserPool } from "../../aws-cognito";
import { ITable } from "../../aws-dynamodb";
import { IFunction } from "../../aws-lambda";
import { Construct, IResolvable } from "../../core";
import { CfnGraphQLSchema } from './appsync.generated';
import { DynamoDbDataSource, HttpDataSource, LambdaDataSource, NoneDataSource } from './data-source';
/**
 * enum with all possible values for AppSync authorization type
 */
export declare enum AuthorizationType {
    /**
     * API Key authorization type
     */
    API_KEY = "API_KEY",
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    IAM = "AWS_IAM",
    /**
     * Cognito User Pool authorization type
     */
    USER_POOL = "AMAZON_COGNITO_USER_POOLS",
    /**
     * OpenID Connect authorization type
     */
    OIDC = "OPENID_CONNECT"
}
/**
 * Interface to specify default or additional authorization(s)
 */
export interface AuthorizationMode {
    /**
     * One of possible four values AppSync supports
     *
     * @see https://docs.aws.amazon.com/appsync/latest/devguide/security.html
     *
     * @default - `AuthorizationType.API_KEY`
     */
    readonly authorizationType: AuthorizationType;
    /**
     * If authorizationType is `AuthorizationType.USER_POOL`, this option is required.
     * @default - none
     */
    readonly userPoolConfig?: UserPoolConfig;
    /**
     * If authorizationType is `AuthorizationType.API_KEY`, this option can be configured.
     * @default - check default values of `ApiKeyConfig` memebers
     */
    readonly apiKeyConfig?: ApiKeyConfig;
    /**
     * If authorizationType is `AuthorizationType.OIDC`, this option is required.
     * @default - none
     */
    readonly openIdConnectConfig?: OpenIdConnectConfig;
}
/**
 * enum with all possible values for Cognito user-pool default actions
 */
export declare enum UserPoolDefaultAction {
    /**
     * ALLOW access to API
     */
    ALLOW = "ALLOW",
    /**
     * DENY access to API
     */
    DENY = "DENY"
}
/**
 * Configuration for Cognito user-pools in AppSync
 */
export interface UserPoolConfig {
    /**
     * The Cognito user pool to use as identity source
     */
    readonly userPool: IUserPool;
    /**
     * the optional app id regex
     *
     * @default -  None
     */
    readonly appIdClientRegex?: string;
    /**
     * Default auth action
     *
     * @default ALLOW
     */
    readonly defaultAction?: UserPoolDefaultAction;
}
/**
 * Configuration for API Key authorization in AppSync
 */
export interface ApiKeyConfig {
    /**
     * Unique name of the API Key
     * @default - 'DefaultAPIKey'
     */
    readonly name?: string;
    /**
     * Description of API key
     * @default - 'Default API Key created by CDK'
     */
    readonly description?: string;
    /**
     * The time from creation time after which the API key expires, using RFC3339 representation.
     * It must be a minimum of 1 day and a maximum of 365 days from date of creation.
     * Rounded down to the nearest hour.
     * @default - 7 days from creation time
     */
    readonly expires?: string;
}
/**
 * Configuration for OpenID Connect authorization in AppSync
 */
export interface OpenIdConnectConfig {
    /**
     * The number of milliseconds an OIDC token is valid after being authenticated by OIDC provider.
     * `auth_time` claim in OIDC token is required for this validation to work.
     * @default - no validation
     */
    readonly tokenExpiryFromAuth?: number;
    /**
     * The number of milliseconds an OIDC token is valid after being issued to a user.
     * This validation uses `iat` claim of OIDC token.
     * @default - no validation
     */
    readonly tokenExpiryFromIssue?: number;
    /**
     * The client identifier of the Relying party at the OpenID identity provider.
     * A regular expression can be specified so AppSync can validate against multiple client identifiers at a time.
     * @example - 'ABCD|CDEF' where ABCD and CDEF are two different clientId
     * @default - * (All)
     */
    readonly clientId?: string;
    /**
     * The issuer for the OIDC configuration. The issuer returned by discovery must exactly match the value of `iss` in the OIDC token.
     */
    readonly oidcProvider: string;
}
/**
 * Configuration of the API authorization modes.
 */
export interface AuthorizationConfig {
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly defaultAuthorization?: AuthorizationMode;
    /**
     * Additional authorization modes
     *
     * @default - No other modes
     */
    readonly additionalAuthorizationModes?: AuthorizationMode[];
}
/**
 * log-level for fields in AppSync
 */
export declare enum FieldLogLevel {
    /**
     * No logging
     */
    NONE = "NONE",
    /**
     * Error logging
     */
    ERROR = "ERROR",
    /**
     * All logging
     */
    ALL = "ALL"
}
/**
 * Logging configuration for AppSync
 */
export interface LogConfig {
    /**
     * exclude verbose content
     *
     * @default false
     */
    readonly excludeVerboseContent?: boolean | IResolvable;
    /**
     * log level for fields
     *
     * @default - Use AppSync default
     */
    readonly fieldLogLevel?: FieldLogLevel;
}
/**
 * Properties for an AppSync GraphQL API
 */
export interface GraphQLApiProps {
    /**
     * the name of the GraphQL API
     */
    readonly name: string;
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly authorizationConfig?: AuthorizationConfig;
    /**
     * Logging configuration for this api
     *
     * @default - None
     */
    readonly logConfig?: LogConfig;
    /**
     * GraphQL schema definition. You have to specify a definition or a file containing one.
     *
     * @default - Use schemaDefinitionFile
     */
    readonly schemaDefinition?: string;
    /**
     * File containing the GraphQL schema definition. You have to specify a definition or a file containing one.
     *
     * @default - Use schemaDefinition
     */
    readonly schemaDefinitionFile?: string;
}
/**
 * An AppSync GraphQL API
 */
export declare class GraphQLApi extends Construct {
    /**
     * the id of the GraphQL API
     */
    readonly apiId: string;
    /**
     * the ARN of the API
     */
    readonly arn: string;
    /**
     * the URL of the endpoint created by AppSync
     */
    readonly graphQlUrl: string;
    /**
     * the name of the API
     */
    name: string;
    /**
     * underlying CFN schema resource
     */
    readonly schema: CfnGraphQLSchema;
    /**
     * the configured API key, if present
     */
    get apiKey(): string | undefined;
    private api;
    private _apiKey?;
    constructor(scope: Construct, id: string, props: GraphQLApiProps);
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name: string, description: string): NoneDataSource;
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name: string, description: string, table: ITable): DynamoDbDataSource;
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name: string, description: string, endpoint: string): HttpDataSource;
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name: string, description: string, lambdaFunction: IFunction): LambdaDataSource;
    private validateAuthorizationProps;
    private formatOpenIdConnectConfig;
    private formatUserPoolConfig;
    private createAPIKey;
    private formatAdditionalAuthorizationModes;
    private formatAdditionalAuthenticationProviders;
}
