"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpApi = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const route_1 = require("./route");
const stage_1 = require("./stage");
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
class HttpApi extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const apiName = (_a = props === null || props === void 0 ? void 0 : props.apiName) !== null && _a !== void 0 ? _a : id;
        let corsConfiguration;
        if (props === null || props === void 0 ? void 0 : props.corsPreflight) {
            const { allowCredentials, allowHeaders, allowMethods, allowOrigins, exposeHeaders, maxAge, } = props.corsPreflight;
            corsConfiguration = {
                allowCredentials,
                allowHeaders,
                allowMethods,
                allowOrigins,
                exposeHeaders,
                maxAge: maxAge === null || maxAge === void 0 ? void 0 : maxAge.toSeconds(),
            };
        }
        const apiProps = {
            name: apiName,
            protocolType: 'HTTP',
            corsConfiguration,
        };
        const resource = new apigatewayv2_generated_1.CfnApi(this, 'Resource', apiProps);
        this.httpApiId = resource.ref;
        if (props === null || props === void 0 ? void 0 : props.defaultIntegration) {
            new route_1.HttpRoute(this, 'DefaultRoute', {
                httpApi: this,
                routeKey: route_1.HttpRouteKey.DEFAULT,
                integration: props.defaultIntegration,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === undefined || props.createDefaultStage === true) {
            this.defaultStage = new stage_1.HttpStage(this, 'DefaultStage', {
                httpApi: this,
                autoDeploy: true,
                domainMapping: props === null || props === void 0 ? void 0 : props.defaultDomainMapping,
            });
        }
        if ((props === null || props === void 0 ? void 0 : props.createDefaultStage) === false && props.defaultDomainMapping) {
            throw new Error('defaultDomainMapping not supported with createDefaultStage disabled');
        }
    }
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope, id, httpApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.httpApiId = httpApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url() {
        return this.defaultStage ? this.defaultStage.url : undefined;
    }
    /**
     * Add a new stage.
     */
    addStage(id, options) {
        const stage = new stage_1.HttpStage(this, id, {
            httpApi: this,
            ...options,
        });
        return stage;
    }
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options) {
        var _a;
        const methods = (_a = options.methods) !== null && _a !== void 0 ? _a : [route_1.HttpMethod.ANY];
        return methods.map((method) => new route_1.HttpRoute(this, `${method}${options.path}`, {
            httpApi: this,
            routeKey: route_1.HttpRouteKey.with(options.path, method),
            integration: options.integration,
        }));
    }
}
exports.HttpApi = HttpApi;
//# sourceMappingURL=data:application/json;base64,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