"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyResource = exports.Resource = exports.ResourceBase = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
class ResourceBase extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    /**
     * @deprecated - Throws error in some use cases that have been enabled since this deprecation notice. Use `RestApi.urlForPath()` instead.
     */
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
class Resource extends ResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.api.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.api = props.parent.api;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * The RestApi associated with this Resource
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this.parentResource) {
            throw new Error('parentResource was unexpectedly not defined');
        }
        return this.parentResource.restApi;
    }
}
exports.Resource = Resource;
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 */
class ProxyResource extends Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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