# -*- coding: utf-8 -*-
import os
import pathlib
import numpy as np
import pandas as pd

class Aggregator:
    """Automate the aggregation of GC results from PLFA and NLFA samples into a single csv file for analysis.
    """
   
    def transform_raw_to_stacked(self, plfa_tools_output: pd.DataFrame) -> pd.DataFrame:
        """Reformats output from PLFA Tools, of a single sample, into proper tabular format

        :param plfa_tools_output: Output from PLFA Tools of a single worksheet in the form of a Pandas DataFrame
        :rtype: DataFrame. Header names: SampleId, GCRunId, GCFileLoc, ProcessingCode, ProcessMethod, RunDateTime, RT, Response, Ar/Ht, RFact, ECL, Peak Name, Percent, Comment1, Comment2
        """

        # Read in disjunct values
        SampleId = plfa_tools_output.iloc[2,2]
        GCRunId = plfa_tools_output.iloc[2,1]
        GCFileLoc = plfa_tools_output.iloc[0, 6]
        ProcessingCode = plfa_tools_output.iloc[0,7]
        ProcessingMethod = plfa_tools_output.iloc[3,6]
        RunDateTime = plfa_tools_output.iloc[3,7]
        
        # Copy input df, extract out table of results, set headers
        result = pd.DataFrame(plfa_tools_output)
        result.drop(result.head(5).index, inplace = True)
        result.drop(result.tail(5).index, inplace = True)
        result.columns = result.iloc[0]
        result = result[1:]
        
        # Create columns for disjunct values
        result.insert(0, 'RunDateTime', RunDateTime)
        result.insert(0, 'ProcessingMethod', ProcessingMethod)
        result.insert(0, 'ProcessingCode', ProcessingCode)
        result.insert(0, 'GCFileLoc', GCFileLoc)
        result.insert(0, 'GCRunID', GCRunId)
        result.insert(0, 'SampleID', SampleId)

        return result

    def transform_stacked_to_tidy(self, df: pd.DataFrame) -> pd.DataFrame:
        """Accepts a Pandas DataFrame generated by "read_dir" or "read_file" and returns a new DataFrame in a tidy format

        :param df: Pandas DataFrame generated by "read_dir" or "read_file"
        :rtype: DataFrame. Headers names: SampleId, GCRunId, GCFileLoc, ProcessingCode, ProcessMethod, RunDateTime, RT, Response, Ar/Ht, RFact, ECL, Peak Name, Percent, Comment1, Comment2
        """
        
        # Arrange lists of columns for dropping and adding cols as needed
        # peak_names is list of all possible categories a sample can have a measurement for
        peak_names = [
            'General FAME', 
            'AM Fungi', 
            'Gram Negative', 
            'Eukaryote', 
            'Fungi', 
            'Gram Positive', 
            'Anaerobe', 
            'Methanotroph', 
            'Actinomycetes', 
            'Protozoa'
        ]
        measurement_vars = ['SampleID', 'Peak Name', 'Response']
        drop_vars_meta = ['Peak Name', 'Response']
        drop_vars = [
            'RT', 
            'Ar/Ht', 
            'RFact', 
            'ECL', 
            'Percent', 
            'Comment1', 
            'Comment2'
        ]
        
        # Create DataFrame of just meta values to keep
        meta_df = df.drop(drop_vars, axis = 1).drop(drop_vars_meta, axis = 1)
        meta_df = meta_df.drop_duplicates()

        # Create DataFrame of measurement values to keep
        measurement_df = df[measurement_vars]
        
        # Pivot peak names to columns
        pivot_peaks = measurement_df.pivot(
            index='SampleID', 
            columns='Peak Name', 
            values='Response')

        # Add columns to resultant DataFrame as needed to match peak_names
        tidy_peaks = pd.DataFrame(columns = peak_names)
        dif_col = tidy_peaks.columns.difference(pivot_peaks.columns)
        dif_col_list = dif_col.to_list()
        for i in range(len(dif_col_list)):
            temp = dif_col_list[i]
            pivot_peaks[temp] = np.nan
        
        # Merge and return value
        tidy = pd.merge(meta_df, pivot_peaks, on = 'SampleID')

        # Order columns, for humans
        cols = [
            'SampleID', 
            'GCRunID', 
            'GCFileLoc', 
            'ProcessingCode', 
            'ProcessingMethod', 
            'RunDateTime', 
            'General FAME', 
            'AM Fungi', 
            'Gram Negative', 
            'Eukaryote', 
            'Fungi', 
            'Gram Positive', 
            'Anaerobe', 
            'Methanotroph', 
            'Actinomycetes', 
            'Protozoa'
        ]

        tidy = tidy[cols]
        
        return tidy
 
    def read_file(self, file_path: pathlib.Path) -> pd.DataFrame:
        """Reads a file from PLFA Tools and returns a pandas DataFrame with all data from all worksheets

        :param file_path: pathlib Path to a xlsx file as produced by PLFA Tools
        :rtype: DataFrame. A tidy version of Dataframe given by the filepath
        """
        
        # Read file
        xlsx = pd.ExcelFile(str(file_path))
        sheets_dict = pd.read_excel(file_path, sheet_name=None)

        # Loop through worksheets, transform raw to tidy, merge transformed sheets to single dataframe
        sheet_num = 0
        for name, sheet in sheets_dict.items():
            # Skip the first sheet, it holds metadata we're not interested in
            if sheet_num != 0:
                raw = xlsx.parse(str(name), header=None)
                stacked = self.transform_raw_to_stacked(raw)
                tidy = self.transform_stacked_to_tidy(stacked)
            
                if sheet_num == 1:
                    master = tidy
                else:
                    master = master.append(tidy)

            sheet_num += 1
        return master

    def read_dir(self, dir_path: pathlib.Path) -> pd.DataFrame:
        """Reads a directory path that contains multiple files from PLFA Tools and returns a pandas DataFrame with all data from all files and worksheets

        :param dir_path: pathlib Path to directory containing one or more xlsx files as produced by PLFA Tools
        :rtype: DataFrame. One or more Tidy DataFrames appended together into one DataFrame
        """
        # Integer for keeping track of what file the loop is on
        fileNum = 0

        for filename in os.listdir(dir_path):
            if filename.endswith('.xlsx'):
                if fileNum != 0:
                    # Turn the directory to file path
                    path_to_file = dir_path / str (filename)
                    # Send file path to be aggregated
                    tidy = self.read_file(path_to_file)
                    # Append it to what has already been aggregated so far
                    master = master.append(tidy)
                else:
                    # Conditional so the first file will go directly into the master dataframe

                    # Turn the directory into a file path
                    path_to_file = dir_path / str (filename)
                    # Send path directory to be aggregated
                    master = self.read_file(path_to_file)
            fileNum += 1    
        
        return master

