"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, slackConfig) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = slackConfig;
        this.componentBuilder = new component_builder_1.ComponentBuilder(this, componentDepsConfig, this.id, this.pipelineConfig.parent_image);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            if (slackConfig && slackConfig.channel && slackConfig.slackWebhookUrl && slackConfig.username) {
                new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, slackConfig, `${this.id}Recipe`);
            }
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let compMap = this.componentBuilder.createComponentDependenciesMap();
        let image_id;
        let parentImage;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.pipelineConfig.recipe.components.map((c) => ({
                componentArn: compMap[c.name].ref,
            })),
            blockDeviceMappings: blockDeviceMappings,
            tags: {
                ShortName: `${this.id}Recipe`,
            },
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.pipelineConfig.shared_with ? new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: {
                        Name: this.id,
                    },
                    TargetAccountIds: s.account_ids,
                },
            })),
        }) : undefined;
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : undefined;
        this.ebsEncryptionKey = key;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp2',
                    kmsKeyId: key === null || key === void 0 ? void 0 : key.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let envVariables = {
                IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                PIPELINE_NAME: { value: this.imagePipeline.name },
            };
            if (this.slackConfig && this.slackConfig.slackWebhookUrl && this.slackConfig.channel && this.slackConfig.username) {
                envVariables.SLACK_WEBHOOK_URL = { value: this.slackConfig.slackWebhookUrl };
                envVariables.SLACK_CHANNEL = { value: this.slackConfig.channel };
                envVariables.SLACK_USERNAME = { value: this.slackConfig.username };
            }
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'if [ -n "$SLACK_WEBHOOK_URL" ]; then curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL; fi',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: envVariables,
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: this.pipelineConfig.name,
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1pX3BpcGVsaW5lX2xpYi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9hbWlfcGlwZWxpbmVfbGliLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsb0RBQW9EO0FBQ3BELDBEQUEwRDtBQUMxRCwwRUFBMEU7QUFDMUUsd0NBQXdDO0FBQ3hDLDhDQUE4QztBQUM5Qyw2REFBNkQ7QUFDN0Qsd0NBQXdDO0FBQ3hDLDBEQUEwRDtBQUMxRCx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHFDQUFxQztBQUdyQywyREFBdUQ7QUFDdkQsMkVBQXFFO0FBQ3JFLDZEQUE2RTtBQUM3RSxtRUFBOEQ7QUFDOUQsMENBQTBDO0FBQzFDLGlDQUFpQzs7Ozs7O0FBS2pDLE1BQWEsY0FBZSxTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7Ozs7SUF1Qi9DLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsY0FBbUIsRUFBRSxtQkFBMEIsRUFBRSxXQUErQjtRQUM1SCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxFQUFFLEdBQUcsRUFBRSxDQUFDO1FBQ2IsSUFBSSxDQUFDLGNBQWMsR0FBRyxjQUFjLENBQUM7UUFDckMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsbUJBQW1CLENBQUM7UUFDL0MsSUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUM7UUFFL0IsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksQ0FBQyxFQUFFLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNuSCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsSUFBSSwyQ0FBbUIsQ0FBQyxJQUFJLEVBQUUsY0FBYyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFMUYsSUFBSSxDQUFDLDBCQUEwQixFQUFFLENBQUM7UUFDbEMsSUFBSSxDQUFDLHlCQUF5QixFQUFFLENBQUM7UUFDakMsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFM0IsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ2QsSUFBSSxTQUFTLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUVyRixJQUFJLFdBQVcsSUFBSSxXQUFXLENBQUMsT0FBTyxJQUFJLFdBQVcsQ0FBQyxlQUFlLElBQUksV0FBVyxDQUFDLFFBQVEsRUFBRTtnQkFDN0YsSUFBSSxzQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQzthQUMvRjtTQUVGO0lBQ0gsQ0FBQzs7OztJQUVELG1CQUFtQjtRQUNqQixJQUFJLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUU7WUFDckQsTUFBTSxjQUFjLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN6RSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtnQkFDcEMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDO2dCQUNsRSxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7YUFDMUIsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDOzs7O0lBRUQsMEJBQTBCO1FBQ3hCLElBQUksT0FBTyxHQUF1QixJQUFJLENBQUMsZ0JBQWdCLENBQUMsOEJBQThCLEVBQUUsQ0FBQztRQUV6RixJQUFJLFFBQVEsQ0FBQztRQUNiLElBQUksV0FBVyxDQUFDO1FBRWhCLElBQUksV0FBVyxJQUFJLElBQUksQ0FBQyxjQUFjLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTLEVBQUU7WUFDdkUsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQztTQUMvQztRQUNELElBQUksVUFBVSxJQUFJLElBQUksQ0FBQyxjQUFjLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUU7WUFDckUsUUFBUSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDO1NBQ3pDO2FBQU07WUFDTCxRQUFRLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxFQUFFO2dCQUN4QyxLQUFLLGNBQWM7b0JBQ2pCLFdBQVcsR0FBRyxHQUFHLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDO3dCQUMvQyxVQUFVLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLGNBQWM7d0JBQ3BELE9BQU8sRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsUUFBUTt3QkFDeEMsY0FBYyxFQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRzt3QkFDdkMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlO3dCQUMvQyxPQUFPLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE1BQU07cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUNSO29CQUNFLFdBQVcsR0FBRyxHQUFHLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO29CQUNoRyxNQUFNO2FBQ1Q7U0FDRjtRQUVELCtEQUErRDtRQUMvRCxJQUFJLG1CQUFtQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRTFELElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDakUsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLEVBQUUsUUFBUTtZQUN4QixXQUFXLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUEsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUEsQ0FBQyxDQUFDLElBQUk7WUFDeEYsT0FBTyxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxHQUFHLElBQUksQ0FBQyxFQUFFLFFBQVEsQ0FBQztZQUN0RCxVQUFVLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFDakUsWUFBWSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRzthQUNsQyxDQUFDLENBQUM7WUFDSCxtQkFBbUIsRUFBRSxtQkFBbUI7WUFDeEMsSUFBSSxFQUFFO2dCQUNKLFNBQVMsRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLFFBQVE7YUFDOUI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtZQUM3QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7WUFDeEQsZUFBZSxFQUFFO2dCQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsOEJBQThCLENBQUM7Z0JBQzFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMscURBQXFELENBQUM7Z0JBQ2pHLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsbUNBQW1DLENBQUM7YUFDaEY7WUFDRCxjQUFjLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUM7U0FDNUQsQ0FBQyxDQUFDO1FBRUgsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQzFFLEtBQUssRUFBRSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUM7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDckQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLFlBQVksQ0FBQyw4QkFBOEIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDNUYsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLEVBQUUsMEJBQTBCO1lBQzFDLG1CQUFtQixFQUFFLGVBQWUsQ0FBQyxHQUFHO1lBQ3hDLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVE7WUFDaEMsYUFBYSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbEcsMEJBQTBCLEVBQUUsQ0FBQyxDQUFDLHNCQUFzQixJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLG9CQUFvQjtZQUN4SCxRQUFRLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTO1lBQ3ZDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ3pILENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsSUFBSSxZQUFZLENBQUMsNEJBQTRCLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO1lBQ3BJLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLGlCQUFpQjtZQUNqQyxhQUFhLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUM1RDtnQkFDRSxNQUFNLEVBQUUsQ0FBQyxDQUFDLE1BQU07Z0JBQ2hCLDRCQUE0QixFQUFFO29CQUM1QixJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsRUFBRSxpQ0FBaUM7b0JBQ2pELFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLGVBQWU7b0JBQ3RDLE9BQU8sRUFBRTt3QkFDUCxJQUFJLEVBQUUsSUFBSSxDQUFDLEVBQUU7cUJBQ2Q7b0JBQ0QsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLFdBQVc7aUJBQ2hDO2FBQ0YsQ0FBQyxDQUFDO1NBQ04sQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFZixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksWUFBWSxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDMUUsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLEVBQUUsaUJBQWlCO1lBQ2pDLGNBQWMsRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUc7WUFDL0IsOEJBQThCLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHO1lBQ3ZELDRCQUE0QixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNoRyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8saUJBQWlCLENBQUMsY0FBbUI7UUFDM0MsSUFBSSxNQUFNLEdBQVMsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sVUFBVSxHQUFHLGNBQWMsQ0FBQyxpQkFBaUIsQ0FBQztRQUNwRCxJQUFJLFVBQVUsRUFBRTtZQUNkLEtBQUssSUFBSSxTQUFTLElBQUksVUFBVSxFQUFFO2dCQUNoQyxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUN6QyxVQUFVLEVBQUU7d0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDOzRCQUN0QixHQUFHLEVBQUUsU0FBUzs0QkFDZCxPQUFPLEVBQUUsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU87NEJBQ3RDLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7NEJBQ3hCLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUMsU0FBUzt5QkFDM0MsQ0FBQztxQkFDSDtpQkFDRixDQUFDLENBQUM7YUFDSjtTQUNGO1FBRUQsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUVPLHdCQUF3QjtRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUNsRixXQUFXLEVBQUUsd0JBQXdCLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFO1lBQy9ELEtBQUssRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsTUFBTTtTQUMxRCxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVmLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxHQUFHLENBQUM7UUFFNUIsSUFBSSxtQkFBbUIsR0FBRztZQUN4QjtnQkFDRSxVQUFVLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLFdBQVc7Z0JBQ2hHLEdBQUcsRUFBRTtvQkFDSCxTQUFTLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUs7b0JBQzdCLFVBQVUsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQzdFLFVBQVUsRUFBRSxLQUFLO29CQUNqQixRQUFRLEVBQUUsR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLE1BQU07aUJBQ3RCO2FBQ0Y7U0FDRixDQUFDO1FBQ0YsT0FBTyxtQkFBbUIsQ0FBQztJQUM3QixDQUFDOzs7O0lBRUQsb0JBQW9CLENBQUMsVUFBa0I7UUFDckMsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLGtEQUFzQixDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUN4RyxPQUFPLHdCQUF3QixDQUFDLFdBQVcsQ0FBQztJQUM5QyxDQUFDOzs7O0lBRUQseUJBQXlCOztRQUN2QixJQUFJLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdEIsOEJBQThCO1lBQzlCLElBQUksWUFBWSxHQUFRO2dCQUN0QixrQkFBa0IsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRTtnQkFDckQsYUFBYSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFO2FBRWxELENBQUM7WUFFRixJQUFJLElBQUksQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUU7Z0JBQ2pILFlBQVksQ0FBQyxpQkFBaUIsR0FBRyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGVBQWUsRUFBRSxDQUFDO2dCQUM3RSxZQUFZLENBQUMsYUFBYSxHQUFHLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBQ2pFLFlBQVksQ0FBQyxjQUFjLEdBQUcsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsQ0FBQzthQUNwRTtZQUVELElBQUksaUJBQWlCLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDOUUsU0FBUyxFQUFFLFNBQVMsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO29CQUN4QyxPQUFPLEVBQUUsS0FBSztvQkFDZCxNQUFNLEVBQUU7d0JBQ04sT0FBTyxFQUFFOzRCQUNQLFFBQVEsRUFBRTtnQ0FDUixnQkFBZ0I7Z0NBQ2hCLDJCQUEyQjs2QkFDNUI7eUJBQ0Y7d0JBQ0QsU0FBUyxFQUFFOzRCQUNULFFBQVEsRUFBRTtnQ0FDUiw4S0FBOEs7NkJBQy9LO3lCQUNGO3dCQUNELEtBQUssRUFBRTs0QkFDTCxRQUFRLEVBQUUsMEZBQTBGO3lCQUNyRztxQkFDRjtpQkFDRixDQUFDO2dCQUNGLFdBQVcsRUFBRTtvQkFDWCxVQUFVLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxZQUFZO29CQUNsRCxvQkFBb0IsRUFBRSxZQUFZO2lCQUNuQzthQUNGLENBQUMsQ0FBQztZQUVILGlCQUFpQixDQUFDLGVBQWUsQ0FDL0IsR0FBRyxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUM7Z0JBQzNCLEdBQUcsRUFBRSxtQkFBbUI7Z0JBQ3hCLE1BQU0sRUFBRSxPQUFPO2dCQUNmLE1BQU0sRUFBRSxDQUFDLDBDQUEwQyxDQUFDO2dCQUNwRCxRQUFRLEVBQUUsR0FBRzthQUNkLENBQUMsQ0FBQyxDQUFDO1lBRU4sTUFBTSxrQkFBa0IsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUV2RSxJQUFJLGFBQWEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMscUJBQXFCLEVBQUUsQ0FBQztZQUNyRSxNQUFNLFdBQVcsR0FBRyxJQUFJLG9CQUFvQixDQUFDLGVBQWUsQ0FBQztnQkFDM0QsVUFBVSxFQUFFLG1CQUFtQjtnQkFDL0IsT0FBTyxFQUFFLGlCQUFpQjtnQkFDMUIsS0FBSyxFQUFFLE1BQUEsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsWUFBWSxDQUFDLDBDQUFFLFlBQWE7Z0JBQ2xFLFdBQVcsRUFBRSxhQUFhLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQzVGLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixDQUFDO2FBQzlCLENBQUMsQ0FBQztZQUVILDRCQUE0QjtZQUM1QixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO2dCQUM5RCxZQUFZLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJO2dCQUN0Qyx3QkFBd0IsRUFBRSxJQUFJO2dCQUM5QixNQUFNLEVBQUU7b0JBQ047d0JBQ0UsU0FBUyxFQUFFLFFBQVE7d0JBQ25CLE9BQU8sRUFBRSxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBNkIsRUFBRSxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUM7cUJBQzlGO29CQUNEO3dCQUNFLFNBQVMsRUFBRSxPQUFPO3dCQUNsQixPQUFPLEVBQUUsQ0FBQyxXQUFXLENBQUM7cUJBQ3ZCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDOzs7O0lBRUQsaUJBQWlCLENBQUMsV0FBZ0I7UUFDaEMsUUFBUSxXQUFXLEVBQUU7WUFDbkIsS0FBSyxZQUFZO2dCQUNmLE9BQU87b0JBQ0wsSUFBSSxFQUFFLGtEQUFrRDtvQkFDeEQsTUFBTSxFQUFFLENBQUMsY0FBYyxDQUFDO2lCQUN6QixDQUFDO1lBQ0osS0FBSyxZQUFZO2dCQUNmLE9BQU87b0JBQ0wsSUFBSSxFQUFFLGlEQUFpRDtvQkFDdkQsTUFBTSxFQUFFLENBQUMsY0FBYyxDQUFDO2lCQUN6QixDQUFDO1lBQ0osS0FBSyxTQUFTO2dCQUNaLE9BQU87b0JBQ0wsSUFBSSxFQUFFLEdBQUc7b0JBQ1QsTUFBTSxFQUFFLENBQUMsaUJBQWlCLENBQUM7b0JBQzNCLE9BQU8sRUFBRTt3QkFDUCxjQUFjLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQztxQkFDOUM7aUJBQ0YsQ0FBQztZQUNKLEtBQUssU0FBUztnQkFDWixPQUFPO29CQUNMLElBQUksRUFBRSxHQUFHO29CQUNULE1BQU0sRUFBRSxDQUFDLGlCQUFpQixDQUFDO29CQUMzQixPQUFPLEVBQUU7d0JBQ1AsY0FBYyxFQUFFLENBQUMsMkJBQTJCLENBQUM7cUJBQzlDO2lCQUNGLENBQUM7WUFDSjtnQkFDRSxPQUFPO29CQUNMLElBQUksRUFBRSxZQUFZO2lCQUNuQixDQUFDO1NBQ0w7SUFDSCxDQUFDOztBQXZUSCx3Q0F5VEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlYnVpbGQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVidWlsZCc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmVfYWN0aW9ucyBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lLWFjdGlvbnMnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgZXZlbnRfdGFyZ2V0cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzLXRhcmdldHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgaW1hZ2VidWlsZGVyIGZyb20gJ0Bhd3MtY2RrL2F3cy1pbWFnZWJ1aWxkZXInO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgc25zIGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQgeyBTdHJpbmdDb21wb25lbnRNYXAgfSBmcm9tICcuJztcbmltcG9ydCB7IENvbXBvbmVudEJ1aWxkZXIgfSBmcm9tICcuL2NvbXBvbmVudF9idWlsZGVyJztcbmltcG9ydCB7IEltYWdlUmVjaXBlVmVyc2lvbkJ1bXAgfSBmcm9tICcuL2ltYWdlX3JlY2lwZV92ZXJzaW9uX2J1bXAnO1xuaW1wb3J0IHsgU2xhY2tDb25maWd1cmF0aW9uLCBTbGFja05vdGlmaWNhdGlvbiB9IGZyb20gJy4vc2xhY2tfbm90aWZpY2F0aW9uJztcbmltcG9ydCB7IFNvdXJjZUFjdGlvbkJ1aWxkZXIgfSBmcm9tICcuL3NvdXJjZV9hY3Rpb25fYnVpbGRlcic7XG5pbXBvcnQgKiBhcyBzc211cGRhdGUgZnJvbSAnLi9zc21fdXBkYXRlJztcbmltcG9ydCAqIGFzIHV0aWxzIGZyb20gJy4vdXRpbHMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFtaVBpcGVsaW5lTGliIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHBpcGVsaW5lQ29uZmlnOiBhbnlcbiAgY29tcG9uZW50RGVwc0NvbmZpZzogYW55W11cbiAgaWQ6IHN0cmluZ1xuICBzb3VyY2VBY3Rpb25CdWlsZGVyOiBTb3VyY2VBY3Rpb25CdWlsZGVyO1xuICBjb21wb25lbnRCdWlsZGVyOiBDb21wb25lbnRCdWlsZGVyO1xuICBpbWFnZVBpcGVsaW5lOiBpbWFnZWJ1aWxkZXIuQ2ZuSW1hZ2VQaXBlbGluZSB8IHVuZGVmaW5lZDtcbiAgdG9waWM6IHNucy5Ub3BpYyB8IHVuZGVmaW5lZDtcbiAgY29kZXBpcGVsaW5lOiBjb2RlcGlwZWxpbmUuUGlwZWxpbmUgfCB1bmRlZmluZWQ7XG4gIHNsYWNrQ29uZmlnOiBTbGFja0NvbmZpZ3VyYXRpb247XG4gIHJlY2lwZTogaW1hZ2VidWlsZGVyLkNmbkltYWdlUmVjaXBlIHwgdW5kZWZpbmVkO1xuICBpbmZyYXN0cnVjdHVyZTogaW1hZ2VidWlsZGVyLkNmbkluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbiB8IHVuZGVmaW5lZDtcbiAgZGlzdHJpYnV0aW9uQ29uZmlnOiBpbWFnZWJ1aWxkZXIuQ2ZuRGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbiB8IHVuZGVmaW5lZDtcbiAgZGlza1NpemU6IG51bWJlciB8IHVuZGVmaW5lZDtcbiAgZWJzRW5jcnlwdGlvbktleToga21zLktleSB8IHVuZGVmaW5lZDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwaXBlbGluZUNvbmZpZzogYW55LCBjb21wb25lbnREZXBzQ29uZmlnOiBhbnlbXSwgc2xhY2tDb25maWc6IFNsYWNrQ29uZmlndXJhdGlvbikge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmlkID0gaWQ7XG4gICAgdGhpcy5waXBlbGluZUNvbmZpZyA9IHBpcGVsaW5lQ29uZmlnO1xuICAgIHRoaXMucGlwZWxpbmVDb25maWcubmFtZSA9IHRoaXMuaWQ7XG4gICAgdGhpcy5jb21wb25lbnREZXBzQ29uZmlnID0gY29tcG9uZW50RGVwc0NvbmZpZztcbiAgICB0aGlzLnNsYWNrQ29uZmlnID0gc2xhY2tDb25maWc7XG5cbiAgICB0aGlzLmNvbXBvbmVudEJ1aWxkZXIgPSBuZXcgQ29tcG9uZW50QnVpbGRlcih0aGlzLCBjb21wb25lbnREZXBzQ29uZmlnLCB0aGlzLmlkLCB0aGlzLnBpcGVsaW5lQ29uZmlnLnBhcmVudF9pbWFnZSk7XG4gICAgdGhpcy5zb3VyY2VBY3Rpb25CdWlsZGVyID0gbmV3IFNvdXJjZUFjdGlvbkJ1aWxkZXIodGhpcywgcGlwZWxpbmVDb25maWcuc291cmNlcywgdGhpcy5pZCk7XG5cbiAgICB0aGlzLmNyZWF0ZUltYWdlYnVpbGRlclBpcGVsaW5lKCk7XG4gICAgdGhpcy5jcmVhdGVDb2RlcGlwZWxpbmVQcm9qZWN0KCk7XG4gICAgdGhpcy5jcmVhdGVTY2hlZHVsZWRUYXNrKCk7XG5cbiAgICBpZiAodGhpcy50b3BpYykge1xuICAgICAgbmV3IHNzbXVwZGF0ZS5Tc21VcGRhdGVDb25zdHJ1Y3QodGhpcywgJ1NTTVVwZGF0ZScsIHRoaXMudG9waWMsIHRoaXMucGlwZWxpbmVDb25maWcpO1xuXG4gICAgICBpZiAoc2xhY2tDb25maWcgJiYgc2xhY2tDb25maWcuY2hhbm5lbCAmJiBzbGFja0NvbmZpZy5zbGFja1dlYmhvb2tVcmwgJiYgc2xhY2tDb25maWcudXNlcm5hbWUpIHtcbiAgICAgICAgbmV3IFNsYWNrTm90aWZpY2F0aW9uKHRoaXMsICdTbGFja05vdGlmaWNhdGlvbicsIHRoaXMudG9waWMsIHNsYWNrQ29uZmlnLCBgJHt0aGlzLmlkfVJlY2lwZWApO1xuICAgICAgfVxuXG4gICAgfVxuICB9XG5cbiAgY3JlYXRlU2NoZWR1bGVkVGFzaygpIHtcbiAgICBpZiAodGhpcy5jb2RlcGlwZWxpbmUgJiYgdGhpcy5waXBlbGluZUNvbmZpZy5zY2hlZHVsZSkge1xuICAgICAgY29uc3QgcGlwZWxpbmVUYXJnZXQgPSBuZXcgZXZlbnRfdGFyZ2V0cy5Db2RlUGlwZWxpbmUodGhpcy5jb2RlcGlwZWxpbmUpO1xuICAgICAgbmV3IGV2ZW50cy5SdWxlKHRoaXMsICdTY2hlZHVsZVJ1bGUnLCB7XG4gICAgICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUuZXhwcmVzc2lvbih0aGlzLnBpcGVsaW5lQ29uZmlnLnNjaGVkdWxlKSxcbiAgICAgICAgdGFyZ2V0czogW3BpcGVsaW5lVGFyZ2V0XSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIGNyZWF0ZUltYWdlYnVpbGRlclBpcGVsaW5lKCkge1xuICAgIGxldCBjb21wTWFwOiBTdHJpbmdDb21wb25lbnRNYXAgPSB0aGlzLmNvbXBvbmVudEJ1aWxkZXIuY3JlYXRlQ29tcG9uZW50RGVwZW5kZW5jaWVzTWFwKCk7XG5cbiAgICBsZXQgaW1hZ2VfaWQ7XG4gICAgbGV0IHBhcmVudEltYWdlO1xuXG4gICAgaWYgKCdkaXNrX3NpemUnIGluIHRoaXMucGlwZWxpbmVDb25maWcgJiYgdGhpcy5waXBlbGluZUNvbmZpZy5kaXNrX3NpemUpIHtcbiAgICAgIHRoaXMuZGlza1NpemUgPSB0aGlzLnBpcGVsaW5lQ29uZmlnLmRpc2tfc2l6ZTtcbiAgICB9XG4gICAgaWYgKCdpbWFnZV9pZCcgaW4gdGhpcy5waXBlbGluZUNvbmZpZyAmJiB0aGlzLnBpcGVsaW5lQ29uZmlnLmltYWdlX2lkKSB7XG4gICAgICBpbWFnZV9pZCA9IHRoaXMucGlwZWxpbmVDb25maWcuaW1hZ2VfaWQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHN3aXRjaCAodGhpcy5waXBlbGluZUNvbmZpZy5wYXJlbnRfaW1hZ2UpIHtcbiAgICAgICAgY2FzZSAnQW1hem9uTGludXgyJzpcbiAgICAgICAgICBwYXJlbnRJbWFnZSA9IGVjMi5NYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoe1xuICAgICAgICAgICAgZ2VuZXJhdGlvbjogZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMixcbiAgICAgICAgICAgIGVkaXRpb246IGVjMi5BbWF6b25MaW51eEVkaXRpb24uU1RBTkRBUkQsXG4gICAgICAgICAgICB2aXJ0dWFsaXphdGlvbjogZWMyLkFtYXpvbkxpbnV4VmlydC5IVk0sXG4gICAgICAgICAgICBzdG9yYWdlOiBlYzIuQW1hem9uTGludXhTdG9yYWdlLkdFTkVSQUxfUFVSUE9TRSxcbiAgICAgICAgICAgIGNwdVR5cGU6IGVjMi5BbWF6b25MaW51eENwdVR5cGUuWDg2XzY0LFxuICAgICAgICAgIH0pO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgIHBhcmVudEltYWdlID0gZWMyLk1hY2hpbmVJbWFnZS5sb29rdXAodGhpcy5nZXRMb29rdXBDcml0ZXJpYSh0aGlzLnBpcGVsaW5lQ29uZmlnLnBhcmVudF9pbWFnZSkpO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIFVzZSBhIGN1c3RvbSBibG9jayBkZXZpY2UgbWFwcGluZyBpZiBlbmNyeXB0aW9uIGlzIHJlcXVpcmVkLlxuICAgIGxldCBibG9ja0RldmljZU1hcHBpbmdzID0gdGhpcy5jcmVhdGVCbG9ja0RldmljZU1hcHBpbmcoKTtcblxuICAgIHRoaXMucmVjaXBlID0gbmV3IGltYWdlYnVpbGRlci5DZm5JbWFnZVJlY2lwZSh0aGlzLCAnSW1hZ2VSZWNpcGUnLCB7XG4gICAgICBuYW1lOiBgJHt0aGlzLmlkfVJlY2lwZWAsXG4gICAgICBwYXJlbnRJbWFnZTogaW1hZ2VfaWQgPyBpbWFnZV9pZDogcGFyZW50SW1hZ2UgPyBwYXJlbnRJbWFnZS5nZXRJbWFnZSh0aGlzKS5pbWFnZUlkOiBudWxsLFxuICAgICAgdmVyc2lvbjogdGhpcy5nZXROZXh0UmVjaXBlVmVyc2lvbihgJHt0aGlzLmlkfVJlY2lwZWApLFxuICAgICAgY29tcG9uZW50czogdGhpcy5waXBlbGluZUNvbmZpZy5yZWNpcGUuY29tcG9uZW50cy5tYXAoKGM6IGFueSkgPT4gKHtcbiAgICAgICAgY29tcG9uZW50QXJuOiBjb21wTWFwW2MubmFtZV0ucmVmLFxuICAgICAgfSkpLFxuICAgICAgYmxvY2tEZXZpY2VNYXBwaW5nczogYmxvY2tEZXZpY2VNYXBwaW5ncyxcbiAgICAgIHRhZ3M6IHtcbiAgICAgICAgU2hvcnROYW1lOiBgJHt0aGlzLmlkfVJlY2lwZWAsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgY29uc3QgYnVpbGRlclJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNTTU1hbmFnZWRJbnN0YW5jZUNvcmUnKSxcbiAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdFQzJJbnN0YW5jZVByb2ZpbGVGb3JJbWFnZUJ1aWxkZXJFQ1JDb250YWluZXJCdWlsZHMnKSxcbiAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdFQzJJbnN0YW5jZVByb2ZpbGVGb3JJbWFnZUJ1aWxkZXInKSxcbiAgICAgIF0sXG4gICAgICBpbmxpbmVQb2xpY2llczogdGhpcy5nZXRJbmxpbmVQb2xpY2llcyh0aGlzLnBpcGVsaW5lQ29uZmlnKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGluc3RhbmNlUHJvZmlsZSA9IG5ldyBpYW0uQ2ZuSW5zdGFuY2VQcm9maWxlKHRoaXMsICdJbnN0YW5jZVByb2ZpbGUnLCB7XG4gICAgICByb2xlczogW2J1aWxkZXJSb2xlLnJvbGVOYW1lXSxcbiAgICB9KTtcblxuICAgIHRoaXMudG9waWMgPSBuZXcgc25zLlRvcGljKHRoaXMsICdBbWlQaXBlbGluZVRvcGljJyk7XG4gICAgdGhpcy5pbmZyYXN0cnVjdHVyZSA9IG5ldyBpbWFnZWJ1aWxkZXIuQ2ZuSW5mcmFzdHJ1Y3R1cmVDb25maWd1cmF0aW9uKHRoaXMsICdJbmZyYXN0cnVjdHVyZScsIHtcbiAgICAgIG5hbWU6IGAke3RoaXMuaWR9IC0gRUMyIElCIEluZnJhc3RydWN0dXJlYCxcbiAgICAgIGluc3RhbmNlUHJvZmlsZU5hbWU6IGluc3RhbmNlUHJvZmlsZS5yZWYsXG4gICAgICBzbnNUb3BpY0FybjogdGhpcy50b3BpYy50b3BpY0FybixcbiAgICAgIGluc3RhbmNlVHlwZXM6IHRoaXMucGlwZWxpbmVDb25maWcuaW5zdGFuY2VfdHlwZSA/IFt0aGlzLnBpcGVsaW5lQ29uZmlnLmluc3RhbmNlX3R5cGVdIDogdW5kZWZpbmVkLFxuICAgICAgdGVybWluYXRlSW5zdGFuY2VPbkZhaWx1cmU6ICEoJ3Rlcm1pbmF0ZV9vbl9mYWlsdXJlJyBpbiB0aGlzLnBpcGVsaW5lQ29uZmlnKSB8fCB0aGlzLnBpcGVsaW5lQ29uZmlnLnRlcm1pbmF0ZV9vbl9mYWlsdXJlLFxuICAgICAgc3VibmV0SWQ6IHRoaXMucGlwZWxpbmVDb25maWcuc3VibmV0X2lkLFxuICAgICAgc2VjdXJpdHlHcm91cElkczogdGhpcy5waXBlbGluZUNvbmZpZy5zZWN1cml0eV9ncm91cF9pZHMgPyB0aGlzLnBpcGVsaW5lQ29uZmlnLnNlY3VyaXR5X2dyb3VwX2lkcy5zcGxpdCgnLCcpIDogdW5kZWZpbmVkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kaXN0cmlidXRpb25Db25maWcgPSB0aGlzLnBpcGVsaW5lQ29uZmlnLnNoYXJlZF93aXRoID8gbmV3IGltYWdlYnVpbGRlci5DZm5EaXN0cmlidXRpb25Db25maWd1cmF0aW9uKHRoaXMsICdEaXN0cmlidXRpb25Db25maWcnLCB7XG4gICAgICBuYW1lOiBgJHt0aGlzLmlkfSAtIERpc3RyaWJ1dGlvbmAsXG4gICAgICBkaXN0cmlidXRpb25zOiB0aGlzLnBpcGVsaW5lQ29uZmlnLnNoYXJlZF93aXRoLm1hcCgoczphbnkpID0+IChcbiAgICAgICAge1xuICAgICAgICAgIHJlZ2lvbjogcy5yZWdpb24sXG4gICAgICAgICAgYW1pRGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgTmFtZTogYCR7dGhpcy5pZH0gLSB7eyBpbWFnZWJ1aWxkZXI6YnVpbGREYXRlIH19YCxcbiAgICAgICAgICAgIERlc2NyaXB0aW9uOiBgJHt0aGlzLmlkfSBTaGFyZWQgaW1hZ2VgLFxuICAgICAgICAgICAgQW1pVGFnczoge1xuICAgICAgICAgICAgICBOYW1lOiB0aGlzLmlkLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFRhcmdldEFjY291bnRJZHM6IHMuYWNjb3VudF9pZHMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSkpLFxuICAgIH0pIDogdW5kZWZpbmVkO1xuXG4gICAgdGhpcy5pbWFnZVBpcGVsaW5lID0gbmV3IGltYWdlYnVpbGRlci5DZm5JbWFnZVBpcGVsaW5lKHRoaXMsICdBbWlQaXBlbGluZScsIHtcbiAgICAgIG5hbWU6IGAke3RoaXMuaWR9IC0gQU1JIFBpcGVsaW5lYCxcbiAgICAgIGltYWdlUmVjaXBlQXJuOiB0aGlzLnJlY2lwZS5yZWYsXG4gICAgICBpbmZyYXN0cnVjdHVyZUNvbmZpZ3VyYXRpb25Bcm46IHRoaXMuaW5mcmFzdHJ1Y3R1cmUucmVmLFxuICAgICAgZGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbkFybjogdGhpcy5kaXN0cmlidXRpb25Db25maWcgPyB0aGlzLmRpc3RyaWJ1dGlvbkNvbmZpZy5yZWYgOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGdldElubGluZVBvbGljaWVzKHBpcGVsaW5lQ29uZmlnOiBhbnkpOiB7IFtuYW1lOiBzdHJpbmddOiBpYW0uUG9saWN5RG9jdW1lbnQgfSB8IHVuZGVmaW5lZCB7XG4gICAgbGV0IHJlc3VsdCA6IGFueSA9IHt9O1xuICAgIGNvbnN0IHN0YXRlbWVudHMgPSBwaXBlbGluZUNvbmZpZy5wb2xpY3lfc3RhdGVtZW50cztcbiAgICBpZiAoc3RhdGVtZW50cykge1xuICAgICAgZm9yIChsZXQgc3RhdGVtZW50IGluIHN0YXRlbWVudHMpIHtcbiAgICAgICAgcmVzdWx0W3N0YXRlbWVudF0gPSBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIHNpZDogc3RhdGVtZW50LFxuICAgICAgICAgICAgICBhY3Rpb25zOiBzdGF0ZW1lbnRzW3N0YXRlbWVudF0uYWN0aW9ucyxcbiAgICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IHN0YXRlbWVudHNbc3RhdGVtZW50XS5yZXNvdXJjZXMsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVCbG9ja0RldmljZU1hcHBpbmcoKSB7XG4gICAgY29uc3Qga2V5ID0gdGhpcy5waXBlbGluZUNvbmZpZy5lbmNyeXB0ZWQgPyBuZXcga21zLktleSh0aGlzLCAnUGlwZWxpbmVFbmNyeXB0aW9uJywge1xuICAgICAgZGVzY3JpcHRpb246IGBJbWFnZSBlbmNyeXB0aW9uIGZvciAke3RoaXMucGlwZWxpbmVDb25maWcubmFtZX1gLFxuICAgICAgYWxpYXM6IGAke3V0aWxzLnBhc2NhbGl6ZSh0aGlzLnBpcGVsaW5lQ29uZmlnLm5hbWUpfV9rZXlgLFxuICAgIH0pIDogdW5kZWZpbmVkO1xuXG4gICAgdGhpcy5lYnNFbmNyeXB0aW9uS2V5ID0ga2V5O1xuXG4gICAgbGV0IGJsb2NrRGV2aWNlTWFwcGluZ3MgPSBbXG4gICAgICB7XG4gICAgICAgIGRldmljZU5hbWU6IHRoaXMucGlwZWxpbmVDb25maWcucGFyZW50X2ltYWdlLmluY2x1ZGVzKCdBbWF6b25MaW51eCcpID8gJy9kZXYveHZkYScgOiAnL2Rldi9zZGExJyxcbiAgICAgICAgZWJzOiB7XG4gICAgICAgICAgZW5jcnlwdGVkOiBrZXkgPyB0cnVlIDogZmFsc2UsXG4gICAgICAgICAgdm9sdW1lU2l6ZTogdGhpcy5waXBlbGluZUNvbmZpZy5kaXNrX3NpemUgPyB0aGlzLnBpcGVsaW5lQ29uZmlnLmRpc2tfc2l6ZSA6IDgsXG4gICAgICAgICAgdm9sdW1lVHlwZTogJ2dwMicsXG4gICAgICAgICAga21zS2V5SWQ6IGtleT8ua2V5QXJuLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdO1xuICAgIHJldHVybiBibG9ja0RldmljZU1hcHBpbmdzO1xuICB9XG5cbiAgZ2V0TmV4dFJlY2lwZVZlcnNpb24ocmVjaXBlTmFtZTogc3RyaW5nKSB7XG4gICAgY29uc3QgcmVjaXBlQnVtcEN1c3RvbVJlc291cmNlID0gbmV3IEltYWdlUmVjaXBlVmVyc2lvbkJ1bXAodGhpcywgJ0ltYWdlUmVjaXBlVmVyc2lvbkJ1bXAnLCByZWNpcGVOYW1lKTtcbiAgICByZXR1cm4gcmVjaXBlQnVtcEN1c3RvbVJlc291cmNlLm5leHRWZXJzaW9uO1xuICB9XG5cbiAgY3JlYXRlQ29kZXBpcGVsaW5lUHJvamVjdCgpIHtcbiAgICBpZiAodGhpcy5pbWFnZVBpcGVsaW5lKSB7XG4gICAgICAvLyBDcmVhdGUgQ29kZVBpcGVsaW5lIHByb2plY3RcbiAgICAgIGxldCBlbnZWYXJpYWJsZXM6IGFueSA9IHtcbiAgICAgICAgSU1BR0VfUElQRUxJTkVfQVJOOiB7IHZhbHVlOiB0aGlzLmltYWdlUGlwZWxpbmUucmVmIH0sXG4gICAgICAgIFBJUEVMSU5FX05BTUU6IHsgdmFsdWU6IHRoaXMuaW1hZ2VQaXBlbGluZS5uYW1lIH0sXG5cbiAgICAgIH07XG5cbiAgICAgIGlmICh0aGlzLnNsYWNrQ29uZmlnICYmIHRoaXMuc2xhY2tDb25maWcuc2xhY2tXZWJob29rVXJsICYmIHRoaXMuc2xhY2tDb25maWcuY2hhbm5lbCAmJiB0aGlzLnNsYWNrQ29uZmlnLnVzZXJuYW1lKSB7XG4gICAgICAgIGVudlZhcmlhYmxlcy5TTEFDS19XRUJIT09LX1VSTCA9IHsgdmFsdWU6IHRoaXMuc2xhY2tDb25maWcuc2xhY2tXZWJob29rVXJsIH07XG4gICAgICAgIGVudlZhcmlhYmxlcy5TTEFDS19DSEFOTkVMID0geyB2YWx1ZTogdGhpcy5zbGFja0NvbmZpZy5jaGFubmVsIH07XG4gICAgICAgIGVudlZhcmlhYmxlcy5TTEFDS19VU0VSTkFNRSA9IHsgdmFsdWU6IHRoaXMuc2xhY2tDb25maWcudXNlcm5hbWUgfTtcbiAgICAgIH1cblxuICAgICAgbGV0IGltYWdlYnVpbGRlckJ1aWxkID0gbmV3IGNvZGVidWlsZC5QaXBlbGluZVByb2plY3QodGhpcywgJ0FtaVBpcGVsaW5lQnVpbGQnLCB7XG4gICAgICAgIGJ1aWxkU3BlYzogY29kZWJ1aWxkLkJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgICAgICB2ZXJzaW9uOiAnMC4yJyxcbiAgICAgICAgICBwaGFzZXM6IHtcbiAgICAgICAgICAgIGluc3RhbGw6IHtcbiAgICAgICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICAgICAnYXB0LWdldCB1cGRhdGUnLFxuICAgICAgICAgICAgICAgICdhcHQtZ2V0IGluc3RhbGwgYXdzY2xpIC15JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwcmVfYnVpbGQ6IHtcbiAgICAgICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICAgICAnaWYgWyAtbiBcIiRTTEFDS19XRUJIT09LX1VSTFwiIF07IHRoZW4gY3VybCAtWCBQT1NUIC1IIFxcJ0NvbnRlbnQtdHlwZTogYXBwbGljYXRpb24vanNvblxcJyAtLWRhdGEgXCJ7XFxcXFwidGV4dFxcXFxcIjpcXFxcXCIkUElQRUxJTkVfTkFNRSBidWlsZCBoYXMgc3RhcnRlZC5cXFxcXCJ9XCIgJFNMQUNLX1dFQkhPT0tfVVJMOyBmaScsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgYnVpbGQ6IHtcbiAgICAgICAgICAgICAgY29tbWFuZHM6ICdhd3MgaW1hZ2VidWlsZGVyIHN0YXJ0LWltYWdlLXBpcGVsaW5lLWV4ZWN1dGlvbiAtLWltYWdlLXBpcGVsaW5lLWFybiAkSU1BR0VfUElQRUxJTkVfQVJOJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgYnVpbGRJbWFnZTogY29kZWJ1aWxkLkxpbnV4QnVpbGRJbWFnZS5TVEFOREFSRF80XzAsXG4gICAgICAgICAgZW52aXJvbm1lbnRWYXJpYWJsZXM6IGVudlZhcmlhYmxlcyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBpbWFnZWJ1aWxkZXJCdWlsZC5hZGRUb1JvbGVQb2xpY3koXG4gICAgICAgIGlhbS5Qb2xpY3lTdGF0ZW1lbnQuZnJvbUpzb24oe1xuICAgICAgICAgIFNpZDogJ2ltYWdlYnVpbGRlcnN0YXJ0JyxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgQWN0aW9uOiBbJ2ltYWdlYnVpbGRlcjpTdGFydEltYWdlUGlwZWxpbmVFeGVjdXRpb24nXSxcbiAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICB9KSk7XG5cbiAgICAgIGNvbnN0IGltYWdlQnVpbGRlck91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoJ0Nka0J1aWxkT3V0cHV0Jyk7XG5cbiAgICAgIGxldCBzb3VyY2VBY3Rpb25zID0gdGhpcy5zb3VyY2VBY3Rpb25CdWlsZGVyLmNyZWF0ZVBpcGVsaW5lU291cmNlcygpO1xuICAgICAgY29uc3QgYnVpbGRBY3Rpb24gPSBuZXcgY29kZXBpcGVsaW5lX2FjdGlvbnMuQ29kZUJ1aWxkQWN0aW9uKHtcbiAgICAgICAgYWN0aW9uTmFtZTogJ1N0YXJ0SW1hZ2VCdWlsZGVyJyxcbiAgICAgICAgcHJvamVjdDogaW1hZ2VidWlsZGVyQnVpbGQsXG4gICAgICAgIGlucHV0OiBzb3VyY2VBY3Rpb25zLmZpbmQoYSA9PiBhICYmIGEuc291cmNlT3V0cHV0KT8uc291cmNlT3V0cHV0ISxcbiAgICAgICAgZXh0cmFJbnB1dHM6IHNvdXJjZUFjdGlvbnMubGVuZ3RoID4gMSA/IHNvdXJjZUFjdGlvbnMuc2xpY2UoMSkubWFwKGEgPT4gYS5zb3VyY2VPdXRwdXQpIDogW10sXG4gICAgICAgIG91dHB1dHM6IFtpbWFnZUJ1aWxkZXJPdXRwdXRdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIENvbXBsZXRlIFBpcGVsaW5lIFByb2plY3RcbiAgICAgIHRoaXMuY29kZXBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZSh0aGlzLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgIHBpcGVsaW5lTmFtZTogdGhpcy5waXBlbGluZUNvbmZpZy5uYW1lLFxuICAgICAgICByZXN0YXJ0RXhlY3V0aW9uT25VcGRhdGU6IHRydWUsXG4gICAgICAgIHN0YWdlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICBhY3Rpb25zOiBzb3VyY2VBY3Rpb25zLm1hcChjID0+IGMuYWN0aW9uKS5maWx0ZXIoKGMpOiBjIGlzIGNvZGVwaXBlbGluZS5JQWN0aW9uID0+IGMgIT0gbnVsbCksXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzdGFnZU5hbWU6ICdCdWlsZCcsXG4gICAgICAgICAgICBhY3Rpb25zOiBbYnVpbGRBY3Rpb25dLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICBnZXRMb29rdXBDcml0ZXJpYShwYXJlbnRJbWFnZTogYW55KTogZWMyLkxvb2t1cE1hY2hpbmVJbWFnZVByb3BzIHtcbiAgICBzd2l0Y2ggKHBhcmVudEltYWdlKSB7XG4gICAgICBjYXNlICdVYnVudHUxODA0JzpcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBuYW1lOiAndWJ1bnR1L2ltYWdlcy9odm0tc3NkL3VidW50dS1iaW9uaWMtMTguMDQtYW1kNjQqJyxcbiAgICAgICAgICBvd25lcnM6IFsnMDk5NzIwMTA5NDc3J10sXG4gICAgICAgIH07XG4gICAgICBjYXNlICdVYnVudHUyMDA0JzpcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBuYW1lOiAndWJ1bnR1L2ltYWdlcy9odm0tc3NkL3VidW50dS1mb2NhbC0yMC4wNC1hbWQ2NConLFxuICAgICAgICAgIG93bmVyczogWycwOTk3MjAxMDk0NzcnXSxcbiAgICAgICAgfTtcbiAgICAgIGNhc2UgJ0NlbnRPUzcnOlxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG5hbWU6ICcqJyxcbiAgICAgICAgICBvd25lcnM6IFsnYXdzLW1hcmtldHBsYWNlJ10sXG4gICAgICAgICAgZmlsdGVyczoge1xuICAgICAgICAgICAgJ3Byb2R1Y3QtY29kZSc6IFsnY3Z1Z3ppa252bXhncW5hOW5vaWJxbm5zeSddLFxuICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgICBjYXNlICdDZW50T1M4JzpcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBuYW1lOiAnKicsXG4gICAgICAgICAgb3duZXJzOiBbJ2F3cy1tYXJrZXRwbGFjZSddLFxuICAgICAgICAgIGZpbHRlcnM6IHtcbiAgICAgICAgICAgICdwcm9kdWN0LWNvZGUnOiBbJzQ3azlpYTJpZ3hwY2NlMmJ6bzh1M2tqMDMnXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgZGVmYXVsdDpcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBuYW1lOiAnYW16bi1saW51eCcsXG4gICAgICAgIH07XG4gICAgfVxuICB9XG5cbn0iXX0=