# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/01_data.kssl.ipynb.

# %% auto 0
__all__ = ['KSSL']

# %% ../../nbs/01_data.kssl.ipynb 4
#import subprocess
import shutil
from .core import (select_rows, read_opus)
from .transforms.core import interp
from fastcore.basics import patch
from fastcore.xtras import mkdir
import pandas as pd
from pathlib import Path
import glob

#import numpy as np
import re
from tqdm import tqdm
from typing import Dict, Callable

# %% ../../nbs/01_data.kssl.ipynb 5
class KSSL:
    def __init__(self, 
                 src_dir:str, # KSSL directory 
                ):
        self.src_dir = Path(src_dir)

# %% ../../nbs/01_data.kssl.ipynb 6
@patch
def to_csv(self:KSSL,
           dest_dir:str, # Destination directory
          ):
    "Convert KSSL Microsoft Access DB tables to individual `.csv` files" 
    pass

# %% ../../nbs/01_data.kssl.ipynb 7
@patch
def get_sample(self:KSSL):
    """Return KSSL DB `sample` table as `pd.DataFrame`
    
        - Only `smp_id` > 1000 are selected
    """ 
    return pd.read_csv(self.src_dir / 'sample.csv', low_memory=False) \
        .pipe(select_rows, {'smp_id': lambda d: d > 1000}) \
        .loc[:, ['smp_id', 'lay_id']]

# %% ../../nbs/01_data.kssl.ipynb 8
@patch
def get_layer(self:KSSL) -> pd.DataFrame: # Processed DataFrame
    "Return KSSL DB `layer` table as `pd.DataFrame`" 
    return pd.read_csv(self.src_dir / 'layer.csv', low_memory=False) \
        .loc[:, ['lay_id', 'lims_pedon_id', 'lims_site_id', 'lay_depth_to_top']] \
        .dropna() \
        .astype({'lims_pedon_id': 'int32', 'lims_site_id': 'int32'})

# %% ../../nbs/01_data.kssl.ipynb 9
@patch
def get_layer_analyte(self:KSSL):
    """Return KSSL DB `layer_analyte` table as `pd.DataFrame`
        
        - Only `master_prep_id` relevant to MIRS analysis selected: `[18, 19, 27, 28]`
        - `calc_value` are by default `str` as possibly containing values such as (`slight`, `1:2`, ...). 
          Only numeric ones are selected.
    """
    return pd.read_csv(self.src_dir / 'layer_analyte.csv', low_memory=False) \
        .dropna(subset=['analyte_id', 'calc_value']) \
        .pipe(select_rows, {
            'master_prep_id': lambda d: d in [18, 19, 27, 28],
            'calc_value': lambda d: re.search(r'[a-zA-Z]|:|\s', str(d)) is None}) \
        .loc[:, ['lay_id', 'analyte_id', 'calc_value']] \
        .astype({'calc_value': float})

# %% ../../nbs/01_data.kssl.ipynb 10
@patch
def get_mirs_mas(self:KSSL):
    "Return KSSL DB `mir_scan_mas_data.csv` table as `pd.DataFrame`"
    return pd.read_csv(self.src_dir / 'mir_scan_mas_data.csv', low_memory=False) \
        .loc[:, ['smp_id', 'mir_scan_mas_id']]

# %% ../../nbs/01_data.kssl.ipynb 11
@patch
def get_mirs_det(self:KSSL):
    """Return KSSL DB `sample` table as `pd.DataFrame`
    
        - Only `scan_path_name` containing valid substring `['XN', 'XS']

    """ 
    return pd.read_csv(self.src_dir / 'mir_scan_det_data.csv', low_memory=False) \
        .dropna(subset=['scan_path_name', 'mir_scan_mas_id']) \
        .loc[:, ['mir_scan_mas_id', 'scan_path_name']] \
        .pipe(select_rows, {
            'scan_path_name': lambda d: re.search(r'X.', str(d))[0] in ['XN', 'XS']})

# %% ../../nbs/01_data.kssl.ipynb 12
@patch
def get_analyte_lut(self:KSSL):
    "Return KSSL DB `analyte` lookup table as `pd.DataFrame`"""
    df = pd.read_csv(self.src_dir / 'analyte.csv') \
        .loc[:, ['analyte_id', 'analyte_name', 'analyte_abbrev', 'uom_abbrev']]
    return df

# %% ../../nbs/01_data.kssl.ipynb 15
@patch
def get_taxonomy_lut(self:KSSL):
    pass

# %% ../../nbs/01_data.kssl.ipynb 16
@patch
def get_counts(self:KSSL):
    pass

# %% ../../nbs/01_data.kssl.ipynb 17
@patch
def join_mirs(self:KSSL,
              analytes:list[int]=[725, 336], # Analytes of interest
             ):
    "Return master MIRS join to be exported as `pd.DataFrame`"
    lay_smp = pd.merge(self.get_layer(),
                       self.get_sample(), 
                       on='lay_id', how='inner').sort_values(by='lay_id')

    lay_smp_mirsmas = pd.merge(lay_smp, 
                               self.get_mirs_mas(), 
                               on='smp_id', how='inner')
    
    layer_analyte_of_interest = self.get_layer_analyte() \
                                    .pipe(select_rows, 
                                          {'analyte_id': lambda d: d in analytes})
    
    lay_anal_wide = pd.pivot_table(layer_analyte_of_interest,
                                   values='calc_value',
                                   index=['lay_id'], columns=['analyte_id'])
    
    lay_smp_mirs_mas_det = pd.merge(lay_smp_mirsmas,
                                    self.get_mirs_det(),
                                    on='mir_scan_mas_id', how='inner')
    
    return pd.merge(lay_smp_mirs_mas_det,
                    lay_anal_wide, on='lay_id', how='left').dropna(subset=analytes)

# %% ../../nbs/01_data.kssl.ipynb 19
@patch
def join_nirs(self:KSSL):
    pass

# %% ../../nbs/01_data.kssl.ipynb 20
@patch
def bundle(self:KSSL,
           src_dir_spectra:str, # Directory containing infrared spectra
           dest_dir:str, # Destination directory
           analytes:list[int]=[725, 723, 334], # Analytes of interest
           compressed=True, # True if folder should be compressed
          ):
    """Bundle KSSL dataset as follows:

        - /dest_dir/
        -          /layer_id/
        -                    target.csv
    """
    dest_dir = Path(dest_dir)        
    mkdir(dest_dir, overwrite=True)
    
    df = self.join_mirs(analytes=analytes)
    
    # iterate and create tree structure
    for index, row in tqdm(df.iterrows(), total=df.shape[0]):
        path = dest_dir / str(row['lay_id'])
        if not path.exists():
            path.mkdir()
            target = row[analytes]
            target.index.name = 'analyte'
            target.name = 'value'
            target.to_csv(path/'target.csv')

        # read, process and bundle spectra    
        fnames = glob.glob(src_dir_spectra+ '/*/' + row['scan_path_name'], 
                             recursive=True)
        if len(fnames):
            absorbance, wn = interp(read_opus(fnames[0]))
            spectrum = pd.Series(data=absorbance, index=wn, name='absorbance')
            spectrum.index.name = 'wavenumber'
            out_fname = str(path / Path(row['scan_path_name']).stem) + '.csv'
            spectrum.to_csv(out_fname)
