
import argparse
from pathlib import Path

import requests
from codeowners_backstage.backstage_user_groups import BackstageUserGroups
from codeowners_backstage.codeowners_preprocessor import CodeOwnersPreprocessor

parser = argparse.ArgumentParser(description="Preprocesses CODEOWNERS file, substituting group names with member "
                                             "emails using information taken from a Backstage catalog."
                                             "Group names are expected to start with the \"@\" symbol.")
parser.add_argument('file', type=Path, help='Input file')
parser.add_argument('-H', '--host', required=True, type=str, help='URL to the Backstage host')
parser.add_argument('--disable-ssl-verification', action='store_const', const=True, default=False)
parser.add_argument('-n', '--namespace', default='default', type=str,
                    help='Namespace in Backstage (default: %(default)r)')
parser.add_argument('-o', '--out', type=Path, help='Output file (default: stdout)')
parser.add_argument('--preamble', type=str, default='# THIS FILE IS AUTOGENERATED FROM {filename}. DO NOT EDIT THIS FILE DIRECTLY.',
                    help="Content to prepend to the generated file. Default:\n%(default)s")
parser.add_argument('--preamble-separator-length', type=int, default=100, help='Preamble separator length. Default:\n%(default)s')


def preprocess(read_input, write_output, get_group_members, **options):
    input_str = read_input()
    preprocessor = CodeOwnersPreprocessor(get_group_members, **options)
    output_str = preprocessor(input_str)
    write_output(output_str)


if __name__ == "__main__":
    args = parser.parse_args()
    session = requests.Session()
    session.verify = args.disable_ssl_verification
    user_groups = BackstageUserGroups.load(args.host, args.namespace, session)
    preprocess(read_input=args.file.read_text,
               write_output=args.out.write_text if args.out is not None else print,
               get_group_members=user_groups.get_group_members,
               preamble=args.preamble.format(filename=args.file.name),
               preamble_separator_length=args.preamble_separator_length)
