"""Main Sandbox class for Firework client."""

import asyncio
from typing import Callable, Dict, List, Optional, Any

from firework.backend import SandboxBackend, get_backend
from firework.config import LocalConfig, get_config
from firework.exceptions import SandboxNotFound
from firework.filesystem import FilesystemManager
from firework.models import SandboxInfo, SandboxMetrics, SSHInfo
from firework.process import ProcessManager


class HTTPClient:
    """HTTP client for making requests to sandbox services."""

    def __init__(self, sandbox: "Sandbox"):
        self._sandbox = sandbox
        self._port_map: Dict[int, int] = {}

    async def get(
        self,
        port: int,
        path: str = "/",
        headers: Optional[Dict[str, str]] = None,
    ) -> Dict[str, Any]:
        """Make a GET request to a sandbox service."""
        import aiohttp

        url = self._sandbox.local_url(port, path)
        async with aiohttp.ClientSession() as session:
            async with session.get(url, headers=headers) as resp:
                return {
                    "status": resp.status,
                    "headers": dict(resp.headers),
                    "body": await resp.text(),
                }

    async def post(
        self,
        port: int,
        path: str = "/",
        json: Optional[Dict] = None,
        data: Optional[str] = None,
        headers: Optional[Dict[str, str]] = None,
    ) -> Dict[str, Any]:
        """Make a POST request to a sandbox service."""
        import aiohttp

        url = self._sandbox.local_url(port, path)
        async with aiohttp.ClientSession() as session:
            async with session.post(
                url, json=json, data=data, headers=headers
            ) as resp:
                return {
                    "status": resp.status,
                    "headers": dict(resp.headers),
                    "body": await resp.text(),
                }


class Sandbox:
    """High-level sandbox interface.

    Example:
        async with Sandbox.create(template="base") as sandbox:
            result = await sandbox.process.exec("echo Hello")
            print(result.stdout)
    """

    def __init__(
        self,
        info: SandboxInfo,
        backend: SandboxBackend,
        config: LocalConfig,
    ):
        self._info = info
        self._backend = backend
        self._config = config
        self._event_handlers: Dict[str, List[Callable]] = {}

        # Sub-managers
        self.process = ProcessManager(self)
        self.filesystem = FilesystemManager(self)
        self.http = HTTPClient(self)

    @property
    def id(self) -> str:
        """Sandbox ID."""
        return self._info.id

    @property
    def root_path(self) -> str:
        """Root path of sandbox on host."""
        return self._info.root_path

    @property
    def info(self) -> SandboxInfo:
        """Full sandbox information."""
        return self._info

    @classmethod
    async def create(
        cls,
        template: str = "base",
        name: Optional[str] = None,
        environment: Optional[Dict[str, str]] = None,
        vcpu: Optional[int] = None,
        memory_mb: Optional[int] = None,
        timeout_seconds: Optional[int] = None,
        max_idle_seconds: Optional[int] = None,
        persistent: bool = False,
        keep_alive: bool = False,
        config: Optional[LocalConfig] = None,
    ) -> "Sandbox":
        """Create a new sandbox.

        Args:
            template: Template name (base, python-ml, nodejs, custom).
            name: Optional human-readable name.
            environment: Environment variables.
            vcpu: Number of virtual CPUs (1-2).
            memory_mb: Memory in MB (256-1024).
            timeout_seconds: Default command timeout.
            max_idle_seconds: Auto-destroy after idle time.
            persistent: Keep sandbox after context exit.
            keep_alive: Don't destroy on idle.
            config: Optional custom configuration.

        Returns:
            Sandbox instance.
        """
        cfg = config or get_config()
        backend = get_backend(cfg.runtime_dir)

        info = await backend.create(
            template=template,
            name=name,
            environment=environment or {},
            vcpu=vcpu or cfg.default_vcpu,
            memory_mb=memory_mb or cfg.default_memory_mb,
            timeout_seconds=timeout_seconds or cfg.default_timeout,
        )

        sandbox = cls(info=info, backend=backend, config=cfg)
        sandbox._emit("created", sandbox)
        return sandbox

    @classmethod
    async def reconnect(
        cls,
        sandbox_id: str,
        config: Optional[LocalConfig] = None,
    ) -> "Sandbox":
        """Reconnect to an existing sandbox.

        Args:
            sandbox_id: ID of existing sandbox.
            config: Optional custom configuration.

        Returns:
            Sandbox instance.

        Raises:
            SandboxNotFound: If sandbox doesn't exist.
        """
        cfg = config or get_config()
        backend = get_backend(cfg.runtime_dir)

        info = await backend.get_info(sandbox_id)
        if info is None:
            raise SandboxNotFound(sandbox_id)

        return cls(info=info, backend=backend, config=cfg)

    async def get_state(self) -> str:
        """Get current sandbox state.

        Returns:
            State string: "running", "paused", or "stopped".
        """
        return await self._backend.get_state(self.id)

    async def pause(self) -> None:
        """Pause the sandbox (freeze state)."""
        await self._backend.pause(self.id)
        self._info.state = "paused"

    async def resume(self) -> None:
        """Resume a paused sandbox."""
        await self._backend.resume(self.id)
        self._info.state = "running"

    async def destroy(self) -> None:
        """Destroy the sandbox and cleanup resources."""
        self._emit("destroyed", self)
        await self._backend.destroy(self.id)
        self._info.state = "stopped"

    async def get_metrics(self) -> SandboxMetrics:
        """Get resource usage metrics.

        Returns:
            SandboxMetrics with cpu, memory, disk, uptime, process count.
        """
        return await self._backend.get_metrics(self.id)

    async def get_ssh_info(self) -> SSHInfo:
        """Get SSH connection information for debugging.

        Returns:
            SSHInfo with host, port, user.
        """
        # For Docker, we'd need to expose SSH port
        return SSHInfo(
            host="127.0.0.1",
            port=22,
            user="root",
        )

    def local_url(self, port: int, path: str = "/") -> str:
        """Get local URL for a sandbox service.

        Args:
            port: Port number inside sandbox.
            path: URL path.

        Returns:
            Full URL string.
        """
        # For Docker, we'd need port mapping
        # This is a simplified version
        return f"http://127.0.0.1:{port}{path}"

    def on(self, event: str, callback: Callable) -> None:
        """Register an event handler.

        Args:
            event: Event name (created, destroyed).
            callback: Callback function.
        """
        if event not in self._event_handlers:
            self._event_handlers[event] = []
        self._event_handlers[event].append(callback)

    def _emit(self, event: str, *args) -> None:
        """Emit an event to registered handlers."""
        for handler in self._event_handlers.get(event, []):
            try:
                handler(*args)
            except Exception:
                pass  # Don't let handler errors break sandbox

    async def __aenter__(self) -> "Sandbox":
        """Async context manager entry."""
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb) -> None:
        """Async context manager exit - auto-destroy."""
        await self.destroy()
