"""Process execution for Firework sandbox client."""

import asyncio
import time
from typing import AsyncIterator, List, Optional, TYPE_CHECKING

from firework.models import ProcessResult, ProcessStreamEvent
from firework.exceptions import ProcessExecutionError, SandboxTimeout

if TYPE_CHECKING:
    from firework.sandbox import Sandbox


class BackgroundProcess:
    """Handle for a background process running in a sandbox."""

    def __init__(
        self,
        sandbox: "Sandbox",
        command: str,
        pid: Optional[int] = None,
    ):
        self._sandbox = sandbox
        self._command = command
        self._pid = pid
        self._exit_code: Optional[int] = None

    @property
    def pid(self) -> Optional[int]:
        """Process ID if available."""
        return self._pid

    async def is_running(self) -> bool:
        """Check if the process is still running."""
        if self._exit_code is not None:
            return False
        # Check via sandbox backend
        return await self._sandbox._backend.is_process_running(
            self._sandbox.id, self._pid
        )

    async def wait(self, timeout_seconds: Optional[float] = None) -> int:
        """Wait for the process to complete."""
        start = time.time()
        while await self.is_running():
            if timeout_seconds and (time.time() - start) > timeout_seconds:
                raise asyncio.TimeoutError()
            await asyncio.sleep(0.1)
        return self._exit_code or 0

    async def kill(self) -> None:
        """Kill the background process."""
        if self._pid:
            await self._sandbox._backend.kill_process(self._sandbox.id, self._pid)
        self._exit_code = -9


class ProcessManager:
    """Manages process execution within a sandbox."""

    def __init__(self, sandbox: "Sandbox"):
        self._sandbox = sandbox

    async def exec(
        self,
        command: str,
        cwd: str = "/",
        environment: Optional[dict] = None,
        timeout_seconds: Optional[float] = None,
    ) -> ProcessResult:
        """Execute a command and wait for completion.

        Args:
            command: Command to execute.
            cwd: Working directory.
            environment: Additional environment variables.
            timeout_seconds: Maximum execution time.

        Returns:
            ProcessResult with stdout, stderr, exit_code, runtime_seconds.

        Raises:
            SandboxTimeout: If execution exceeds timeout.
            ProcessExecutionError: If process fails and raise_on_error is True.
        """
        timeout = timeout_seconds or self._sandbox._config.default_timeout
        start_time = time.time()

        try:
            result = await asyncio.wait_for(
                self._sandbox._backend.exec_command(
                    sandbox_id=self._sandbox.id,
                    command=command,
                    cwd=cwd,
                    environment=environment or {},
                ),
                timeout=timeout,
            )
            runtime = time.time() - start_time
            return ProcessResult(
                stdout=result.get("stdout", ""),
                stderr=result.get("stderr", ""),
                exit_code=result.get("exit_code", 0),
                runtime_seconds=runtime,
            )
        except asyncio.TimeoutError:
            raise SandboxTimeout("exec", timeout)

    async def exec_stream(
        self,
        command: str,
        cwd: str = "/",
        environment: Optional[dict] = None,
    ) -> AsyncIterator[ProcessStreamEvent]:
        """Execute a command with streaming output.

        Args:
            command: Command to execute.
            cwd: Working directory.
            environment: Additional environment variables.

        Yields:
            ProcessStreamEvent for stdout, stderr, and exit.
        """
        async for event in self._sandbox._backend.exec_stream(
            sandbox_id=self._sandbox.id,
            command=command,
            cwd=cwd,
            environment=environment or {},
        ):
            yield ProcessStreamEvent(
                type=event.get("type", "stdout"),
                content=event.get("content", ""),
                exit_code=event.get("exit_code", 0),
            )

    async def start(
        self,
        command: str,
        cwd: str = "/",
        environment: Optional[dict] = None,
        background: bool = True,
    ) -> BackgroundProcess:
        """Start a background process.

        Args:
            command: Command to execute.
            cwd: Working directory.
            environment: Additional environment variables.
            background: Must be True for this method.

        Returns:
            BackgroundProcess handle.
        """
        result = await self._sandbox._backend.start_background(
            sandbox_id=self._sandbox.id,
            command=command,
            cwd=cwd,
            environment=environment or {},
        )
        return BackgroundProcess(
            sandbox=self._sandbox,
            command=command,
            pid=result.get("pid"),
        )

    async def batch_exec(
        self,
        commands: List[str],
        cwd: str = "/",
        environment: Optional[dict] = None,
        stop_on_error: bool = True,
    ) -> List[ProcessResult]:
        """Execute multiple commands sequentially.

        Args:
            commands: List of commands to execute.
            cwd: Working directory for all commands.
            environment: Additional environment variables.
            stop_on_error: Stop execution if a command fails.

        Returns:
            List of ProcessResult for each command.
        """
        results = []
        for command in commands:
            result = await self.exec(command, cwd=cwd, environment=environment)
            results.append(result)
            if stop_on_error and result.exit_code != 0:
                break
        return results
