"""Filesystem operations for Firework sandbox client."""

import os
import shutil
from pathlib import Path
from typing import List, Optional, AsyncIterator, TYPE_CHECKING

from firework.models import FileEntry
from firework.exceptions import FilesystemError

if TYPE_CHECKING:
    from firework.sandbox import Sandbox


class FileWatcher:
    """Watches a directory for changes."""

    def __init__(self, sandbox: "Sandbox", path: str):
        self._sandbox = sandbox
        self._path = path
        self._running = False

    async def __aiter__(self) -> AsyncIterator[dict]:
        """Iterate over file change events."""
        self._running = True
        async for event in self._sandbox._backend.watch_directory(
            self._sandbox.id, self._path
        ):
            if not self._running:
                break
            yield event

    async def stop(self) -> None:
        """Stop watching."""
        self._running = False


class FilesystemManager:
    """Manages filesystem operations within a sandbox."""

    def __init__(self, sandbox: "Sandbox"):
        self._sandbox = sandbox

    async def read(self, path: str) -> str:
        """Read file contents as text.

        Args:
            path: Path to file in sandbox.

        Returns:
            File contents as string.

        Raises:
            FilesystemError: If read fails.
        """
        try:
            return await self._sandbox._backend.read_file(self._sandbox.id, path)
        except Exception as e:
            raise FilesystemError("read", path, str(e))

    async def read_bytes(self, path: str) -> bytes:
        """Read file contents as bytes.

        Args:
            path: Path to file in sandbox.

        Returns:
            File contents as bytes.
        """
        try:
            return await self._sandbox._backend.read_file_bytes(self._sandbox.id, path)
        except Exception as e:
            raise FilesystemError("read_bytes", path, str(e))

    async def write(self, path: str, content: str) -> None:
        """Write text content to file.

        Args:
            path: Path to file in sandbox.
            content: Text content to write.
        """
        try:
            await self._sandbox._backend.write_file(self._sandbox.id, path, content)
        except Exception as e:
            raise FilesystemError("write", path, str(e))

    async def write_bytes(self, path: str, content: bytes) -> None:
        """Write binary content to file.

        Args:
            path: Path to file in sandbox.
            content: Binary content to write.
        """
        try:
            await self._sandbox._backend.write_file_bytes(
                self._sandbox.id, path, content
            )
        except Exception as e:
            raise FilesystemError("write_bytes", path, str(e))

    async def upload(self, local_path: str, remote_path: str) -> None:
        """Upload a local file to the sandbox.

        Args:
            local_path: Path to local file.
            remote_path: Destination path in sandbox.
        """
        try:
            local = Path(local_path)
            if not local.exists():
                raise FileNotFoundError(f"Local file not found: {local_path}")

            content = local.read_bytes()
            await self.write_bytes(remote_path, content)
        except FilesystemError:
            raise
        except Exception as e:
            raise FilesystemError("upload", local_path, str(e))

    async def download(self, remote_path: str, local_path: str) -> None:
        """Download a file from the sandbox.

        Args:
            remote_path: Path to file in sandbox.
            local_path: Destination path on local filesystem.
        """
        try:
            content = await self.read_bytes(remote_path)
            local = Path(local_path)
            local.parent.mkdir(parents=True, exist_ok=True)
            local.write_bytes(content)
        except FilesystemError:
            raise
        except Exception as e:
            raise FilesystemError("download", remote_path, str(e))

    async def upload_dir(self, local_path: str, remote_path: str) -> None:
        """Upload a local directory to the sandbox.

        Args:
            local_path: Path to local directory.
            remote_path: Destination path in sandbox.
        """
        local = Path(local_path)
        if not local.is_dir():
            raise FilesystemError("upload_dir", local_path, "Not a directory")

        await self.mkdir(remote_path, recursive=True)

        for item in local.rglob("*"):
            if item.is_file():
                rel_path = item.relative_to(local)
                dest = f"{remote_path}/{rel_path}"
                await self.upload(str(item), dest)

    async def download_dir(self, remote_path: str, local_path: str) -> None:
        """Download a directory from the sandbox.

        Args:
            remote_path: Path to directory in sandbox.
            local_path: Destination path on local filesystem.
        """
        local = Path(local_path)
        local.mkdir(parents=True, exist_ok=True)

        entries = await self.list(remote_path)
        for entry in entries:
            remote_item = f"{remote_path}/{entry.name}"
            local_item = local / entry.name

            if entry.type == "directory":
                await self.download_dir(remote_item, str(local_item))
            else:
                await self.download(remote_item, str(local_item))

    async def list(self, path: str) -> List[FileEntry]:
        """List directory contents.

        Args:
            path: Path to directory in sandbox.

        Returns:
            List of FileEntry objects.
        """
        try:
            entries = await self._sandbox._backend.list_directory(self._sandbox.id, path)
            return [
                FileEntry(
                    name=e.get("name", ""),
                    size=e.get("size", 0),
                    type=e.get("type", "file"),
                    modified_at=e.get("modified_at"),
                )
                for e in entries
            ]
        except Exception as e:
            raise FilesystemError("list", path, str(e))

    async def mkdir(self, path: str, recursive: bool = False) -> None:
        """Create a directory.

        Args:
            path: Path to create.
            recursive: Create parent directories if needed.
        """
        try:
            await self._sandbox._backend.mkdir(self._sandbox.id, path, recursive)
        except Exception as e:
            raise FilesystemError("mkdir", path, str(e))

    async def remove(self, path: str) -> None:
        """Remove a file.

        Args:
            path: Path to file to remove.
        """
        try:
            await self._sandbox._backend.remove(self._sandbox.id, path)
        except Exception as e:
            raise FilesystemError("remove", path, str(e))

    async def remove_dir(self, path: str) -> None:
        """Remove a directory and its contents.

        Args:
            path: Path to directory to remove.
        """
        try:
            await self._sandbox._backend.remove_dir(self._sandbox.id, path)
        except Exception as e:
            raise FilesystemError("remove_dir", path, str(e))

    async def exists(self, path: str) -> bool:
        """Check if a path exists.

        Args:
            path: Path to check.

        Returns:
            True if path exists.
        """
        try:
            return await self._sandbox._backend.exists(self._sandbox.id, path)
        except Exception:
            return False

    async def watch_dir(self, path: str) -> FileWatcher:
        """Watch a directory for changes.

        Args:
            path: Path to directory to watch.

        Returns:
            FileWatcher async iterator.
        """
        return FileWatcher(self._sandbox, path)
