"""Exception classes for Firework sandbox client."""


class SandboxError(Exception):
    """Base exception for all sandbox errors."""

    pass


class SandboxNotFound(SandboxError):
    """Raised when a sandbox cannot be found."""

    def __init__(self, sandbox_id: str):
        self.sandbox_id = sandbox_id
        super().__init__(f"Sandbox not found: {sandbox_id}")


class SandboxTimeout(SandboxError):
    """Raised when a sandbox operation times out."""

    def __init__(self, operation: str, timeout_seconds: float):
        self.operation = operation
        self.timeout_seconds = timeout_seconds
        super().__init__(f"Operation '{operation}' timed out after {timeout_seconds}s")


class ProcessExecutionError(SandboxError):
    """Raised when a process execution fails."""

    def __init__(self, exit_code: int, stderr: str, command: str = ""):
        self.exit_code = exit_code
        self.stderr = stderr
        self.command = command
        super().__init__(f"Process failed with exit code {exit_code}: {stderr[:200]}")


class FilesystemError(SandboxError):
    """Raised when a filesystem operation fails."""

    def __init__(self, operation: str, path: str, message: str):
        self.operation = operation
        self.path = path
        super().__init__(f"Filesystem {operation} failed for '{path}': {message}")
