"""Pre-built environment definitions and builder for Firework sandboxes."""

import asyncio
import json
import tempfile
from dataclasses import dataclass, field
from datetime import datetime, timezone
from pathlib import Path
from typing import Dict, List, Optional


@dataclass
class EnvironmentSpec:
    """Specification for a sandbox environment."""

    name: str
    base_image: str
    description: str
    python_version: str = "3.11"
    system_packages: List[str] = field(default_factory=list)
    pip_packages: List[str] = field(default_factory=list)
    env_vars: Dict[str, str] = field(default_factory=dict)
    setup_commands: List[str] = field(default_factory=list)


# Pre-defined environments
BUILT_IN_ENVIRONMENTS: Dict[str, EnvironmentSpec] = {
    "base": EnvironmentSpec(
        name="base",
        base_image="python:3.11-slim",
        description="Minimal Python 3.11 environment",
        pip_packages=["pip", "setuptools", "wheel"],
    ),
    "python-ml": EnvironmentSpec(
        name="python-ml",
        base_image="python:3.11-slim",
        description="Python with NumPy, Pandas, Scikit-learn",
        pip_packages=[
            "numpy",
            "pandas",
            "scikit-learn",
            "matplotlib",
            "seaborn",
            "scipy",
        ],
        env_vars={"PYTHONUNBUFFERED": "1"},
    ),
    "python-torch": EnvironmentSpec(
        name="python-torch",
        base_image="python:3.11-slim",
        description="Python with PyTorch for deep learning",
        pip_packages=[
            "torch",
            "torchvision",
            "torchaudio",
            "numpy",
            "pandas",
        ],
        env_vars={"PYTHONUNBUFFERED": "1"},
    ),
    "python-tensorflow": EnvironmentSpec(
        name="python-tensorflow",
        base_image="python:3.11-slim",
        description="Python with TensorFlow for deep learning",
        pip_packages=[
            "tensorflow",
            "numpy",
            "pandas",
            "keras",
        ],
        env_vars={"PYTHONUNBUFFERED": "1"},
    ),
    "python-data": EnvironmentSpec(
        name="python-data",
        base_image="python:3.11-slim",
        description="Python for data engineering",
        pip_packages=[
            "numpy",
            "pandas",
            "polars",
            "pyarrow",
            "duckdb",
            "sqlalchemy",
        ],
        env_vars={"PYTHONUNBUFFERED": "1"},
    ),
    "python-web": EnvironmentSpec(
        name="python-web",
        base_image="python:3.11-slim",
        description="Python for web development",
        pip_packages=[
            "fastapi",
            "uvicorn",
            "httpx",
            "aiohttp",
            "pydantic",
            "jinja2",
        ],
        env_vars={"PYTHONUNBUFFERED": "1"},
    ),
    "python-llm": EnvironmentSpec(
        name="python-llm",
        base_image="python:3.11-slim",
        description="Python for LLM/AI applications",
        pip_packages=[
            "openai",
            "anthropic",
            "langchain",
            "tiktoken",
            "transformers",
            "sentence-transformers",
        ],
        env_vars={"PYTHONUNBUFFERED": "1"},
    ),
    "nodejs": EnvironmentSpec(
        name="nodejs",
        base_image="node:20-slim",
        description="Node.js 20 runtime",
        python_version="",
        system_packages=["curl"],
    ),
    "nodejs-full": EnvironmentSpec(
        name="nodejs-full",
        base_image="node:20-slim",
        description="Node.js with common packages",
        python_version="",
        system_packages=["curl", "git"],
        setup_commands=[
            "npm install -g typescript ts-node",
        ],
    ),
}


@dataclass
class BuiltEnvironment:
    """Information about a built environment."""

    name: str
    image_path: Path
    size_mb: int
    built_at: datetime
    spec: EnvironmentSpec


def _default_cache_dir() -> Path:
    """Get default cache directory (user-writable).
    
    Uses FIREWORK_ENV_DIR environment variable if set,
    otherwise ~/.firework/environments.
    """
    from firework.config import _default_env_dir
    return Path(_default_env_dir())


class EnvironmentBuilder:
    """Builds sandbox environments from specifications."""

    def __init__(self, cache_dir: Optional[Path] = None):
        self.cache_dir = cache_dir or _default_cache_dir()
        self.cache_dir.mkdir(parents=True, exist_ok=True)
        self._built_envs: Dict[str, BuiltEnvironment] = {}
        self._load_cache()

    def _load_cache(self) -> None:
        """Load cached environment metadata."""
        cache_file = self.cache_dir / "environments.json"
        if cache_file.exists():
            try:
                data = json.loads(cache_file.read_text())
                for name, info in data.items():
                    spec = BUILT_IN_ENVIRONMENTS.get(name)
                    if spec and Path(info["image_path"]).exists():
                        self._built_envs[name] = BuiltEnvironment(
                            name=name,
                            image_path=Path(info["image_path"]),
                            size_mb=info["size_mb"],
                            built_at=datetime.fromisoformat(info["built_at"]),
                            spec=spec,
                        )
            except Exception:
                pass

    def _save_cache(self) -> None:
        """Save environment metadata to cache."""
        cache_file = self.cache_dir / "environments.json"
        data = {
            name: {
                "image_path": str(env.image_path),
                "size_mb": env.size_mb,
                "built_at": env.built_at.isoformat(),
            }
            for name, env in self._built_envs.items()
        }
        cache_file.write_text(json.dumps(data, indent=2))

    def list_available(self) -> List[EnvironmentSpec]:
        """List all available environment specifications."""
        return list(BUILT_IN_ENVIRONMENTS.values())

    def list_built(self) -> List[BuiltEnvironment]:
        """List all built environments."""
        return list(self._built_envs.values())

    def is_built(self, name: str) -> bool:
        """Check if an environment is already built."""
        return name in self._built_envs

    def get_image_path(self, name: str) -> Optional[Path]:
        """Get the rootfs image path for a built environment."""
        if name in self._built_envs:
            return self._built_envs[name].image_path
        return None

    async def build(
        self,
        name: str,
        force_rebuild: bool = False,
        size_mb: int = 2048,
    ) -> BuiltEnvironment:
        """Build an environment from specification.

        Args:
            name: Environment name (from BUILT_IN_ENVIRONMENTS or custom).
            force_rebuild: Rebuild even if cached.
            size_mb: Size of the rootfs image in MB.

        Returns:
            BuiltEnvironment with image path and metadata.
        """
        if not force_rebuild and name in self._built_envs:
            return self._built_envs[name]

        if name not in BUILT_IN_ENVIRONMENTS:
            raise ValueError(f"Unknown environment: {name}. Use list_available() to see options.")

        spec = BUILT_IN_ENVIRONMENTS[name]
        return await self._build_from_spec(spec, size_mb)

    async def build_custom(
        self,
        spec: EnvironmentSpec,
        size_mb: int = 2048,
    ) -> BuiltEnvironment:
        """Build a custom environment from specification.

        Args:
            spec: Custom environment specification.
            size_mb: Size of the rootfs image in MB.

        Returns:
            BuiltEnvironment with image path and metadata.
        """
        return await self._build_from_spec(spec, size_mb)

    async def _build_from_spec(
        self,
        spec: EnvironmentSpec,
        size_mb: int,
    ) -> BuiltEnvironment:
        """Build environment from specification."""
        # Generate Dockerfile
        dockerfile = self._generate_dockerfile(spec)

        with tempfile.TemporaryDirectory() as temp_dir:
            temp_path = Path(temp_dir)

            # Write Dockerfile
            dockerfile_path = temp_path / "Dockerfile"
            dockerfile_path.write_text(dockerfile)

            # Write init script
            init_script = self._generate_init_script(spec)
            init_path = temp_path / "init.sh"
            init_path.write_text(init_script)

            # Build Docker image
            image_tag = f"firework-env-{spec.name}:latest"
            await self._docker_build(temp_path, image_tag)

            # Convert to rootfs (for Firecracker)
            # For now, we just track the Docker image
            image_path = self.cache_dir / f"{spec.name}.ext4"

            # Create placeholder (actual rootfs conversion would happen here)
            # In production, use RootfsBuilder to create ext4 image
            image_path.touch()

        built = BuiltEnvironment(
            name=spec.name,
            image_path=image_path,
            size_mb=size_mb,
            built_at=datetime.now(timezone.utc),
            spec=spec,
        )

        self._built_envs[spec.name] = built
        self._save_cache()

        return built

    def _generate_dockerfile(self, spec: EnvironmentSpec) -> str:
        """Generate Dockerfile from specification."""
        lines = [f"FROM {spec.base_image}"]

        # Environment variables
        for key, value in spec.env_vars.items():
            lines.append(f"ENV {key}={value}")

        # System packages
        if spec.system_packages:
            packages = " ".join(spec.system_packages)
            lines.append(
                f"RUN apt-get update && apt-get install -y {packages} && rm -rf /var/lib/apt/lists/*"
            )

        # Python packages
        if spec.pip_packages:
            packages = " ".join(spec.pip_packages)
            lines.append(f"RUN pip install --no-cache-dir {packages}")

        # Setup commands
        for cmd in spec.setup_commands:
            lines.append(f"RUN {cmd}")

        # Copy init script
        lines.append("COPY init.sh /sbin/init")
        lines.append("RUN chmod +x /sbin/init")

        # Create workspace
        lines.append("RUN mkdir -p /workspace /app")
        lines.append("WORKDIR /workspace")

        return "\n".join(lines)

    def _generate_init_script(self, spec: EnvironmentSpec) -> str:
        """Generate init script for the environment."""
        return """#!/bin/sh
set -e

# Mount essential filesystems
mount -t proc proc /proc 2>/dev/null || true
mount -t sysfs sysfs /sys 2>/dev/null || true
mount -t devtmpfs devtmpfs /dev 2>/dev/null || true
mkdir -p /dev/pts
mount -t devpts devpts /dev/pts 2>/dev/null || true
mount -t tmpfs tmpfs /tmp 2>/dev/null || true
mount -t tmpfs tmpfs /run 2>/dev/null || true

# Set up environment
export PATH=/usr/local/bin:/usr/bin:/bin:/sbin:/usr/sbin
export HOME=/root
export PYTHONUNBUFFERED=1

# Start shell or specified command
exec "$@"
"""

    async def _docker_build(self, context_path: Path, tag: str) -> None:
        """Build Docker image."""
        proc = await asyncio.create_subprocess_exec(
            "docker", "build", "-t", tag, str(context_path),
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, stderr = await proc.communicate()

        if proc.returncode != 0:
            raise RuntimeError(f"Docker build failed: {stderr.decode()}")


# Convenience functions
def list_environments() -> List[Dict]:
    """List all available environments with their details.

    Returns:
        List of environment info dictionaries.
    """
    return [
        {
            "name": spec.name,
            "description": spec.description,
            "base_image": spec.base_image,
            "pip_packages": spec.pip_packages,
        }
        for spec in BUILT_IN_ENVIRONMENTS.values()
    ]


def get_environment(name: str) -> Optional[EnvironmentSpec]:
    """Get environment specification by name."""
    return BUILT_IN_ENVIRONMENTS.get(name)
