"""Configuration for Firework sandbox client.

Environment Variables:
    FIREWORK_RUNTIME_DIR: Override the default runtime directory for sandboxes
    FIREWORK_ENV_DIR: Override the default directory for built environments
    FIREWORK_DEFAULT_TEMPLATE: Override the default template (default: "base")
    FIREWORK_DEFAULT_TIMEOUT: Override the default timeout in seconds (default: 60)
    FIREWORK_DEFAULT_VCPU: Override the default vCPU count (default: 1)
    FIREWORK_DEFAULT_MEMORY_MB: Override the default memory in MB (default: 512)
    FIREWORK_LOG_LEVEL: Override the log level (default: "INFO")
"""

from dataclasses import dataclass, field
from pathlib import Path
from typing import Dict, Optional
import os


# Environment variable names
ENV_RUNTIME_DIR = "FIREWORK_RUNTIME_DIR"
ENV_ENV_DIR = "FIREWORK_ENV_DIR"
ENV_DEFAULT_TEMPLATE = "FIREWORK_DEFAULT_TEMPLATE"
ENV_DEFAULT_TIMEOUT = "FIREWORK_DEFAULT_TIMEOUT"
ENV_DEFAULT_VCPU = "FIREWORK_DEFAULT_VCPU"
ENV_DEFAULT_MEMORY_MB = "FIREWORK_DEFAULT_MEMORY_MB"
ENV_LOG_LEVEL = "FIREWORK_LOG_LEVEL"


def _get_default_base_dir() -> Path:
    """Get the base directory for firework data."""
    return Path.home() / ".firework"


def _default_runtime_dir() -> str:
    """Get default runtime directory.
    
    Priority:
    1. FIREWORK_RUNTIME_DIR environment variable
    2. ~/.firework/sandboxes
    """
    env_dir = os.environ.get(ENV_RUNTIME_DIR)
    if env_dir:
        return env_dir
    return str(_get_default_base_dir() / "sandboxes")


def _default_env_dir() -> str:
    """Get default environments directory.
    
    Priority:
    1. FIREWORK_ENV_DIR environment variable
    2. ~/.firework/environments
    """
    env_dir = os.environ.get(ENV_ENV_DIR)
    if env_dir:
        return env_dir
    return str(_get_default_base_dir() / "environments")


def get_env_int(name: str, default: int) -> int:
    """Get an integer from environment variable."""
    value = os.environ.get(name)
    if value:
        try:
            return int(value)
        except ValueError:
            pass
    return default


@dataclass
class LocalConfig:
    """Configuration for the sandbox client.

    All settings have sensible defaults for zero-config usage.
    Settings can be overridden via environment variables.
    
    Environment Variables:
        FIREWORK_RUNTIME_DIR: Runtime directory for sandboxes
        FIREWORK_ENV_DIR: Directory for built environments
        FIREWORK_DEFAULT_TEMPLATE: Default template name
        FIREWORK_DEFAULT_TIMEOUT: Default timeout in seconds
        FIREWORK_DEFAULT_VCPU: Default vCPU count
        FIREWORK_DEFAULT_MEMORY_MB: Default memory in MB
        FIREWORK_LOG_LEVEL: Log level
    """

    runtime_dir: str = ""
    env_dir: str = ""
    default_template: str = ""
    default_timeout: int = 0
    default_vcpu: int = 0
    default_memory_mb: int = 0
    log_level: str = ""

    # Template definitions
    templates: Dict[str, dict] = field(default_factory=dict)

    def __post_init__(self):
        # Set defaults from environment variables or hardcoded defaults
        if not self.runtime_dir:
            self.runtime_dir = _default_runtime_dir()
        
        if not self.env_dir:
            self.env_dir = _default_env_dir()
        
        if not self.default_template:
            self.default_template = os.environ.get(ENV_DEFAULT_TEMPLATE, "base")
        
        if not self.default_timeout:
            self.default_timeout = get_env_int(ENV_DEFAULT_TIMEOUT, 60)
        
        if not self.default_vcpu:
            self.default_vcpu = get_env_int(ENV_DEFAULT_VCPU, 1)
        
        if not self.default_memory_mb:
            self.default_memory_mb = get_env_int(ENV_DEFAULT_MEMORY_MB, 512)
        
        if not self.log_level:
            self.log_level = os.environ.get(ENV_LOG_LEVEL, "INFO")
        
        # Load default templates
        if not self.templates:
            self.templates = {
                "base": {
                    "image": "ubuntu:22.04",
                    "packages": ["python3.11", "curl"],
                },
                "python-ml": {
                    "image": "python:3.11-slim",
                    "packages": ["pip"],
                    "pip": ["numpy", "pandas"],
                },
                "nodejs": {
                    "image": "node:20-slim",
                    "packages": [],
                },
            }

    def get_template(self, name: str) -> dict:
        """Get template configuration by name."""
        if name not in self.templates:
            raise ValueError(f"Unknown template: {name}")
        return self.templates[name]


# Global default configuration
_default_config: Optional[LocalConfig] = None


def get_config() -> LocalConfig:
    """Get the current configuration."""
    global _default_config
    if _default_config is None:
        _default_config = LocalConfig()
    return _default_config


def set_config(config: LocalConfig) -> None:
    """Set the global configuration."""
    global _default_config
    _default_config = config
