"""CLI interface for Firework sandbox client."""

import argparse
import asyncio
import sys


def main():
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(
        description="Firework - Local sandbox client",
        prog="firework",
    )
    
    subparsers = parser.add_subparsers(dest="command", help="Commands")
    
    # Create command
    create_parser = subparsers.add_parser("create", help="Create a new sandbox")
    create_parser.add_argument(
        "--template", "-t",
        default="base",
        help="Template/environment to use (default: base)",
    )
    create_parser.add_argument(
        "--name", "-n",
        help="Optional sandbox name",
    )
    create_parser.add_argument(
        "--vcpu",
        type=int,
        default=1,
        help="Number of vCPUs (default: 1)",
    )
    create_parser.add_argument(
        "--memory",
        type=int,
        default=512,
        help="Memory in MB (default: 512)",
    )
    
    # List command
    list_parser = subparsers.add_parser("list", help="List sandboxes")
    
    # Destroy command
    destroy_parser = subparsers.add_parser("destroy", help="Destroy a sandbox")
    destroy_parser.add_argument("sandbox_id", help="Sandbox ID to destroy")
    
    # Exec command
    exec_parser = subparsers.add_parser("exec", help="Execute command in sandbox")
    exec_parser.add_argument("sandbox_id", help="Sandbox ID")
    exec_parser.add_argument("command", help="Command to execute")
    
    # Environment commands
    env_parser = subparsers.add_parser("env", help="Manage environments")
    env_subparsers = env_parser.add_subparsers(dest="env_command", help="Environment commands")
    
    # env list
    env_list_parser = env_subparsers.add_parser("list", help="List available environments")
    env_list_parser.add_argument(
        "--built", "-b",
        action="store_true",
        help="Show only built environments",
    )
    
    # env build
    env_build_parser = env_subparsers.add_parser("build", help="Build an environment")
    env_build_parser.add_argument("env_name", help="Environment name to build")
    env_build_parser.add_argument(
        "--force", "-f",
        action="store_true",
        help="Force rebuild even if cached",
    )
    env_build_parser.add_argument(
        "--size",
        type=int,
        default=2048,
        help="Rootfs size in MB (default: 2048)",
    )
    
    # env info
    env_info_parser = env_subparsers.add_parser("info", help="Show environment details")
    env_info_parser.add_argument("env_name", help="Environment name")
    
    args = parser.parse_args()
    
    if args.command is None:
        parser.print_help()
        sys.exit(0)
    
    asyncio.run(run_command(args))


async def run_command(args):
    """Run the specified command."""
    from firework import Sandbox
    
    if args.command == "create":
        sandbox = await Sandbox.create(
            template=args.template,
            name=args.name,
            vcpu=args.vcpu,
            memory_mb=args.memory,
        )
        print(f"Created sandbox: {sandbox.id}")
        print(f"  Root path: {sandbox.root_path}")
        print(f"  Template: {args.template}")
        print(f"  vCPU: {args.vcpu}")
        print(f"  Memory: {args.memory}MB")
    
    elif args.command == "destroy":
        sandbox = await Sandbox.reconnect(args.sandbox_id)
        await sandbox.destroy()
        print(f"Destroyed sandbox: {args.sandbox_id}")
    
    elif args.command == "exec":
        sandbox = await Sandbox.reconnect(args.sandbox_id)
        result = await sandbox.process.exec(args.command)
        print(result.stdout, end="")
        if result.stderr:
            print(result.stderr, end="", file=sys.stderr)
        sys.exit(result.exit_code)
    
    elif args.command == "list":
        import os
        from pathlib import Path
        from firework.config import get_config
        
        config = get_config()
        runtime_dir = Path(config.runtime_dir)
        
        if not runtime_dir.exists():
            print("No sandboxes found")
            return
        
        print(f"{'ID':<20} {'State':<10} {'Template':<15}")
        print("-" * 50)
        
        for item in runtime_dir.iterdir():
            if item.is_dir() and item.name.startswith("sbx_"):
                metadata_path = item / "metadata.json"
                if metadata_path.exists():
                    import json
                    data = json.loads(metadata_path.read_text())
                    print(f"{data['id']:<20} {data.get('state', 'unknown'):<10} {data.get('template', 'base'):<15}")
    
    elif args.command == "env":
        await run_env_command(args)


async def run_env_command(args):
    """Run environment subcommands."""
    from firework.environments import (
        EnvironmentBuilder,
        list_environments,
        get_environment,
        BUILT_IN_ENVIRONMENTS,
    )
    
    if args.env_command == "list":
        if args.built:
            # Show only built environments
            builder = EnvironmentBuilder()
            built = builder.list_built()
            
            if not built:
                print("No environments built yet. Use 'firework env build <name>' to build one.")
                return
            
            print(f"{'Name':<20} {'Size (MB)':<12} {'Built At':<25}")
            print("-" * 60)
            for env in built:
                print(f"{env.name:<20} {env.size_mb:<12} {env.built_at.strftime('%Y-%m-%d %H:%M:%S'):<25}")
        else:
            # Show all available environments
            print("Available Environments:")
            print("=" * 70)
            print()
            
            for env in list_environments():
                print(f"  {env['name']:<20} - {env['description']}")
                if env['pip_packages']:
                    packages = ", ".join(env['pip_packages'][:5])
                    if len(env['pip_packages']) > 5:
                        packages += f" (+{len(env['pip_packages']) - 5} more)"
                    print(f"    Packages: {packages}")
                print()
            
            print("Use 'firework env build <name>' to build an environment.")
            print("Use 'firework env info <name>' for full details.")
    
    elif args.env_command == "build":
        builder = EnvironmentBuilder()
        
        if args.env_name not in BUILT_IN_ENVIRONMENTS:
            print(f"Unknown environment: {args.env_name}")
            print("Available: " + ", ".join(BUILT_IN_ENVIRONMENTS.keys()))
            sys.exit(1)
        
        print(f"Building environment: {args.env_name}")
        print(f"  Size: {args.size} MB")
        print(f"  Force rebuild: {args.force}")
        print()
        
        try:
            result = await builder.build(
                args.env_name,
                force_rebuild=args.force,
                size_mb=args.size,
            )
            print(f"✓ Environment built successfully!")
            print(f"  Image path: {result.image_path}")
            print(f"  Size: {result.size_mb} MB")
        except Exception as e:
            print(f"✗ Build failed: {e}")
            sys.exit(1)
    
    elif args.env_command == "info":
        spec = get_environment(args.env_name)
        
        if not spec:
            print(f"Unknown environment: {args.env_name}")
            sys.exit(1)
        
        print(f"Environment: {spec.name}")
        print("=" * 50)
        print(f"Description: {spec.description}")
        print(f"Base Image:  {spec.base_image}")
        
        if spec.python_version:
            print(f"Python:      {spec.python_version}")
        
        if spec.system_packages:
            print(f"\nSystem Packages:")
            for pkg in spec.system_packages:
                print(f"  - {pkg}")
        
        if spec.pip_packages:
            print(f"\nPython Packages:")
            for pkg in spec.pip_packages:
                print(f"  - {pkg}")
        
        if spec.env_vars:
            print(f"\nEnvironment Variables:")
            for k, v in spec.env_vars.items():
                print(f"  {k}={v}")
        
        # Check if built
        builder = EnvironmentBuilder()
        if builder.is_built(args.env_name):
            built = builder._built_envs[args.env_name]
            print(f"\nStatus: Built")
            print(f"  Image: {built.image_path}")
            print(f"  Built: {built.built_at.strftime('%Y-%m-%d %H:%M:%S')}")
        else:
            print(f"\nStatus: Not built")
            print(f"  Run 'firework env build {args.env_name}' to build")
    
    else:
        print("Usage: firework env {list|build|info}")
        sys.exit(1)


if __name__ == "__main__":
    main()
