"""Backend abstraction for Firework sandbox client."""

import asyncio
import json
import os
import shutil
import signal
import subprocess
import time
import uuid
from abc import ABC, abstractmethod
from datetime import datetime, timezone
from pathlib import Path
from typing import Any, AsyncIterator, Dict, List, Optional

from firework.models import SandboxInfo, SandboxMetrics


class SandboxBackend(ABC):
    """Abstract base class for sandbox backends."""

    @abstractmethod
    async def create(
        self,
        template: str,
        name: Optional[str],
        environment: Dict[str, str],
        vcpu: int,
        memory_mb: int,
        timeout_seconds: int,
    ) -> SandboxInfo:
        """Create a new sandbox."""
        pass

    @abstractmethod
    async def destroy(self, sandbox_id: str) -> None:
        """Destroy a sandbox."""
        pass

    @abstractmethod
    async def get_info(self, sandbox_id: str) -> Optional[SandboxInfo]:
        """Get sandbox information."""
        pass

    @abstractmethod
    async def get_state(self, sandbox_id: str) -> str:
        """Get sandbox state."""
        pass

    @abstractmethod
    async def pause(self, sandbox_id: str) -> None:
        """Pause a sandbox."""
        pass

    @abstractmethod
    async def resume(self, sandbox_id: str) -> None:
        """Resume a sandbox."""
        pass

    @abstractmethod
    async def exec_command(
        self,
        sandbox_id: str,
        command: str,
        cwd: str,
        environment: Dict[str, str],
    ) -> Dict[str, Any]:
        """Execute a command in the sandbox."""
        pass

    @abstractmethod
    async def exec_stream(
        self,
        sandbox_id: str,
        command: str,
        cwd: str,
        environment: Dict[str, str],
    ) -> AsyncIterator[Dict[str, Any]]:
        """Execute a command with streaming output."""
        pass

    @abstractmethod
    async def start_background(
        self,
        sandbox_id: str,
        command: str,
        cwd: str,
        environment: Dict[str, str],
    ) -> Dict[str, Any]:
        """Start a background process."""
        pass

    @abstractmethod
    async def is_process_running(self, sandbox_id: str, pid: int) -> bool:
        """Check if a process is running."""
        pass

    @abstractmethod
    async def kill_process(self, sandbox_id: str, pid: int) -> None:
        """Kill a process."""
        pass

    @abstractmethod
    async def read_file(self, sandbox_id: str, path: str) -> str:
        """Read file as text."""
        pass

    @abstractmethod
    async def read_file_bytes(self, sandbox_id: str, path: str) -> bytes:
        """Read file as bytes."""
        pass

    @abstractmethod
    async def write_file(self, sandbox_id: str, path: str, content: str) -> None:
        """Write text to file."""
        pass

    @abstractmethod
    async def write_file_bytes(
        self, sandbox_id: str, path: str, content: bytes
    ) -> None:
        """Write bytes to file."""
        pass

    @abstractmethod
    async def list_directory(self, sandbox_id: str, path: str) -> List[Dict[str, Any]]:
        """List directory contents."""
        pass

    @abstractmethod
    async def mkdir(self, sandbox_id: str, path: str, recursive: bool) -> None:
        """Create directory."""
        pass

    @abstractmethod
    async def remove(self, sandbox_id: str, path: str) -> None:
        """Remove file."""
        pass

    @abstractmethod
    async def remove_dir(self, sandbox_id: str, path: str) -> None:
        """Remove directory."""
        pass

    @abstractmethod
    async def exists(self, sandbox_id: str, path: str) -> bool:
        """Check if path exists."""
        pass

    @abstractmethod
    async def get_metrics(self, sandbox_id: str) -> SandboxMetrics:
        """Get sandbox metrics."""
        pass

    async def watch_directory(
        self, sandbox_id: str, path: str
    ) -> AsyncIterator[Dict[str, Any]]:
        """Watch directory for changes (optional)."""
        yield {}  # Default no-op


class DockerBackend(SandboxBackend):
    """Docker-based sandbox backend."""

    def __init__(self, runtime_dir: str = "/var/run/agent-sandboxes"):
        self.runtime_dir = Path(runtime_dir)
        self.runtime_dir.mkdir(parents=True, exist_ok=True)

    def _sandbox_dir(self, sandbox_id: str) -> Path:
        return self.runtime_dir / sandbox_id

    def _container_name(self, sandbox_id: str) -> str:
        return f"firework_{sandbox_id}"

    async def create(
        self,
        template: str,
        name: Optional[str],
        environment: Dict[str, str],
        vcpu: int,
        memory_mb: int,
        timeout_seconds: int,
    ) -> SandboxInfo:
        sandbox_id = f"sbx_{uuid.uuid4().hex[:12]}"
        sandbox_dir = self._sandbox_dir(sandbox_id)
        sandbox_dir.mkdir(parents=True, exist_ok=True)

        # Get image from template
        from firework.config import get_config
        config = get_config()
        template_config = config.get_template(template)
        image = template_config.get("image", "ubuntu:22.04")

        # Build docker run command
        container_name = self._container_name(sandbox_id)
        env_args = []
        for k, v in environment.items():
            env_args.extend(["-e", f"{k}={v}"])

        cmd = [
            "docker", "run", "-d",
            "--name", container_name,
            "--cpus", str(vcpu),
            "--memory", f"{memory_mb}m",
            "-v", f"{sandbox_dir}/workspace:/workspace",
            *env_args,
            image,
            "sleep", "infinity",
        ]

        proc = await asyncio.create_subprocess_exec(
            *cmd,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, stderr = await proc.communicate()

        if proc.returncode != 0:
            raise RuntimeError(f"Failed to create container: {stderr.decode()}")

        now = datetime.now(timezone.utc)
        info = SandboxInfo(
            id=sandbox_id,
            root_path=str(sandbox_dir),
            state="running",
            vcpu=vcpu,
            memory_mb=memory_mb,
            created_at=now,
            last_used_at=now,
            template=template,
            name=name,
        )

        # Save metadata
        metadata_path = sandbox_dir / "metadata.json"
        metadata_path.write_text(json.dumps({
            "id": info.id,
            "root_path": info.root_path,
            "state": info.state,
            "vcpu": info.vcpu,
            "memory_mb": info.memory_mb,
            "created_at": info.created_at.isoformat(),
            "last_used_at": info.last_used_at.isoformat(),
            "template": info.template,
            "name": info.name,
            "container_name": container_name,
        }))

        return info

    async def destroy(self, sandbox_id: str) -> None:
        container_name = self._container_name(sandbox_id)

        # Stop and remove container
        proc = await asyncio.create_subprocess_exec(
            "docker", "rm", "-f", container_name,
            stdout=asyncio.subprocess.DEVNULL,
            stderr=asyncio.subprocess.DEVNULL,
        )
        await proc.communicate()

        # Remove sandbox directory
        sandbox_dir = self._sandbox_dir(sandbox_id)
        if sandbox_dir.exists():
            shutil.rmtree(sandbox_dir, ignore_errors=True)

    async def get_info(self, sandbox_id: str) -> Optional[SandboxInfo]:
        sandbox_dir = self._sandbox_dir(sandbox_id)
        metadata_path = sandbox_dir / "metadata.json"

        if not metadata_path.exists():
            return None

        data = json.loads(metadata_path.read_text())
        return SandboxInfo(
            id=data["id"],
            root_path=data["root_path"],
            state=await self.get_state(sandbox_id),
            vcpu=data["vcpu"],
            memory_mb=data["memory_mb"],
            created_at=datetime.fromisoformat(data["created_at"]),
            last_used_at=datetime.fromisoformat(data["last_used_at"]),
            template=data.get("template", "base"),
            name=data.get("name"),
        )

    async def get_state(self, sandbox_id: str) -> str:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "inspect", "-f", "{{.State.Status}}", container_name,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, _ = await proc.communicate()
        if proc.returncode != 0:
            return "stopped"
        status = stdout.decode().strip()
        if status == "running":
            return "running"
        elif status == "paused":
            return "paused"
        return "stopped"

    async def pause(self, sandbox_id: str) -> None:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "pause", container_name,
            stdout=asyncio.subprocess.DEVNULL,
            stderr=asyncio.subprocess.PIPE,
        )
        await proc.communicate()

    async def resume(self, sandbox_id: str) -> None:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "unpause", container_name,
            stdout=asyncio.subprocess.DEVNULL,
            stderr=asyncio.subprocess.PIPE,
        )
        await proc.communicate()

    async def exec_command(
        self,
        sandbox_id: str,
        command: str,
        cwd: str,
        environment: Dict[str, str],
    ) -> Dict[str, Any]:
        container_name = self._container_name(sandbox_id)
        env_args = []
        for k, v in environment.items():
            env_args.extend(["-e", f"{k}={v}"])

        cmd = [
            "docker", "exec",
            "-w", cwd,
            *env_args,
            container_name,
            "sh", "-c", command,
        ]

        proc = await asyncio.create_subprocess_exec(
            *cmd,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, stderr = await proc.communicate()

        return {
            "stdout": stdout.decode(),
            "stderr": stderr.decode(),
            "exit_code": proc.returncode,
        }

    async def exec_stream(
        self,
        sandbox_id: str,
        command: str,
        cwd: str,
        environment: Dict[str, str],
    ) -> AsyncIterator[Dict[str, Any]]:
        container_name = self._container_name(sandbox_id)
        env_args = []
        for k, v in environment.items():
            env_args.extend(["-e", f"{k}={v}"])

        cmd = [
            "docker", "exec",
            "-w", cwd,
            *env_args,
            container_name,
            "sh", "-c", command,
        ]

        proc = await asyncio.create_subprocess_exec(
            *cmd,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )

        async def read_stream(stream, stream_type):
            while True:
                line = await stream.readline()
                if not line:
                    break
                yield {"type": stream_type, "content": line.decode()}

        # Read stdout and stderr concurrently
        async for event in read_stream(proc.stdout, "stdout"):
            yield event

        await proc.wait()
        yield {"type": "exit", "exit_code": proc.returncode}

    async def start_background(
        self,
        sandbox_id: str,
        command: str,
        cwd: str,
        environment: Dict[str, str],
    ) -> Dict[str, Any]:
        container_name = self._container_name(sandbox_id)
        env_args = []
        for k, v in environment.items():
            env_args.extend(["-e", f"{k}={v}"])

        cmd = [
            "docker", "exec", "-d",
            "-w", cwd,
            *env_args,
            container_name,
            "sh", "-c", command,
        ]

        proc = await asyncio.create_subprocess_exec(
            *cmd,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        await proc.communicate()

        return {"pid": None}  # Docker doesn't return PID easily

    async def is_process_running(self, sandbox_id: str, pid: int) -> bool:
        return True  # Simplified for Docker

    async def kill_process(self, sandbox_id: str, pid: int) -> None:
        pass  # Simplified for Docker

    async def read_file(self, sandbox_id: str, path: str) -> str:
        return (await self.read_file_bytes(sandbox_id, path)).decode()

    async def read_file_bytes(self, sandbox_id: str, path: str) -> bytes:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "exec", container_name, "cat", path,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, stderr = await proc.communicate()
        if proc.returncode != 0:
            raise FileNotFoundError(stderr.decode())
        return stdout

    async def write_file(self, sandbox_id: str, path: str, content: str) -> None:
        await self.write_file_bytes(sandbox_id, path, content.encode())

    async def write_file_bytes(
        self, sandbox_id: str, path: str, content: bytes
    ) -> None:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "exec", "-i", container_name, "sh", "-c", f"cat > {path}",
            stdin=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        _, stderr = await proc.communicate(input=content)
        if proc.returncode != 0:
            raise IOError(stderr.decode())

    async def list_directory(self, sandbox_id: str, path: str) -> List[Dict[str, Any]]:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "exec", container_name,
            "ls", "-la", "--time-style=+%Y-%m-%dT%H:%M:%S", path,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, _ = await proc.communicate()

        entries = []
        for line in stdout.decode().strip().split("\n")[1:]:  # Skip total line
            parts = line.split()
            if len(parts) >= 8:
                perms = parts[0]
                size = int(parts[4]) if parts[4].isdigit() else 0
                name = parts[-1]
                file_type = "directory" if perms.startswith("d") else "file"
                entries.append({"name": name, "size": size, "type": file_type})

        return entries

    async def mkdir(self, sandbox_id: str, path: str, recursive: bool) -> None:
        container_name = self._container_name(sandbox_id)
        cmd = ["docker", "exec", container_name, "mkdir"]
        if recursive:
            cmd.append("-p")
        cmd.append(path)

        proc = await asyncio.create_subprocess_exec(
            *cmd,
            stderr=asyncio.subprocess.PIPE,
        )
        await proc.communicate()

    async def remove(self, sandbox_id: str, path: str) -> None:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "exec", container_name, "rm", "-f", path,
            stderr=asyncio.subprocess.PIPE,
        )
        await proc.communicate()

    async def remove_dir(self, sandbox_id: str, path: str) -> None:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "exec", container_name, "rm", "-rf", path,
            stderr=asyncio.subprocess.PIPE,
        )
        await proc.communicate()

    async def exists(self, sandbox_id: str, path: str) -> bool:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "exec", container_name, "test", "-e", path,
            stderr=asyncio.subprocess.DEVNULL,
        )
        await proc.communicate()
        return proc.returncode == 0

    async def get_metrics(self, sandbox_id: str) -> SandboxMetrics:
        container_name = self._container_name(sandbox_id)
        proc = await asyncio.create_subprocess_exec(
            "docker", "stats", "--no-stream", "--format",
            "{{.CPUPerc}},{{.MemUsage}}",
            container_name,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        stdout, _ = await proc.communicate()

        # Parse stats
        parts = stdout.decode().strip().split(",")
        cpu = float(parts[0].rstrip("%")) if parts else 0.0
        mem_str = parts[1].split("/")[0].strip() if len(parts) > 1 else "0MiB"
        mem = float(mem_str.rstrip("MiB").rstrip("GiB")) if mem_str else 0.0

        return SandboxMetrics(
            cpu_percent=cpu,
            memory_mb=mem,
            disk_mb_used=0.0,
            uptime_seconds=0.0,
            process_count=1,
        )


def get_backend(runtime_dir: str = "/var/run/agent-sandboxes") -> SandboxBackend:
    """Get the appropriate backend based on available runtimes."""
    # For now, default to Docker
    return DockerBackend(runtime_dir)
