# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nnext',
 'nnext.conversions',
 'nnext.grpc',
 'nnext.http',
 'nnext.http.api',
 'nnext.http.models',
 'nnext.uploader']

package_data = \
{'': ['*']}

install_requires = \
['betterproto==2.0.0b4',
 'grpcio>=1.46.0,<2.0.0',
 'httpx[http2]>=0.23.0,<0.24.0',
 'loguru>=0.5.3,<0.6.0',
 'nest-asyncio>=1.5.5,<2.0.0',
 'numpy>=1.21,<2.0',
 'pydantic>=1.8,<2.0',
 'tqdm>=4.56.0,<5.0.0',
 'typing-extensions>=4.0.0,<5.0.0']

setup_kwargs = {
    'name': 'nnext',
    'version': '0.0.37',
    'description': 'Client library for the NNext vector search engine',
    'long_description': '\n# <a href="https://nnext.ai/"><img src="https://d135j1zm1liera.cloudfront.net/nnext-logo-wide.png" height="100" alt="Apollo Client"></a>\n\n## About\n\nThis repository houses the source code for pynnext, the python client associated with NNext server.\n\nNNext is a\n* ⚡ blazingly fast\n* 📖 source-available [[Elastic License 2.0]](https://www.elastic.co/licensing/elastic-license)\n* 🔍 nearest-neighbors vector search engine\n\n<a href="https://tiny.one/nnext-slk-comm-gh"><img src="https://img.shields.io/badge/chat-slack-orange.svg?logo=slack&style=flat"></a>\n<a href="https://twitter.com/intent/follow?screen_name=nnextai"><img src="https://img.shields.io/badge/Follow-nnextai-blue.svg?style=flat&logo=twitter"></a>\n\n[Installation](#installation) |  [Quick Start](#quick-start) | [Documentation](#documentation) | [Contributing](#contributing)\n\n## Installation\nYou will need to setup and run an NNext server to utilize this client, or [outsource server management to someone else](https://nnext.ai).\nFor detailed server installation instructions, please see the [nnext repo](https://github.com/nnext-ai/nnext).\n\nTo install the pynnext client, activate a virtual environment, and install via pip:\n\n1. In editable mode from source:\n```zsh\ngit clone https://github.com/nnext-ai/pynnext\ncd pynnext\npip install -e .\n```\n\nor \n\n2. From PyPI:\n\n```zsh\npip install nnext\n```\n\n## Quick Start\n\nHere\'s a quick example showcasing how you can create an index, insert vectors/documents and search among them via NNext.\n\nLet\'s begin by installing the NNext server.\n\n```shell\nNNEXT_PKG=nnext-0.0.12-amd64.deb\nNNEXT_URL=https://trove.nnext.io/downloads\nwget $NNEXT_URL/$NNEXT_PKG\nwget $NNEXT_URL/$NNEXT_PKG.sha512\nshasum -a 512 -c $NNEXT_PKG.sha512\nsudo dpkg -i $NNEXT_PKG\n```\n\nRun nnext\n```shell\nsudo nnext\n```\n\nYou should see something like this:\n```shell\n...\n...\n[2022-04-27 13:02:10.029] [info] 🏁 Started NNext at ▸ 127.0.0.1:6040\n```\n\nInstall the Python client for NNext:\n\n```shell\npip install nnext\n```\n\nFinally, to follow the examples below, install numpy:\n\n1.  From the souce here\n\n```shell\npip install -r requirements.txt\n```\n\nor\n\n2. Via PyPi\n\n```shell\npip install numpy==1.22.3\n```\n\nWe can now initialize the client and create a index:\n\n```python\nimport numpy as np\nimport nnext\nfrom nnext import _and, _eq, _gte, _in\n\n# Create and initialize the vector client.\nnnclient = nnext.Client(\n    nodes=[\n    {\'host\': \'localhost\', \'port\': \'6040\'}\n  ])\n```\n\n\nBroadly speaking, you can create two types of indices\n### 1. Simple indices\n```python\nn_dim = 768\n\n# Create an vector index.\nnnindex = nnclient.index.create(\n    d=n_dim,\n    name=\'test_index\')\n\n# Insert vectors into the index.\nn_vecs = 1000\nvectors = np.random.rand(n_vecs, n_dim)\nnnindex.add(vectors)\n\n# Create a query vector set.\nn_queries = 10\nq_vectors = np.random.rand(n_queries, n_dim)\n\n# Search for the nearest neighbors of the\n# query set among the indexed vectors.\nk = 5\n_idx, _res = nnindex.search(q_vectors, k, return_vector=True)\n\n# The search operation returns a 2d list of the indices of the nearest neighbors\n# for each vector in the query set (i.e. a nested list with shape (n_queries, k)),\n# and optionally the data associated with the neighbor vectors themselves \n# (i.e. a nested list of shape (n_queries, k, n_dim))\nassert len(_idx) == n_queries\nassert len(_idx[0]) == k\nassert len(_res) == n_queries\nassert len(_res[0]) == k\nassert len(_res[0][0]) == n_dim\n```\n\n### 2. Compound indices\n🚧 WIP 🚧.\n\nNot implemented.\n\nNNext is capable of storing additional metadata related to your vectors in a rich format. In this example we will use the\n[movie plots dataset from Kaggle](https://www.kaggle.com/datasets/jrobischon/wikipedia-movie-plots).\n```python\nnnindex = client.index.create({\n  "name": "movies",\n  "schema": {\n      "id" : "string", #⬅ inferred primary key\n      "title" : "string",\n      "released_year" : "int32",\n      "genre" :  "float",\n      "wikipage" : "string",\n      "plot" : "string",\n      "rating" :  "float"\n  },\n  "index_type": "approximated", #⬅ indexes assumed to be approximated by default.\n  "dims": n_dim\n})\n```\n\n\nNow, let\'s add a vector to the collection we just created:\n\n```python\nvector = {\n "id": "124",\n "company_name": "Stark Industries",\n "num_employees": 5215,\n "country": "USA",\n}\n\nnnindex.documents.create(document)\n```\n\nFinally, let\'s search for the document we just indexed:\n\n```python\nq_filter = {\n    _and: [\n        { "Release Year": { _gte: 2015 } },\n        { "Genre": { _eq: "comedy" } },\n        { "actors": { _in: ["Russell Crowe"] } }\n    ]\n}\n\nclient.collections[\'companies\'].documents.search(search_parameters)\n```\n\n## Documentation\n\nAll NNext Server and Client documentation, including pynnext integration articles and helpful recipes, can be found at:\n<br/>\n\n🚧 WIP 🚧<br>\n[https://nnext.ai/docs/](https://nnext.ai/docs)\n\n## FAQs\n\n<details><summary>How does this differ from Faiss, ScaNN and Annoy?</summary>\n<p>\nFirst of all, NNext uses Faiss under the hood. All of these libraries have python\npackages installable via PIP or Conda, and those are very easy to use, from install to the API. However, while\nthey allow you to quickly get started, they don\'t allow for persistence, index growth or high availability. If your\napplication goes down for whatever reason, so do your search indices and data.\n</p>\n</details>\n\n<details><summary>How does this differ from Milvus?</summary>\n<p>\nMilvus is a large piece of software, that takes a non-trivial amount of effort to setup, administer, scale and fine-tune.\nIt offers you a few thousand configuration parameters that may need to be tuned to get to your ideal configuration. As a result, it\'s better suited for large teams\nwho have the bandwidth to get it production-ready, and regularly monitor it and scale it, especially when they have a need to store\nbillions of documents and petabytes of data (eg: logs).\n\nNNext is built specifically for decreasing the "time to market" for a delightful nearest-neighbor search experience. It \nis a light-weight yet powerful & scaleable alternative that focuses on Developer Happiness and Experience with a \nclean well-documented API, clear semantics and smart defaults so it just works well out-of-the-box, without you having to turn many knobs.\n\nSee a side-by-side feature comparison [here](https://typesense.org/typesense-vs-algolia-vs-elasticsearch-vs-meilisearch/).\n</p>\n</details>\n\n<details><summary>How does this differ from other fully managed solutions like Pinecone?</summary>\n<p>\nIn brief - **no vendor lock-in**. Tired of using NNext cloud? Pack up your vectors and go. Obviously we don\'t want you \nto go, but if you have to, NNext Cloud allows you to download a compressed zip file containing the latest backup of \nyour vectors to your machine. These vectors can then be used with another installation of NNext on premise or on \nanother cloud provider.\n\nPinecone is a proprietary, hosted, nearest-neighbour search-as-a-service product that works well, when cost is not an \nissue. However, fast growing applications will quickly run into search & indexing limits, accompanied by expensive plan\nupgrades as they scale.\n\nNNext on the other hand is an open-source product that you can run on your own infrastructure or\nuse our managed SaaS offering - [NNext Cloud](https://app.nnext.ai).\nThe open source version is free to use (besides of course your own infra costs).\nWith NNext Cloud we do not charge by records or search operations. Instead, you get a dedicated cluster\nand you can throw as much data and traffic at it as it can handle. You only pay a fixed hourly cost & bandwidth charges\nfor it, depending on the configuration your choose, similar to most modern cloud platforms.\n\nFrom a product perspective, NNext is closer in spirit to Jina.ai than Pinecone.\n\nSee a side-by-side feature comparison [here](https://nnext.ai/product-matrix?source=gitreadme).\n</p>\n</details>\n\n<details><summary>Why the Elastic License 2.0?</summary>\n<p>\nNNext Server is **source available**, **server software** and we expect users to typically run it as a separate daemon, \nand not integrate it \nwith their own code. Elastic Licence 2.0 (EL2) covers and allows for this use case **generously**. We aim to set the\nminimum limitations necessary to strike a fair balance between freedom to use, share and change the software, and \npreventing actions that will harm the community.\n\nIf you have specifics that prevent you from using NNext due to a licensing issue, we\'re happy to explore this topic \nfurther with you. Please reach out to us legal@nnext.ai.\n</p>\n</details>\n<details><summary>I heard Elasticsearch and OpenSearch were planning on implementing ANN Search?</summary>\n<p>\nFundamentally, Elasticsearch and it\'s variants, run on the JVM, which by itself can be quite an effort to tune to run \noptimally. NNext, on the other hand, is a single light-weight self-contained native binary, so it\'s simple to setup and\noperate. Furthermore, ANN search on Elasticseach runs as a secondary process, a sidecar, which is not natively \nsupported by the main indexing engine.\n</p>\n</details>\n\n## Who is NNext?\n\n[NNext](https://nnext.io/) builds open-source ML-Ops software to help make development and deployment of machine \nlearning applications painless.\n\n## Contributing\n\n### Introduction\nFirst off, 🙏🏾 thank you for considering contributing to nnext. We value community contributions!\n\n### How can you help?\n\nYou may already know what you want to contribute -- a fix for a bug you encountered, or a new feature your team wants to use.\n\nIf you don\'t know what to contribute, keep an open mind! Here\'s some examples of helpful contributions that mean\nless work for you\n* Improving documentation\n* bug triaging\n* writing tutorials\n\nCheckout the [guide to contributing](#) to learn more.\n\n',
    'author': 'NNext Team',
    'author_email': 'pynnext@nnext.ai',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/nnextdb/pynnext',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
