"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    CodeCommitTrigger["NONE"] = "None";
    /**
     * CodePipeline will poll the repository to detect changes.
     */
    CodeCommitTrigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * This is the default method of detecting changes.
     */
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 *
 * If the CodeCommit repository is in a different account, you must use
 * `CodeCommitTrigger.EVENTS` to trigger the pipeline.
 *
 * (That is because the Pipeline structure normally only has a `RepositoryName`
 * field, and that is not enough for the pipeline to locate the repository's
 * source account. However, if the pipeline is triggered via an EventBridge
 * event, the event itself has the full repository ARN in there, allowing the
 * pipeline to locate the repository).
 */
class CodeCommitSourceAction extends action_1.Action {
    constructor(props) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_actions_CodeCommitSourceActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const branch = (_b = props.branch) !== null && _b !== void 0 ? _b : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        if (props.codeBuildCloneOutput === true) {
            props.output.setMetadata(CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY, props.repository.repositoryArn);
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline, {
                    eventRole: this.props.eventRole,
                }),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // when this action is cross-account,
        // the Role needs the s3:PutObjectAcl permission for some not yet fully understood reason
        if (core_1.Token.compareStrings(this.props.repository.env.account, core_1.Stack.of(stage.pipeline).account) === core_1.TokenComparison.DIFFERENT) {
            options.bucket.grantPutAcl(options.role);
        }
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
                ...(this.props.codeBuildCloneOutput === true ? ['codecommit:GetRepository'] : []),
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
                OutputArtifactFormat: this.props.codeBuildCloneOutput === true
                    ? 'CODEBUILD_CLONE_REF'
                    : undefined,
            },
        };
    }
    generateEventId(stage) {
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node);
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : `-${this.branch}-`;
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
_a = JSII_RTTI_SYMBOL_1;
CodeCommitSourceAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CodeCommitSourceAction", version: "1.156.1" };
/**
 * The name of the property that holds the ARN of the CodeCommit Repository
 * inside of the CodePipeline Artifact's metadata.
 *
 * @internal
 */
CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY = 'CodeCommitCloneRepositoryArn';
//# sourceMappingURL=data:application/json;base64,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