"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const appscaling = require("../lib");
const util_1 = require("./util");
module.exports = {
    'test scalable target creation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new appscaling.ScalableTarget(stack, 'Target', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            scalableDimension: 'test:TestCount',
            resourceId: 'test:this/test',
            minCapacity: 1,
            maxCapacity: 20,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ServiceNamespace: 'dynamodb',
            ScalableDimension: 'test:TestCount',
            ResourceId: 'test:this/test',
            MinCapacity: 1,
            MaxCapacity: 20,
        }));
        test.done();
    },
    'validation does not fail when using Tokens'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new appscaling.ScalableTarget(stack, 'Target', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            scalableDimension: 'test:TestCount',
            resourceId: 'test:this/test',
            minCapacity: cdk.Lazy.numberValue({ produce: () => 10 }),
            maxCapacity: cdk.Lazy.numberValue({ produce: () => 1 }),
        });
        // THEN: no exception
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ServiceNamespace: 'dynamodb',
            ScalableDimension: 'test:TestCount',
            ResourceId: 'test:this/test',
            MinCapacity: 10,
            MaxCapacity: 1,
        }));
        test.done();
    },
    'add scheduled scaling'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const target = util_1.createScalableTarget(stack);
        // WHEN
        target.scaleOnSchedule('ScaleUp', {
            schedule: appscaling.Schedule.rate(cdk.Duration.minutes(1)),
            maxCapacity: 50,
            minCapacity: 1,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ScheduledActions: [
                {
                    ScalableTargetAction: {
                        MaxCapacity: 50,
                        MinCapacity: 1
                    },
                    Schedule: 'rate(1 minute)',
                    ScheduledActionName: 'ScaleUp'
                }
            ]
        }));
        test.done();
    },
    'step scaling on MathExpression'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const target = util_1.createScalableTarget(stack);
        // WHEN
        target.scaleOnMetric('Metric', {
            metric: new cloudwatch.MathExpression({
                expression: 'a',
                usingMetrics: {
                    a: new cloudwatch.Metric({
                        namespace: 'Test',
                        metricName: 'Metric',
                    })
                },
            }),
            adjustmentType: appscaling.AdjustmentType.CHANGE_IN_CAPACITY,
            scalingSteps: [
                { change: -1, lower: 0, upper: 49 },
                { change: 0, lower: 50, upper: 99 },
                { change: 1, lower: 100 }
            ]
        });
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            Period: 60
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'LessThanOrEqualToThreshold',
            EvaluationPeriods: 1,
            Metrics: [
                {
                    Expression: 'a',
                    Id: 'expr_1'
                },
                {
                    Id: 'a',
                    MetricStat: {
                        Metric: {
                            MetricName: 'Metric',
                            Namespace: 'Test'
                        },
                        Period: 300,
                        Stat: 'Average'
                    },
                    ReturnData: false
                }
            ],
            Threshold: 49
        }));
        test.done();
    },
    'test service namespace enum'(test) {
        test.equals(appscaling.ServiceNamespace.APPSTREAM, 'appstream');
        test.equals(appscaling.ServiceNamespace.COMPREHEND, 'comprehend');
        test.equals(appscaling.ServiceNamespace.CUSTOM_RESOURCE, 'custom-resource');
        test.equals(appscaling.ServiceNamespace.DYNAMODB, 'dynamodb');
        test.equals(appscaling.ServiceNamespace.EC2, 'ec2');
        test.equals(appscaling.ServiceNamespace.ECS, 'ecs');
        test.equals(appscaling.ServiceNamespace.ELASTIC_MAP_REDUCE, 'elasticmapreduce');
        test.equals(appscaling.ServiceNamespace.LAMBDA, 'lambda');
        test.equals(appscaling.ServiceNamespace.RDS, 'rds');
        test.equals(appscaling.ServiceNamespace.SAGEMAKER, 'sagemaker');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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