import datetime as dt

from czml3 import CZML_VERSION, Document, Packet
from czml3.enums import (
    HorizontalOrigins,
    InterpolationAlgorithms,
    LabelStyles,
    ReferenceFrames,
    VerticalOrigins,
)
from czml3.properties import (
    Billboard,
    Clock,
    Color,
    Label,
    Path,
    PolylineMaterial,
    Position,
    SolidColorMaterial,
)
from czml3.types import IntervalValue, TimeInterval, TimeIntervalCollection

accesses_id = "9927edc4-e87a-4e1f-9b8b-0bfb3b05b227"
start = dt.datetime(2012, 3, 15, 10, tzinfo=dt.timezone.utc)
end = dt.datetime(2012, 3, 16, 10, tzinfo=dt.timezone.utc)

simple = Document(
    packets=[
        Packet(
            id="document",
            name="simple",
            version=CZML_VERSION,
            clock=IntervalValue(
                start=start, end=end, value=Clock(currentTime=start, multiplier=60)
            ),
        ),
        Packet(id=accesses_id, name="Accesses", description="List of Accesses"),
        Packet(
            id="Satellite/Geoeye1-to-Satellite/ISS",
            name="Geoeye1 to ISS",
            parent=accesses_id,
            availability=TimeIntervalCollection(
                values=[
                    TimeInterval(
                        start="2012-03-15T10:16:06.97400000000198Z",
                        end="2012-03-15T10:33:59.3549999999959Z",
                    ),
                    TimeInterval(
                        start="2012-03-15T11:04:09.73799999999756Z",
                        end="2012-03-15T11:21:04.51900000000023Z",
                    ),
                    TimeInterval(
                        start="2012-03-15T11:52:06.94400000000314Z",
                        end="2012-03-15T12:08:18.8840000000055Z",
                    ),
                    TimeInterval(
                        start="2012-03-15T12:40:57.2069999999949Z",
                        end="2012-03-15T12:54:39.301999999996Z",
                    ),
                    TimeInterval(
                        start="2012-03-15T13:29:44.5040000000008Z",
                        end="2012-03-15T13:41:05.96899999999732Z",
                    ),
                    TimeInterval(
                        start="2012-03-15T14:20:16.8450000000012Z",
                        end="2012-03-15T14:25:48.0559999999969Z",
                    ),
                    TimeInterval(
                        start="2012-03-16T07:01:44.4309999999823Z",
                        end="2012-03-16T07:06:19.6309999999939Z",
                    ),
                    TimeInterval(
                        start="2012-03-16T07:46:00.457999999984168Z",
                        end="2012-03-16T07:57:20.8470000000088Z",
                    ),
                    TimeInterval(
                        start="2012-03-16T08:32:14.5289999999804Z",
                        end="2012-03-16T08:46:17.0109999999986Z",
                    ),
                    TimeInterval(
                        start="2012-03-16T09:18:28.4590000000026Z",
                        end="2012-03-16T09:35:16.6410000000033Z",
                    ),
                ]
            ),
        ),
        Packet(
            id="Facility/AGI-to-Satellite/ISS", name="AGI to ISS", parent=accesses_id
        ),
        Packet(
            id="Facility/AGI-to-Satellite/Geoeye1/Sensor/Sensor",
            name="AGI to Sensor",
            parent=accesses_id,
            description="<h2>No accesses</h2>",
        ),
        Packet(
            id="AreaTarget/Pennsylvania",
            name="Pennsylvania",
            label=Label(
                horizontalOrigin=HorizontalOrigins.LEFT,
                show=True,
                font="11pt Lucida Console",
                style=LabelStyles.FILL_AND_OUTLINE,
                outlineWidth=2,
                text="Pennsylvania",
                verticalOrigin=VerticalOrigins.CENTER,
                fillColor=Color(rgba=[255, 0, 0]),
                outlineColor=Color(rgba=[0, 0, 0]),
            ),
            position=Position(
                cartesian=[1152255.80150063, -4694317.951340558, 4147335.9067563135]
            ),
        ),
        Packet(
            id="Facility/AGI",
            name="AGI",
            availability=TimeInterval(start=start, end=end),
            billboard=Billboard(
                horizontalOrigin=HorizontalOrigins.CENTER,
                image=(
                    "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/"
                    "9hAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAc"
                    "dvqGQAAACvSURBVDhPrZDRDcMgDAU9GqN0lIzijw6SUbJJygUeNQgSqepJTyHG91"
                    "LVVpwDdfxM3T9TSl1EXZvDwii471fivK73cBFFQNTT/d2KoGpfGOpSIkhUpgUMxq"
                    "9DFEsWv4IXhlyCnhBFnZcFEEuYqbiUlNwWgMTdrZ3JbQFoEVG53rd8ztG9aPJMnB"
                    "UQf/VFraBJeWnLS0RfjbKyLJA8FkT5seDYS1Qwyv8t0B/5C2ZmH2/eTGNNBgMmAA"
                    "AAAElFTkSuQmCC"
                ),
                scale=1.5,
                show=True,
                verticalOrigin=VerticalOrigins.CENTER,
            ),
            label=Label(
                horizontalOrigin=HorizontalOrigins.LEFT,
                outlineWidth=2,
                show=True,
                font="11pt Lucida Console",
                style=LabelStyles.FILL_AND_OUTLINE,
                text="AGI",
                verticalOrigin=VerticalOrigins.CENTER,
                fillColor=Color(rgba=[0, 255, 255]),
                outlineColor=Color(rgba=[0, 0, 0]),
            ),
            position=Position(
                cartesian=[1216469.9357990976, -4736121.71856379, 4081386.8856866374]
            ),
        ),
        Packet(
            id="Satellite/Geoeye1",
            name="Geoeye1",
            availability=TimeInterval(start=start, end=end),
            billboard=Billboard(
                horizontalOrigin=HorizontalOrigins.CENTER,
                image=(
                    "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9"
                    "hAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdv"
                    "qGQAAADJSURBVDhPnZHRDcMgEEMZjVEYpaNklIzSEfLfD4qNnXAJSFWfhO7w2Zc0T"
                    "f9QG2rXrEzSUeZLOGm47WoH95x3Hl3jEgilvDgsOQUTqsNl68ezEwn1vae6lceSEE"
                    "YvvWNT/Rxc4CXQNGadho1NXoJ+9iaqc2xi2xbt23PJCDIB6TQjOC6Bho/sDy3fBQT"
                    "8PrVhibU7yBFcEPaRxOoeTwbwByCOYf9VGp1BYI1BA+EeHhmfzKbBoJEQwn1yzUZt"
                    "yspIQUha85MpkNIXB7GizqDEECsAAAAASUVORK5CYII="
                ),
                scale=1.5,
                show=True,
                verticalOrigin=VerticalOrigins.CENTER,
            ),
            label=Label(
                horizontalOrigin=HorizontalOrigins.LEFT,
                outlineWidth=2,
                show=True,
                font="11pt Lucida Console",
                style=LabelStyles.FILL_AND_OUTLINE,
                text="Geoeye 1",
                verticalOrigin=VerticalOrigins.CENTER,
                fillColor=Color(rgba=[0, 255, 0]),
                outlineColor=Color(rgba=[0, 0, 0]),
            ),
            path=Path(
                show=TimeIntervalCollection(
                    values=[IntervalValue(start=start, end=end, value=True)]
                ),
                width=1,
                resolution=120,
                material=PolylineMaterial(
                    solidColor=SolidColorMaterial(color=Color(rgba=[0, 255, 0]))
                ),
            ),
            position=Position(
                interpolationAlgorithm=InterpolationAlgorithms.LAGRANGE,
                interpolationDegree=5,
                referenceFrame=ReferenceFrames.INERTIAL,
                epoch=start,
                cartesian=[
                    0,
                    4650397.56551457,
                    -3390535.52275848,
                    -4087729.48877329,
                    300,
                    3169722.12564676,
                    -2787480.80604407,
                    -5661647.74541255,
                    600,
                    1369743.14695017,
                    -1903662.23809705,
                    -6663952.07552171,
                    900,
                    -567881.181741598,
                    -828602.646229013,
                    -6995188.66804375,
                    1200,
                    -2448582.60230996,
                    329568.153528487,
                    -6623075.06683579,
                    1500,
                    -4083754.13823344,
                    1454683.10708859,
                    -5585143.37246992,
                    1800,
                    -5308969.50307564,
                    2433751.75579502,
                    -3985152.1306503,
                    2100,
                    -6000251.41586053,
                    3168038.65147806,
                    -1983402.41619314,
                    2400,
                    -6086888.59948749,
                    3583071.0890244,
                    218668.596985674,
                    2700,
                    -5559191.47781264,
                    3636465.81402216,
                    2398566.73167249,
                    3e3,
                    -4469920.08845903,
                    3322511.24085064,
                    4335511.55404767,
                    3300,
                    -2928982.78408199,
                    2672830.22757979,
                    5833216.31169719,
                    3600,
                    -1092083.75801192,
                    1753068.15442214,
                    6739938.26134568,
                    3900,
                    855172.482158128,
                    656135.125613598,
                    6963670.20631676,
                    4200,
                    2715900.15544456,
                    -507141.10370753,
                    6481233.35568881,
                    4500,
                    4301545.02457765,
                    -1619032.41301719,
                    5340730.82941359,
                    4800,
                    5450887.61456272,
                    -2566695.71952053,
                    3657168.90089312,
                    5100,
                    6046743.71636896,
                    -3253730.76621873,
                    1601308.71661883,
                    5400,
                    6028480.33195964,
                    -3610312.82817681,
                    -617627.593798261,
                    5700,
                    5398572.81294241,
                    -3600605.84777646,
                    -2773801.46073176,
                    6e3,
                    4222131.44045066,
                    -3226403.44705845,
                    -4648544.33370408,
                    6300,
                    2619506.22498348,
                    -2526553.4640733,
                    -6052998.81756411,
                    6600,
                    753243.717795414,
                    -1572460.33494755,
                    -6846895.76189552,
                    6900,
                    -1188681.40488625,
                    -460501.299931452,
                    -6951744.44608042,
                    7200,
                    -3011438.3543463,
                    697635.55422786,
                    -6357762.60297625,
                    7500,
                    -4532151.11938648,
                    1785788.3012355,
                    -5124499.63066102,
                    7800,
                    -5597650.73991331,
                    2694559.67974802,
                    -3375267.6308537,
                    8100,
                    -6099819.6602219,
                    3332144.87819992,
                    -1285551.25441976,
                    8400,
                    -5986960.96848025,
                    3633749.04814585,
                    934060.845169727,
                    8700,
                    -5269630.81665556,
                    3568466.92847822,
                    3059081.28082907,
                    9e3,
                    -4019895.75365332,
                    3142650.51265407,
                    4874208.829971,
                    9300,
                    -2363959.24834243,
                    2399275.2584582,
                    6195511.03850927,
                    9600,
                    -469149.870760219,
                    1413453.85724258,
                    6889101.23661198,
                    9900,
                    1473060.67181718,
                    284763.40426175,
                    6884461.2201842,
                    10200,
                    3266189.40343301,
                    -872716.655583284,
                    6181432.05126297,
                    10500,
                    4728347.28355441,
                    -1941746.59348253,
                    4850477.39746136,
                    10800,
                    5710710.2309505,
                    -2813734.86048061,
                    3026098.49087128,
                    11100,
                    6113095.13961986,
                    -3399958.81868676,
                    893596.861584693,
                    11400,
                    5894739.57129755,
                    -3640946.46595729,
                    -1329913.85220463,
                    11700,
                    5078666.83994692,
                    -3512760.54879695,
                    -3418301.78745728,
                    1.2e4,
                    3748910.95111879,
                    -3029296.38333698,
                    -5160170.44648529,
                    12300,
                    2041115.54074956,
                    -2240388.77388782,
                    -6380510.87024598,
                    12600,
                    128062.841689814,
                    -1226230.06015728,
                    -6957810.72494893,
                    12900,
                    -1797880.02370018,
                    -89029.4500196064,
                    -6835259.47179512,
                    13200,
                    -3543573.22088694,
                    1057095.05449295,
                    -6025636.92847326,
                    13500,
                    -4933723.66130957,
                    2097142.52871785,
                    -4609929.79883268,
                    13800,
                    -5828032.08592363,
                    2926410.8376128,
                    -2729797.16692992,
                    14100,
                    -6135448.82449735,
                    3460972.29119284,
                    -574126.701345544,
                    14400,
                    -5823927.7453399,
                    3646385.46288723,
                    1639558.19879141,
                    14700,
                    -4924221.58519395,
                    3463528.69521585,
                    3687192.93154295,
                    1.5e4,
                    -3526959.31596672,
                    2930706.251085,
                    5361276.70706965,
                    15300,
                    -1773303.2569981,
                    2101739.07409384,
                    6492183.23224769,
                    15600,
                    159547.662987404,
                    1060379.23480408,
                    6965271.78475013,
                    15900,
                    2076237.39607934,
                    -88180.8729786283,
                    6732242.01695268,
                    16200,
                    3782725.81635179,
                    -1227790.39374922,
                    5815983.99990949,
                    16500,
                    5105722.20548866,
                    -2242913.57649818,
                    4308638.06403535,
                    16800,
                    5910441.07421132,
                    -3030353.86516332,
                    2362820.08963428,
                    17100,
                    6114851.51773991,
                    -3510008.83668448,
                    176385.679251524,
                    17400,
                    5698545.28993677,
                    -3633371.19534201,
                    -2028049.09959272,
                    17700,
                    4704828.76468794,
                    -3388594.10597537,
                    -4026549.32730149,
                    1.8e4,
                    3235698.38668044,
                    -2801429.85157001,
                    -5617256.78167319,
                    18300,
                    1440597.87544465,
                    -1932078.39947018,
                    -6640795.78057306,
                    18600,
                    -499271.406753861,
                    -868624.163763964,
                    -6995574.53823806,
                    18900,
                    -2389099.93056153,
                    281955.155073057,
                    -6646967.28761496,
                    19200,
                    -4039374.65314639,
                    1404254.69599011,
                    -5630169.87285918,
                    19500,
                    -5284177.74745136,
                    2385580.5113947,
                    -4046822.21760319,
                    19800,
                    -5997582.07668494,
                    3126987.78863274,
                    -2055530.11530625,
                    20100,
                    -6106647.9295009,
                    3553296.55362796,
                    143356.668050338,
                    20400,
                    -5599404.46957774,
                    3620983.69073271,
                    2327690.75353029,
                    20700,
                    -4526523.6344331,
                    3322884.75009569,
                    4276248.57556407,
                    2.1e4,
                    -2996245.95968125,
                    2689013.19249613,
                    5791558.99005475,
                    21300,
                    -1163200.2741721,
                    1783416.84670725,
                    6720089.01115394,
                    21600,
                    787392.993898573,
                    697578.526522602,
                    6967627.88152446,
                    21900,
                    2658313.93062137,
                    -458794.910147627,
                    6508599.14858285,
                    22200,
                    4259990.67255703,
                    -1568681.0047992,
                    5388741.89641135,
                    22500,
                    5429593.2831701,
                    -2519450.97921993,
                    3720960.53337583,
                    22800,
                    6047882.84940679,
                    -3214395.00924545,
                    1674395.1431695,
                    23100,
                    6051933.55069228,
                    -3582878.93041997,
                    -542688.518363143,
                    23400,
                    5441937.00725033,
                    -3587843.0497094,
                    -2704629.51345146,
                    23700,
                    4280981.3355276,
                    -3229580.32104236,
                    -4592141.67304755,
                    2.4e4,
                    2687870.63239537,
                    -2545325.71682756,
                    -6015038.82305718,
                    24300,
                    824227.476165237,
                    -1604924.09680093,
                    -6831172.62144631,
                    24600,
                    -1122206.7533029,
                    -503393.526280383,
                    -6959823.84518174,
                    24900,
                    -2956141.14030617,
                    648615.184081045,
                    -6388844.91324311,
                    25200,
                    -4493594.58923188,
                    1735560.37697432,
                    -5175495.01335972,
                    25500,
                    -5579742.15764727,
                    2648182.45074863,
                    -3441081.02701185,
                    25800,
                    -6104403.26233424,
                    3294305.38566715,
                    -1359568.96084645,
                    26100,
                    -6013607.17885908,
                    3608279.93904281,
                    859312.893815001,
                    26400,
                    -5315661.64405407,
                    3557946.90172158,
                    2991170.32920959,
                    26700,
                    -4080654.87799535,
                    3148138.47866591,
                    4820009.9588535,
                    2.7e4,
                    -2433295.14169919,
                    2420207.41265824,
                    6160500.86869024,
                    27300,
                    -540048.709695602,
                    1447707.79842875,
                    6876809.17070199,
                    27600,
                    1407767.0671378,
                    328874.327204994,
                    6896124.80284684,
                    27900,
                    3213109.50329342,
                    -823211.994312959,
                    6215875.70465174,
                    28200,
                    4692868.59118734,
                    -1891866.70296935,
                    4904219.94344902,
                    28500,
                    5696446.91317596,
                    -2768546.70023811,
                    3093687.88562859,
                    28800,
                    6121503.42645198,
                    -3364055.24800511,
                    968155.930008613,
                    29100,
                    5924955.48270504,
                    -3617967.44053651,
                    -1255976.3070331,
                    29400,
                    5127597.6892057,
                    -3505018.25529556,
                    -3352495.11813025,
                    29700,
                    3811571.94317904,
                    -3037549.25029656,
                    -5109143.31304328,
                    3e4,
                    2111161.49948368,
                    -2263783.11174549,
                    -6349384.85929529,
                    30300,
                    198440.814370976,
                    -1262402.66740219,
                    -6949691.77194863,
                    30600,
                    -1734232.20942493,
                    -134349.362344097,
                    -6850956.53988907,
                    30900,
                    -3493043.34940257,
                    1007171.75248993,
                    -6063593.6302678,
                    31200,
                    -4901404.59339724,
                    2047631.5553316,
                    -4666367.44252873,
                    31500,
                    -5817210.59449393,
                    2882304.14537745,
                    -2799066.43911321,
                    31800,
                    -6147255.41846079,
                    3426731.53070299,
                    -649254.5431675,
                    32100,
                    -5857198.0801714,
                    3625479.50040521,
                    1566166.40158537,
                    32400,
                    -4975601.20019172,
                    3458072.25151343,
                    3622969.15612808,
                    32700,
                    -3591247.48488601,
                    2941244.1531075,
                    5312719.83645661,
                    3.3e4,
                    -1843992.41265467,
                    2127197.21558911,
                    6464196.01056295,
                    33300,
                    89606.2868318625,
                    1098179.28301232,
                    6960674.00038747,
                    33600,
                    2014116.216064,
                    -41861.9173412649,
                    6751496.49323013,
                    33900,
                    3734716.93464761,
                    -1177641.55889657,
                    5857148.74823608,
                    34200,
                    5076704.10409687,
                    -2194022.38273781,
                    4367547.68932021,
                    34500,
                    5903372.88410761,
                    -2987688.92982215,
                    2433491.61675847,
                    34800,
                    6130453.95083298,
                    -3477904.78368583,
                    251624.603178938,
                    35100,
                    5735217.75213746,
                    -3615077.15768101,
                    -1955899.72625122,
                    35400,
                    4758822.55630629,
                    -3385942.79917116,
                    -3964806.69356341,
                    35700,
                    3301523.19211142,
                    -2814664.43976308,
                    -5572146.00475715,
                    3.6e4,
                    1511601.23664753,
                    -1959847.10663623,
                    -6616835.17806668,
                    36300,
                    -430228.574173179,
                    -908129.800399028,
                    -6995151.89199021,
                    36600,
                    -2328943.09312278,
                    234674.826841814,
                    -6670127.01821151,
                    36900,
                    -3994146.68251508,
                    1353941.93189576,
                    -5674612.94646658,
                    37200,
                    -5258447.49803637,
                    2337295.48282856,
                    -4108116.98999385,
                    37500,
                    -5993978.3578334,
                    3085604.46208182,
                    -2127529.51573296,
                    37792.0,
                    -6130101.19290489,
                    3515932.23049969,
                    9264.18885075031,
                    37792.10937600001,
                    -6130384.46590958,
                    3515584.20899647,
                    6695.95297313175,
                    37800,
                    -6125874.41162028,
                    3522668.06783492,
                    65343.2925253213,
                    38100,
                    -5639991.96555162,
                    3604984.9649137,
                    2253983.5554605,
                    38400,
                    -4584395.94143214,
                    3323147.09203951,
                    4214321.78311218,
                    38700,
                    -3065568.13527098,
                    2705519.4701818,
                    5747698.43677366,
                    3.9e4,
                    -1236975.61319386,
                    1814506.56685453,
                    6698754.90266186,
                    39300,
                    716616.358619228,
                    740116.661060127,
                    6971010.28510801,
                    39600,
                    2597699.24211401,
                    -409106.618477249,
                    6536397.61888921,
                    39900,
                    4215695.47146154,
                    -1516877.62428874,
                    5438182.33102578,
                    40200,
                    5406138.82164494,
                    -2470796.73853129,
                    3787053.90015348,
                    40500,
                    6047673.30554989,
                    -3173840.28074124,
                    1750430.41209045,
                    40800,
                    6074988.85892697,
                    -3554541.56526844,
                    -464452.515495436,
                    41100,
                    5485885.96089374,
                    -3574579.22997436,
                    -2632147.29446065,
                    41400,
                    4341322.9852588,
                    -3232699.06081148,
                    -4532744.1278815,
                    41700,
                    2758464.91140936,
                    -2564475.49099545,
                    -5974684.5730493,
                    4.2e4,
                    897941.378998967,
                    -1638151.03207697,
                    -6813870.18895483,
                    42300,
                    -1052780.29114919,
                    -547350.31501512,
                    -6967266.98101188,
                    42600,
                    -2897962.38786145,
                    598337.730126023,
                    -6420270.92575253,
                    42900,
                    -4452507.67257208,
                    1684006.98810485,
                    -5227762.21729297,
                    43200,
                    -5559902.81446472,
                    2600541.92981829,
                    -3508960.37408922,
                    43500,
                    -6107852.50578303,
                    3255392.58900438,
                    -1436240.414533,
                    43800,
                    -6040042.84057261,
                    3582042.33222067,
                    781588.359021092,
                    44100,
                    -5362448.81628763,
                    3547053.6267202,
                    2920264.28926642,
                    44400,
                    -4143086.34484583,
                    3153703.20118722,
                    4763115.49884062,
                    44700,
                    -2505073.20990168,
                    2441676.25611596,
                    6123395.1710605,
                    4.5e4,
                    -613927.427842828,
                    1482918.7438982,
                    6863271.36754096,
                    45300,
                    1339253.51926787,
                    374274.722489247,
                    6907560.17367724,
                    45600,
                    3156902.43368155,
                    -772213.536520633,
                    6251170.31002552,
                    45900,
                    4654686.28493925,
                    -1840443.09381273,
                    4959835.53773277,
                    46200,
                    5680192.5911522,
                    -2721927.0144689,
                    3163998.96049871,
                    46500,
                    6128844.51276722,
                    -3326981.82199689,
                    1046014.11764365,
                    46800,
                    5955135.19089971,
                    -3594198.42002226,
                    -1178497.88600712,
                    47100,
                    5177515.37761796,
                    -3496939.23059209,
                    -3283263.70343719,
                    47400,
                    3876125.08406537,
                    -3045941.10299702,
                    -5055144.34831043,
                    47700,
                    2183796.65985995,
                    -2287764.94024368,
                    -6316019.07390942,
                    4.8e4,
                    271833.87521692,
                    -1299541.32697329,
                    -6940261.69174425,
                    48300,
                    -1667447.4521302,
                    -180911.112150818,
                    -6866372.68166609,
                    48600,
                    -3439563.26008233,
                    955855.877452593,
                    -6102303.72319006,
                    48900,
                    -4866610.13492781,
                    1996714.47420741,
                    -4724503.76740207,
                    49200,
                    -5804634.05552651,
                    2836917.32537247,
                    -2870806.06589895,
                    49500,
                    -6158210.22317347,
                    3391468.10028257,
                    -727378.529454526,
                    49800,
                    -5890622.43327254,
                    3603920.37185301,
                    1489553.62298514,
                    50100,
                    -5028149.07573745,
                    3452412.7854813,
                    3555632.13063885,
                    50400,
                    -3657625.67732601,
                    2952067.47043493,
                    5261491.80318722,
                    50700,
                    -1917502.9861475,
                    2153417.50812297,
                    6434281.66704866,
                    5.1e4,
                    16385.9030227461,
                    1137154.22735658,
                    6955128.03809566,
                    51300,
                    1948590.02716246,
                    5932.68692494816,
                    6770919.32140635,
                    51600,
                    3683531.56602696,
                    -1125865.3101215,
                    5899617.58513144,
                    51900,
                    5045075.05049747,
                    -2143521.20647514,
                    4428790.31172427,
                    52200,
                    5894537.93372833,
                    -2943600.33881516,
                    2507300.84221936,
                    52500,
                    6145319.33336977,
                    -3444710.88598747,
                    330489.762265361,
                    52800,
                    5772252.92619536,
                    -3596135.17460611,
                    -1880004.68907179,
                    53100,
                    4814227.70009439,
                    -3383142.53754643,
                    -3899575.22652161,
                    53400,
                    3369646.74065646,
                    -2828251.54052332,
                    -5524142.63835726,
                    53700,
                    1585543.23813733,
                    -1988421.42228563,
                    -6590845.93086525,
                    5.4e4,
                    -357910.350092875,
                    -948803.773623948,
                    -6993738.51176149,
                    54300,
                    -2265501.33106186,
                    185984.632498067,
                    -6693404.63427837,
                    54600,
                    -3945946.4455674,
                    1302118.36361238,
                    -5720250.0261776,
                    54900,
                    -5230349.11136857,
                    2287547.71417922,
                    -4171552.53759813,
                    55200,
                    -5988848.52093772,
                    3042952.96658151,
                    -2202401.06485164,
                    55500,
                    -6143975.07878275,
                    3491770.60452847,
                    -10850.1513717031,
                    55800,
                    -5679085.05394414,
                    3588209.85780387,
                    2181750.0146522,
                    56100,
                    -4640533.36129799,
                    3322189.20886295,
                    4153363.06092158,
                    56400,
                    -3133066.67299684,
                    2720467.78933698,
                    5704180.01803677,
                    56700,
                    -1309004.93755622,
                    1843842.17028288,
                    6677068.31359695,
                    5.7e4,
                    647339.025015699,
                    780870.816708856,
                    6973340.32950533,
                    57300,
                    2538181.10727378,
                    -361055.853775189,
                    6562510.21023275,
                    57600,
                    4171969.98041122,
                    -1466397.21111577,
                    5485442.38520507,
                    57900,
                    5382654.17879044,
                    -2423011.01358071,
                    3850674.60326007,
                    58200,
                    6046826.41525439,
                    -3133606.57827198,
                    1823944.99746878,
                    58500,
                    6096863.76991072,
                    -3525944.93394749,
                    -388529.418463662,
                    58800,
                    5528240.75886515,
                    -3560508.1956446,
                    -2561536.94201627,
                    59100,
                    4399833.93046646,
                    -3234555.4105321,
                    -4474596.98883736,
                    59400,
                    2827192.79466463,
                    -2582047.94273418,
                    -5934852.21727813,
                    59700,
                    969951.943537454,
                    -1669655.41291604,
                    -6796335.58749306,
                    6e4,
                    -984719.898824341,
                    -589618.976376381,
                    -6973776.52312107,
                    60300,
                    -2840679.06207876,
                    549542.579643414,
                    -6450181.15540378,
                    60600,
                    -4411762.44368795,
                    1633582.32761397,
                    -5278101.02383552,
                    60900,
                    -5539821.8812025,
                    2553564.38717868,
                    -3574700.65443009,
                    61200,
                    -6110499.4216937,
                    3216608.78411877,
                    -1510778.70527264,
                    61500,
                    -6065185.45343018,
                    3555380.31168716,
                    705776.464615322,
                    61800,
                    -5407563.38400485,
                    3535213.46929476,
                    2850853.06717591,
                    62100,
                    -4203610.62525511,
                    3157878.15298021,
                    4707132.72088483,
                    62400,
                    -2574879.64903651,
                    2461435.61532725,
                    6086499.1699117,
                    62700,
                    -685954.322605996,
                    1516258.5606979,
                    6849182.89364973,
                    6.3e4,
                    1272288.33228187,
                    417822.56705157,
                    6917698.46459339,
                    63300,
                    3101775.8499355,
                    -722863.509031593,
                    6284514.38079727,
                    63600,
                    4616992.47561324,
                    -1790292.97190264,
                    5013016.79790008,
                    63900,
                    5663770.71640256,
                    -2676070.71899226,
                    3231624.16863578,
                    64200,
                    6135371.4121741,
                    -3290080.45730932,
                    1121203.79010737,
                    64500,
                    5983939.59297669,
                    -3569994.43267187,
                    -1103399.34590389,
                    64800,
                    5225648.29405979,
                    -3487870.2459942,
                    -3215885.50983491,
                    65100,
                    3938681.17004128,
                    -3052900.18665214,
                    -5002296.80027205,
                    65400,
                    2254438.80143069,
                    -2310028.26003838,
                    -6283008.92961188,
                    65700,
                    343446.90133987,
                    -1334858.28319514,
                    -6930384.09001317,
                    6.6e4,
                    -1602050.37846173,
                    -225735.909154036,
                    -6880611.66512792,
                    66300,
                    -3386944.40986816,
                    906017.150337268,
                    -6139248.17757944,
                    66600,
                    -4832070.53958701,
                    1946867.81229942,
                    -4780477.35542575,
                    66900,
                    -5791683.20148079,
                    2792087.12003739,
                    -2940208.5728573,
                    67200,
                    -6168194.3033376,
                    3356188.3380088,
                    -803226.305916316,
                    67500,
                    -5922563.36745149,
                    3581765.2245831,
                    1414926.17666612,
                    67800,
                    -5078828.91845141,
                    3445624.16493429,
                    3489781.9709688,
                    68100,
                    -3721915.49406421,
                    2961325.63371038,
                    5211083.51804886,
                    68400,
                    -1988894.99497894,
                    2177777.89839556,
                    6404406.39513106,
                    68700,
                    -54888.2078361055,
                    1174149.33331489,
                    6948796.88236319,
                    6.9e4,
                    1884640.70448398,
                    51820.9903687732,
                    6788770.41227551,
                    69300,
                    3633383.3695744,
                    -1075728.66780282,
                    5939851.50484081,
                    69600,
                    5013822.83771548,
                    -2094221.77216089,
                    4487337.50776009,
                    69900,
                    5885368.24871984,
                    -2900147.9444263,
                    2578216.96936549,
                    70200,
                    6159167.95347313,
                    -3411520.82070492,
                    406556.345744964,
                    70500,
                    5807698.74230004,
                    -3576568.0997769,
                    -1806529.61694144,
                    70800,
                    4867646.30137514,
                    -3379161.20988542,
                    -3836145.68282692,
                    71100,
                    3435604.13167031,
                    -2840233.61266231,
                    -5477156.69730701,
                    71400,
                    1657370.97268364,
                    -2015140.42598643,
                    -6565011.82694444,
                    71700,
                    -287436.06291879,
                    -987566.894613425,
                    -6991631.49004769,
                    7.2e4,
                    -2203449.68389942,
                    139065.542320005,
                    -6715236.62764392,
                    72300,
                    -3898549.0661614,
                    1251748.63425776,
                    -5763854.67859798,
                    72600,
                    -5202391.35629523,
                    2238791.87144665,
                    -4232583.77922341,
                    72900,
                    -5983183.28985049,
                    3000731.20227799,
                    -2274741.09719522,
                    73200,
                    -6161212.76191318,
                    3460356.43482851,
                    -87207.9167899129,
                    73212.0,
                    -6154907.45142338,
                    3472367.94593854,
                    7596.60346604248,
                    73212.755328,
                    -6154869.70054271,
                    3472314.27813954,
                    7580.8794079691,
                    73500,
                    -5717448.81578568,
                    3570745.1629146,
                    2109098.31104496,
                    73800,
                    -4696169.83114091,
                    3320482.84120077,
                    4091785.03705666,
                    74100,
                    -3200328.99009984,
                    2734674.45681108,
                    5659891.92891485,
                    74400,
                    -1381077.01266596,
                    1872509.96377624,
                    6654523.77848442,
                    74700,
                    577748.090126868,
                    821093.769286612,
                    6974795.11523067,
                    7.5e4,
                    2478109.48884739,
                    -313347.47804376,
                    6587807.13665381,
                    75300,
                    4127505.31278952,
                    -1416035.41741213,
                    5532024.2188774,
                    75600,
                    5358320.86187946,
                    -2375108.38976716,
                    3913825.28354394,
                    75900,
                    6045114.87388492,
                    -3093035.47796134,
                    1897250.96078875,
                    76200,
                    6117959.70137134,
                    -3496831.82171999,
                    -312528.286799539,
                    76500,
                    5569999.09392809,
                    -3545804.51320804,
                    -2490571.51128057,
                    76800,
                    4458008.95194446,
                    -3235738.76990929,
                    -4415862.92262388,
                    77100,
                    2895892.83203338,
                    -2598985.49449698,
                    -5894275.70505342,
                    77400,
                    1042253.23018028,
                    -1700634.28000537,
                    -6777992.26999029,
                    77700,
                    -916077.187647013,
                    -631528.141129369,
                    -6979509.92543813,
                    7.8e4,
                    -2782581.13402601,
                    500903.59787269,
                    -6479436.6285118,
                    78300,
                    -4370052.94363424,
                    1583094.46259798,
                    -5327977.01007168,
                    78600,
                    -5518721.85709652,
                    2506309.26460801,
                    -3640216.28040287,
                    78900,
                    -6112168.23541311,
                    3177357.25513652,
                    -1585350.31294321,
                    79200,
                    -6089477.83093395,
                    3528100.98228024,
                    629678.589070026,
                    79500,
                    -5452026.20496159,
                    3522659.69615828,
                    2780930.00344605,
                    79800,
                    -4263731.70020275,
                    3161305.12361798,
                    4650456.05959722,
                    80100,
                    -2644559.15506612,
                    2480478.79327034,
                    6048784.02286021,
                    80400,
                    -758135.633598154,
                    1548979.41575856,
                    6834216.6865031,
                    80700,
                    1204905.05796468,
                    460907.635765506,
                    6926974.41977554,
                    8.1e4,
                    3046008.27476155,
                    -673773.966409097,
                    6317089.29694364,
                    81300,
                    4578498.46402567,
                    -1740174.28613336,
                    5065597.72063859,
                    81600,
                    5646473.68468333,
                    -2630014.64015221,
                    3298882.43495241,
                    81900,
                    6141046.72596935,
                    -3252772.80744689,
                    1196302.81750839,
                    82200,
                    6012018.04859162,
                    -3545227.32649779,
                    -1028103.34443477,
                    82500,
                    5273272.81266634,
                    -3478149.61476025,
                    -3148045.87613995,
                    82800,
                    4001014.56171013,
                    -3059196.63136835,
                    -4948782.04286484,
                    83100,
                    2325178.22912325,
                    -2331694.39515607,
                    -6249209.71055206,
                    83400,
                    415473.178002811,
                    -1369709.78007679,
                    -6919692.49653159,
                    83700,
                    -1535964.66503722,
                    -270276.547186386,
                    -6894107.98457841,
                    8.4e4,
                    -3333431.83769341,
                    856252.019175884,
                    -6175604.75808401,
                    84300,
                    -4796522.04551855,
                    1896876.29005214,
                    -4836079.84361999,
                    84600,
                    -5777706.24779571,
                    2746906.16140639,
                    -3009492.61314694,
                    84900,
                    -6177229.98169397,
                    3320382.63185734,
                    -879217.258095039,
                    85019.0,
                    -6162999.06701901,
                    3457830.12375349,
                    7561.5554257676,
                    85019.62118399999,
                    -6162974.43645965,
                    3457855.60563033,
                    7854.92641443949,
                    85200,
                    -5953637.85681474,
                    3558954.7436216,
                    1340197.97209599,
                    85500,
                    -5128782.17352735,
                    3438064.29435161,
                    3423588.54273433,
                    85800,
                    -3785673.79051689,
                    2969762.46793873,
                    5160112.25565087,
                    86100,
                    -2059990.30391251,
                    2201339.18545503,
                    6373787.29692082,
                    86400,
                    -126123.203602025,
                    1210440.95210778,
                    6941595.81244609,
                ],
            ),
        ),
    ]
)
