"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeadlineClient = void 0;
const http = require("http");
const https = require("https");
/**
 * Implements a simple client that supports HTTP/HTTPS GET and POST requests.
 * It is intended to be used within Custom Resources that need to send the requests to the Render Queue.
 */
class DeadlineClient {
    constructor(props) {
        this.requestOptions = {
            host: props.host,
            port: props.port,
        };
        if (props.protocol === 'HTTPS') {
            this.protocol = https;
            this.requestOptions.agent = new https.Agent({
                pfx: props.tls?.pfx,
                passphrase: props.tls?.passphrase,
                ca: props.tls?.ca,
            });
        }
        else {
            this.protocol = http;
        }
        this.retries = props.retries ?? DeadlineClient.DEFAULT_RETRY_COUNT;
        this.retryWaitMs = props.retryWaitMs ?? DeadlineClient.DEFAULT_RETRY_PERIOD_MS;
    }
    /**
     * Perform an HTTP GET request.
     *
     * @param path The resource to request for.
     * @param requestOptions Other request options, including headers, timeout, etc.
     */
    async GetRequest(path, requestOptions) {
        const options = this.FillRequestOptions(path, 'GET', requestOptions);
        return this.performRequestWithRetry(options, this.retries, this.retryWaitMs);
    }
    /**
     * Perform an HTTP POST request.
     *
     * @param path The resource to request for.
     * @param data The data (body) of the request that contains the information to be sent.
     * @param requestOptions Other request options, including headers, timeout, etc.
     */
    async PostRequest(path, data, requestOptions) {
        const options = this.FillRequestOptions(path, 'POST', requestOptions);
        return this.performRequestWithRetry(options, this.retries, this.retryWaitMs, data ? JSON.stringify(data) : undefined);
    }
    FillRequestOptions(path, method, requestOptions) {
        const options = {
            ...requestOptions,
            port: this.requestOptions.port,
            host: this.requestOptions.host,
            agent: this.requestOptions.agent,
            path: path,
            method: method,
        };
        return options;
    }
    async performRequestWithRetry(options, retriesLeft, retryDelayMs, data) {
        try {
            return await this.performRequest(options, data);
        }
        catch (exception) {
            const { statusCode, statusMessage } = exception;
            if (statusCode !== undefined && statusMessage !== undefined) {
                if (statusCode >= 500 && retriesLeft > 0) {
                    console.log(`Request failed with ${statusCode}: ${statusMessage}. Will retry after ${retryDelayMs} ms.`);
                    console.log(`Retries left: ${retriesLeft}`);
                    const delay = (ms) => new Promise(res => setTimeout(res, ms));
                    await delay(retryDelayMs);
                    return await this.performRequestWithRetry(options, retriesLeft - 1, retryDelayMs, data);
                }
                else {
                    return await Promise.reject(statusMessage);
                }
            }
            else {
                throw (exception);
            }
        }
    }
    async performRequest(options, data) {
        return new Promise((resolve, reject) => {
            const req = this.protocol.request(options, response => {
                const { statusCode } = response;
                if (!statusCode || statusCode >= 300) {
                    return reject(response);
                }
                else {
                    const chunks = [];
                    response.on('data', (chunk) => {
                        chunks.push(chunk);
                    });
                    response.on('end', () => {
                        const stringData = Buffer.concat(chunks).toString();
                        const result = {
                            data: JSON.parse(stringData),
                            fullResponse: response,
                        };
                        return resolve(result);
                    });
                }
            });
            req.on('error', reject);
            if (data) {
                req.write(data);
            }
            req.end();
        });
    }
}
exports.DeadlineClient = DeadlineClient;
/**
 * The default number of retry attempts.
 */
DeadlineClient.DEFAULT_RETRY_COUNT = 3;
/**
 * Specifies the default waiting period between two requests.
 */
DeadlineClient.DEFAULT_RETRY_PERIOD_MS = 10000;
//# sourceMappingURL=data:application/json;base64,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