"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCustomResource = void 0;
const hash_1 = require("./hash");
const reply_1 = require("./reply");
/* istanbul ignore file */
class SimpleCustomResource {
    constructor() {
        // Optionally suppress debugging statements.
        this.debugMode = (process.env.DEBUG ?? 'false') === 'true';
        if (!this.debugMode) {
            console.debug = () => { };
        }
    }
    /**
     * Handler/engine for the CustomResource state machine. Users of this class should
     * instantiate the class, and then immediately call this function.
     * @param event The event passed to the lambda handler.
     * @param context The lambda context passed to the lambda handler.
     */
    async handler(event, context) {
        let status = reply_1.CfnResponseStatus.SUCCESS;
        let failReason;
        let cfnData;
        console.log(`Handling event: ${JSON.stringify(event)}`);
        const resourceProperties = event.ResourceProperties ?? {};
        const physicalId = hash_1.calculateSha256Hash(resourceProperties);
        try {
            const timeout = (prom, time, exception) => {
                let timer;
                return Promise.race([
                    prom,
                    new Promise((_r, rej) => timer = setTimeout(rej, time, exception)),
                ]).finally(() => clearTimeout(timer));
            };
            // We want to always notify CloudFormation about the success/failure of the Lambda at all times.
            // If function execution time is longer than Lambda's timeout, then the function is just stopped
            // and CloudFormation is not notified at all. This would result in a hang-up during deployment.
            // Thus, we want to stop the execution by ourselves before the Lambda timeout and reserve some time
            // for notifying a CloudFormation about a failed deployment because of the timeout.
            // 3 seconds should be enough to resolve the request that signals success/failure of the custom resource,
            // but if Lambda timeout is too small, we would reserve 20% of the remaining time and still try to notify the CF.
            // Check the logs during the development to see if you allocated enough time for your Lambda.
            const defaultReserveTimeMs = 3000;
            const remainingTimeMs = context.getRemainingTimeInMillis();
            let reserveTimeMs = Math.min(0.2 * remainingTimeMs, defaultReserveTimeMs);
            if (reserveTimeMs < defaultReserveTimeMs) {
                console.debug(`The remaining Lambda execution time of ${reserveTimeMs} ` +
                    `ms might not be sufficient to send a CloudFormation response. At least ${defaultReserveTimeMs} ms is required. ` +
                    'Please increase the Lambda timeout.');
            }
            cfnData = await timeout(this.handleEvent(event, context, resourceProperties, physicalId), remainingTimeMs - reserveTimeMs, new Error('Timeout error'));
        }
        catch (e) {
            // We want to always catch the exception for a CfnCustomResource CloudFormation
            // must be notified about the success/failure of the lambda at all times;
            // failure to notify results in a stuck stack that takes at least an hour to
            // timeout.
            status = reply_1.CfnResponseStatus.FAILED;
            failReason = `${e.message}\n${e.stack}`;
        }
        finally {
            // Always send a response to CloudFormation, signal success or
            // failure based on whether or not we had an exception.
            await reply_1.sendCfnResponse({
                event,
                context,
                status,
                reason: failReason,
                physicalId,
                data: cfnData,
            });
        }
        const response = `${status}` + (failReason ?? '');
        console.log(`Result: ${response}`);
        return response;
    }
    async handleEvent(event, context, props, physicalId) {
        const requestType = event.RequestType;
        let cfnData;
        if (requestType === 'Create') {
            if (!this.validateInput(props)) {
                throw Error(`Input did not pass validation check. Check log group "${context.logGroupName}" ` +
                    `for log stream ${context.logStreamName} for additional information.`);
            }
            cfnData = await this.doCreate(physicalId, props);
            console.debug(`Create data: ${JSON.stringify(cfnData)}`);
        }
        else if (requestType === 'Update') {
            if (!this.validateInput(props)) {
                throw Error('Input did not pass validation check');
            }
            const oldResourceProperties = event.OldResourceProperties ?? {};
            const oldPhysicalId = hash_1.calculateSha256Hash(oldResourceProperties);
            if (oldPhysicalId !== physicalId) {
                console.log('Doing Create -- ResourceProperties differ.');
                cfnData = await this.doCreate(physicalId, props);
                console.debug(`Update data: ${JSON.stringify(cfnData)}`);
            }
        }
        else {
            await this.doDelete(physicalId, props);
        }
        return cfnData;
    }
}
exports.SimpleCustomResource = SimpleCustomResource;
//# sourceMappingURL=data:application/json;base64,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