"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const AWS = require("aws-sdk");
const deadline_1 = require("../../../../deadline");
const handler_1 = require("../handler");
jest.mock('../../lib/secrets-manager/read-certificate');
const secretPartialArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
describe('SEPConfiguratorResource', () => {
    const deadlineGroup = 'group_name';
    const deadlinePool = 'pool_name';
    let app;
    let stack;
    let validSepConfiguration;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        const launchTemplate = new aws_ec2_1.LaunchTemplate(stack, 'LaunchTemplate');
        validSepConfiguration = {
            spotPluginConfigurations: {
                AWSInstanceStatus: deadline_1.SpotEventPluginDisplayInstanceStatus.DISABLED,
                DeleteInterruptedSlaves: true,
                DeleteTerminatedSlaves: true,
                IdleShutdown: 20,
                Logging: deadline_1.SpotEventPluginLoggingLevel.STANDARD,
                PreJobTaskMode: deadline_1.SpotEventPluginPreJobTaskMode.CONSERVATIVE,
                Region: 'us-west-2',
                ResourceTracker: true,
                StaggerInstances: 50,
                State: deadline_1.SpotEventPluginState.GLOBAL_ENABLED,
                StrictHardCap: true,
            },
            connection: {
                hostname: 'internal-hostname.com',
                protocol: 'HTTPS',
                port: '4433',
                caCertificateArn: secretPartialArn,
            },
            spotFleetRequestConfigurations: {
                [deadlineGroup]: {
                    AllocationStrategy: deadline_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED,
                    IamFleetRole: 'roleArn',
                    // Explicitly provide empty array for testing comparisons since we inject this for compatibility with SEP
                    LaunchSpecifications: [],
                    LaunchTemplateConfigs: [{
                            LaunchTemplateSpecification: {
                                LaunchTemplateId: launchTemplate.launchTemplateId,
                                LaunchTemplateName: launchTemplate.launchTemplateName,
                                Version: launchTemplate.versionNumber,
                            },
                            Overrides: [],
                        }],
                    ReplaceUnhealthyInstances: true,
                    TargetCapacity: 1,
                    TerminateInstancesWithExpiration: true,
                    Type: deadline_1.SpotFleetRequestType.MAINTAIN,
                    ValidUntil: aws_cdk_lib_1.Expiration.atDate(new Date(2022, 11, 17)).date.toISOString(),
                    TagSpecifications: [{
                            ResourceType: deadline_1.SpotFleetResourceType.SPOT_FLEET_REQUEST,
                            Tags: [
                                {
                                    Key: 'name',
                                    Value: 'test',
                                },
                            ],
                        }],
                },
            },
            deadlineGroups: [deadlineGroup],
            deadlinePools: [deadlinePool],
        };
    });
    describe('doCreate', () => {
        let handler;
        let mockSpotEventPluginClient;
        beforeEach(() => {
            mockSpotEventPluginClient = {
                saveServerData: jest.fn((_a) => Promise.resolve(true)),
                configureSpotEventPlugin: jest.fn((_a) => Promise.resolve(true)),
                addGroups: jest.fn((_a) => Promise.resolve(true)),
                addPools: jest.fn((_a) => Promise.resolve(true)),
            };
            handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            async function returnSpotEventPluginClient(_v1) {
                return mockSpotEventPluginClient;
            }
            // eslint-disable-next-line dot-notation
            handler['spotEventPluginClient'] = jest.fn((a) => returnSpotEventPluginClient(a));
        });
        afterEach(() => {
            jest.clearAllMocks();
        });
        test('does not save server data when no configurations are provided', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const result = await handler.doCreate('physicalId', {
                connection: validSepConfiguration.connection,
            });
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(0);
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(0);
        });
        test('save spot fleet request configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', {
                ...validSepConfiguration,
                spotPluginConfigurations: undefined,
            });
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            expect(mockSaveServerData).toBeCalledWith(JSON.stringify(validSepConfiguration.spotFleetRequestConfigurations));
        });
        test('save spot event plugin configs', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validSepConfiguration.spotPluginConfigurations)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            const result = await handler.doCreate('physicalId', {
                ...validSepConfiguration,
                spotFleetRequestConfigurations: undefined,
            });
            // THEN
            expect(result).toBeUndefined();
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('save server data', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            expect(mockSaveServerData.mock.calls[0][0]).toEqual(JSON.stringify(validSepConfiguration.spotFleetRequestConfigurations));
        });
        test('configure spot event plugin', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validSepConfiguration.spotPluginConfigurations)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('create groups', async () => {
            // GIVEN
            const mockAddGroups = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.addGroups = mockAddGroups;
            // WHEN
            await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(mockAddGroups.mock.calls.length).toBe(1);
            expect(mockAddGroups).toHaveBeenCalledWith([deadlineGroup]);
        });
        test('create pools', async () => {
            // GIVEN
            const mockAddPools = jest.fn((_a) => Promise.resolve(true));
            mockSpotEventPluginClient.addPools = mockAddPools;
            // WHEN
            await handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            expect(mockAddPools.mock.calls.length).toBe(1);
            expect(mockAddPools).toHaveBeenCalledWith([deadlinePool]);
        });
        test('throw when cannot add groups', async () => {
            // GIVEN
            mockSpotEventPluginClient.addGroups = jest.fn((_a) => Promise.resolve(false));
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(`Failed to add Deadline group(s) ${validSepConfiguration.deadlineGroups}`);
        });
        test('throw when cannot add pools', async () => {
            // GIVEN
            mockSpotEventPluginClient.addPools = jest.fn((_a) => Promise.resolve(false));
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(`Failed to add Deadline pool(s) ${validSepConfiguration.deadlinePools}`);
        });
        test('throw when cannot save spot fleet request configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((_a) => Promise.resolve(false));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save spot fleet request with configuration/);
        });
        test('throw when cannot save spot event plugin configs', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(false));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const promise = handler.doCreate('physicalId', validSepConfiguration);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save Spot Event Plugin Configurations/);
        });
    });
    test('doDelete does not do anything', async () => {
        // GIVEN
        const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
        // WHEN
        const promise = await handler.doDelete('physicalId', {
            connection: validSepConfiguration.connection,
        });
        // THEN
        await expect(promise).toBeUndefined();
    });
    describe('.validateInput()', () => {
        describe('should return true', () => {
            test('with valid input', async () => {
                // GIVEN
                const input = validSepConfiguration;
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
            test('without spotPluginConfigurations', async () => {
                // GIVEN
                const input = {
                    ...validSepConfiguration,
                    spotPluginConfigurations: undefined,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
            test('without spotFleetRequestConfigurations', async () => {
                // GIVEN
                const input = {
                    ...validSepConfiguration,
                    spotFleetRequestConfigurations: undefined,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
            test('with only connection', async () => {
                // GIVEN
                const input = {
                    connection: validSepConfiguration.connection,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
        });
        // Invalid connection
        const noProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            port: '4433',
        };
        const noHostnameConnection = {
            protocol: 'HTTPS',
            port: '4433',
        };
        const noPortConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            caCertificateArn: secretPartialArn,
        };
        const invalidHostnameConnection = {
            hostname: 10,
            protocol: 'HTTPS',
            port: '4433',
        };
        const invalidProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'TCP',
            port: '4433',
        };
        const invalidProtocolTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: ['HTTPS'],
            port: '4433',
        };
        const invalidPortTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: 4433,
        };
        const invalidPortRange1Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '-1',
        };
        const invalidPortRange2Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '65536',
        };
        const invalidPortRange3Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: Number.NaN.toString(),
        };
        const invalidCaCertConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '4433',
            caCertificateArn: 'notArn',
        };
        describe('should return false if', () => {
            test.each([
                noProtocolConnection,
                noHostnameConnection,
                noPortConnection,
                invalidCaCertConnection,
                invalidHostnameConnection,
                invalidProtocolConnection,
                invalidProtocolTypeConnection,
                invalidPortTypeConnection,
                invalidPortRange1Connection,
                invalidPortRange2Connection,
                invalidPortRange3Connection,
                undefined,
                [],
            ])('invalid connection', (invalidConnection) => {
                // GIVEN
                const input = {
                    spotPluginConfigurations: validSepConfiguration.spotPluginConfigurations,
                    connection: invalidConnection,
                    spotFleetRequestConfigurations: validSepConfiguration.spotFleetRequestConfigurations,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
            test.each([
                undefined,
                [],
                '',
            ])('{input=%s}', (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.isSecretArnOrUndefined()', () => {
        describe('should return true if', () => {
            test.each([
                secretPartialArn,
                undefined,
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeTruthy();
            });
        });
        describe('should return false if', () => {
            test.each([
                'any string',
                10,
                [],
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.spotEventPluginClient()', () => {
        test('creates a valid object with http', async () => {
            // GIVEN
            const validHTTPConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPConnection);
            expect(result).toBeDefined();
        });
        test('creates a valid object with https', async () => {
            // GIVEN
            const validHTTPSConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
                caCertificateArn: secretPartialArn,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPSConnection);
            expect(result).toBeDefined();
        });
    });
    describe('.toKeyValueArray()', () => {
        test('converts to array of key value pairs', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: 'Disabled',
            };
            const expectedResult = {
                Key: 'AWSInstanceStatus',
                Value: 'Disabled',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const returnValue = handler['toKeyValueArray'](pluginConfig);
            // THEN
            expect(returnValue).toContainEqual(expectedResult);
        });
        test('throws with undefined values', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: undefined,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            function toKeyValueArray() {
                // eslint-disable-next-line dot-notation
                handler['toKeyValueArray'](pluginConfig);
            }
            // THEN
            expect(toKeyValueArray).toThrowError();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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