"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let versionedInstallers;
let dependencyStack;
let dockerContainer;
let images;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let vpc;
let workerFleet;
const DEFAULT_CONSTRUCT_ID = 'UBL';
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new aws_cdk_lib_1.App();
        dependencyStack = new aws_cdk_lib_1.Stack(app, 'DependencyStack', { env });
        versionedInstallers = new lib_1.VersionQuery(dependencyStack, 'VersionQuery');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: versionedInstallers,
            }),
            version: versionedInstallers,
        });
        jest.spyOn(renderQueue, 'configureSecretsManagementAutoRegistration');
        stack = new aws_cdk_lib_1.Stack(app, 'Stack', { env });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(stack, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        licenses = [lib_1.UsageBasedLicense.forMaya()];
    });
    function createUbl(props) {
        return new lib_1.UsageBasedLicensing(stack, DEFAULT_CONSTRUCT_ID, {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
            ...props,
        });
    }
    test('vpcSubnets specified => does not emit warnings', () => {
        // GIVEN
        const vpcSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        };
        // WHEN
        const ubl = createUbl({
            vpcSubnets,
        });
        // THEN
        assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${ubl.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${ubl.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(stack).hasNoError(`/${ubl.node.path}`, assertions_1.Match.anyValue());
    });
    test('vpcSubnets not specified => emits warning about dedicated subnets', () => {
        // WHEN
        const ubl = createUbl();
        // THEN
        assertions_1.Annotations.fromStack(stack).hasWarning(`/${ubl.node.path}`, 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
    });
    describe('configures auto registration', () => {
        test('default to private subnets', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            const expectedCall = {
                dependent: ubl.service.node.defaultChild,
                registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                role: lib_1.SecretsManagementRole.CLIENT,
                vpc,
                vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS },
            };
            // THEN
            expect(renderQueue.configureSecretsManagementAutoRegistration).toHaveBeenCalledWith(expectedCall);
        });
        test.each([
            [{
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                }],
        ])('%s', (vpcSubnets) => {
            // WHEN
            const ubl = createUbl({
                vpcSubnets,
            });
            // THEN
            const expectedCall = {
                dependent: ubl.service.node.defaultChild,
                registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                role: lib_1.SecretsManagementRole.CLIENT,
                vpc,
                vpcSubnets,
            };
            // THEN
            expect(renderQueue.configureSecretsManagementAutoRegistration).toHaveBeenCalledWith(expectedCall);
        });
    });
    test('creates an ECS cluster', () => {
        // WHEN
        createUbl();
        // THEN
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::ECS::Cluster', 1);
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assertions_1.Match.stringLikeRegexp(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet.*`),
                    },
                    {
                        'Fn::ImportValue': assertions_1.Match.stringLikeRegexp(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet.*`),
                    },
                ],
            });
        });
        test('capacity can be specified', () => {
            // WHEN
            createUbl({
                desiredCount: 2,
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            });
        });
        test('gives write access to log group', () => {
            // GIVEN
            const ubl = createUbl();
            // WHEN
            const logGroup = ubl.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
            const asgRoleLogicalId = aws_cdk_lib_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assertions_1.Match.arrayWith([
                        {
                            Action: assertions_1.Match.arrayWith([
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ]),
                            Effect: 'Allow',
                            Resource: stack.resolve(logGroup.logGroupArn),
                        },
                    ]),
                    Version: '2012-10-17',
                },
                Roles: assertions_1.Match.arrayWith([
                    { Ref: asgRoleLogicalId },
                ]),
            });
        });
        test('uses the supplied security group', () => {
            // GIVEN
            const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'UblSecurityGroup', {
                vpc,
            });
            // WHEN
            createUbl({ securityGroup });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(securityGroup.securityGroupId)]),
            });
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            });
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // WHEN
                createUbl();
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                    DesiredCount: 1,
                });
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                // WHEN
                createUbl({ desiredCount });
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                });
            });
        });
        test('sets launch type to EC2', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                LaunchType: 'EC2',
            });
        });
        test('sets distinct instance placement constraint', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                PlacementConstraints: assertions_1.Match.arrayWith([
                    { Type: 'distinctInstance' },
                ]),
            });
        });
        test('uses the task definition', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            });
        });
        test('with the correct deployment configuration', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            });
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            });
        });
        test('container is marked essential', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            });
        });
        test('with increased ulimits', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            });
        });
        test('with awslogs log driver', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            });
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const ubl = createUbl();
            // WHEN
            const taskRoleLogicalId = aws_cdk_lib_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assertions_1.Match.arrayWith([
                            {
                                Name: 'UBL_CERTIFICATES_URI',
                                Value: certificateSecret.secretArn,
                            },
                        ]),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            });
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assertions_1.Match.arrayWith([
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: certificateSecret.secretArn,
                        },
                    ]),
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: aws_cdk_lib_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            });
        });
        test('uses host networking', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            });
        });
        test('is marked EC2 compatible only', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            });
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        // WHEN
        createUbl({
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        });
    });
    test.each([
        'test-prefix/',
        '',
    ])('License Forwarder is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = DEFAULT_CONSTRUCT_ID;
        // WHEN
        createUbl({
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        });
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // WHEN
            createUbl({
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assertions_1.Match.arrayWith([
                            {
                                Name: 'UBL_LIMITS',
                                Value: 'maya:10;vray:10',
                            },
                        ]),
                    },
                ],
            });
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4151, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5555, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const ubl = createUbl();
            const workerStack = new aws_cdk_lib_1.Stack(app, 'WorkerStack', { env });
            workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(workerStack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                }),
            });
            // WHEN
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                assertions_1.Template.fromStack(workerStack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::ImportValue': assertions_1.Match.stringLikeRegexp(`${aws_cdk_lib_1.Stack.of(ubl).stackName}:ExportsOutputFnGetAttUBLClusterASGInstanceSecurityGroup.*`),
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                });
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                createUbl({ licenses: [] });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule from UsageBasedLicensing ASG to RenderQueue ASG', () => {
            // GIVEN
            const renderQueueSecurityGroup = renderQueue.connections.securityGroups[0];
            // WHEN
            const ubl = createUbl();
            const ublSecurityGroup = ubl.connections.securityGroups[0];
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 4433,
                ToPort: 4433,
                GroupId: stack.resolve(renderQueueSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(ublSecurityGroup.securityGroupId),
            });
        });
        test('adds ingress rule from RenderQueue ASG to UsageBasedLicensing ASG', () => {
            // GIVEN
            const renderQueueSecurityGroup = renderQueue.backendConnections.securityGroups[0];
            // WHEN
            const ubl = createUbl();
            const ublSecurityGroup = ubl.connections.securityGroups[0];
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 17004,
                ToPort: 17004,
                GroupId: stack.resolve(ublSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(renderQueueSecurityGroup.securityGroupId),
            });
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // WHEN
            createUbl();
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assertions_1.Match.arrayWith([
                            {
                                Name: 'RENDER_QUEUE_URI',
                                Value: stack.resolve(`${renderQueue.endpoint.applicationProtocol.toLowerCase()}://${renderQueue.endpoint.socketAddress}`),
                            },
                        ]),
                    },
                ],
            });
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                createUbl();
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNhZ2UtYmFzZWQtbGljZW5zaW5nLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ1c2FnZS1iYXNlZC1saWNlbnNpbmcudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOztBQUVILDZDQUlxQjtBQUNyQix1REFJZ0M7QUFDaEMsaURBTzZCO0FBQzdCLCtEQUVvQztBQUNwQyxpREFHNkI7QUFJN0IsdUVBR3dDO0FBRXhDLDZEQUVxQztBQUNyQyxnQ0FjZ0I7QUFFaEIsTUFBTSxHQUFHLEdBQUc7SUFDVixNQUFNLEVBQUUsV0FBVztDQUNwQixDQUFDO0FBQ0YsSUFBSSxHQUFRLENBQUM7QUFDYixJQUFJLGlCQUEwQixDQUFDO0FBQy9CLElBQUksbUJBQTZCLENBQUM7QUFDbEMsSUFBSSxlQUFzQixDQUFDO0FBQzNCLElBQUksZUFBaUMsQ0FBQztBQUN0QyxJQUFJLE1BQWlDLENBQUM7QUFDdEMsSUFBSSxRQUE2QixDQUFDO0FBQ2xDLElBQUksUUFBd0IsQ0FBQztBQUM3QixJQUFJLFdBQXdCLENBQUM7QUFDN0IsSUFBSSxLQUFZLENBQUM7QUFDakIsSUFBSSxHQUFTLENBQUM7QUFDZCxJQUFJLFdBQXlCLENBQUM7QUFFOUIsTUFBTSxvQkFBb0IsR0FBRyxLQUFLLENBQUM7QUFFbkMsUUFBUSxDQUFDLHFCQUFxQixFQUFFLEdBQUcsRUFBRTtJQUNuQyxVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsUUFBUTtRQUNSLEdBQUcsR0FBRyxJQUFJLGlCQUFHLEVBQUUsQ0FBQztRQUVoQixlQUFlLEdBQUcsSUFBSSxtQkFBSyxDQUFDLEdBQUcsRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFFN0QsbUJBQW1CLEdBQUcsSUFBSSxrQkFBWSxDQUFDLGVBQWUsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUV4RSxHQUFHLEdBQUcsSUFBSSxhQUFHLENBQUMsZUFBZSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3RDLFFBQVEsR0FBRyx3QkFBYyxDQUFDLG9CQUFvQixDQUFDLElBQUksaUNBQWdCLENBQUMsZUFBZSxFQUFFLE9BQU8sRUFBRTtZQUM1RixTQUFTLEVBQUUsU0FBUztTQUNyQixDQUFDLENBQUMsQ0FBQztRQUNKLFdBQVcsR0FBRyxJQUFJLGlCQUFXLENBQUMsZUFBZSxFQUFFLG1CQUFtQixFQUFFO1lBQ2xFLEdBQUc7WUFDSCxNQUFNLEVBQUUsRUFBRSxzQkFBc0IsRUFBRSxRQUFRLEVBQUU7WUFDNUMsVUFBVSxFQUFFLElBQUksZ0JBQVUsQ0FBQyxlQUFlLEVBQUUsc0JBQXNCLEVBQUU7Z0JBQ2xFLEdBQUc7Z0JBQ0gsT0FBTyxFQUFFLG1CQUFtQjthQUM3QixDQUFDO1lBQ0YsT0FBTyxFQUFFLG1CQUFtQjtTQUM3QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSw0Q0FBNEMsQ0FBQyxDQUFDO1FBRXRFLEtBQUssR0FBRyxJQUFJLG1CQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDekMsaUJBQWlCLEdBQUcsMkJBQU0sQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLHdFQUF3RSxDQUFDLENBQUM7UUFDaEosZUFBZSxHQUFHLElBQUssaUNBQWdCLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1lBQ2xFLFNBQVMsRUFBRSxTQUFTO1NBQ3JCLENBQUMsQ0FBQztRQUNILE1BQU0sR0FBRztZQUNQLGdCQUFnQixFQUFFLHdCQUFjLENBQUMsb0JBQW9CLENBQUMsZUFBZSxDQUFDO1NBQ3ZFLENBQUM7UUFDRixRQUFRLEdBQUcsQ0FBQyx1QkFBaUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQzNDLENBQUMsQ0FBQyxDQUFDO0lBRUgsU0FBUyxTQUFTLENBQUMsS0FBeUM7UUFDMUQsT0FBTyxJQUFJLHlCQUFtQixDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtZQUMxRCxpQkFBaUI7WUFDakIsTUFBTTtZQUNOLFFBQVE7WUFDUixXQUFXO1lBQ1gsR0FBRztZQUNILEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO1FBQzFELFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBb0I7WUFDbEMsVUFBVSxFQUFFLG9CQUFVLENBQUMsbUJBQW1CO1NBQzNDLENBQUM7UUFFRixPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsU0FBUyxDQUFDO1lBQ3BCLFVBQVU7U0FDWCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1Asd0JBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxrQkFBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDOUUsd0JBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxrQkFBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDakYsd0JBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsVUFBVSxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxrQkFBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDakYsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsbUVBQW1FLEVBQUUsR0FBRyxFQUFFO1FBQzdFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxTQUFTLEVBQUUsQ0FBQztRQUV4QixPQUFPO1FBQ1Asd0JBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsVUFBVSxDQUNyQyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQ25CLHFSQUFxUixDQUN0UixDQUFDO0lBQ0osQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsOEJBQThCLEVBQUUsR0FBRyxFQUFFO1FBQzVDLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxHQUFHLEVBQUU7WUFDdEMsT0FBTztZQUNQLE1BQU0sR0FBRyxHQUFHLFNBQVMsRUFBRSxDQUFDO1lBRXhCLE9BQU87WUFDUCxNQUFNLFlBQVksR0FBNEM7Z0JBQzVELFNBQVMsRUFBRSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxZQUEwQjtnQkFDdEQsa0JBQWtCLEVBQUUseUNBQW1DLENBQUMsVUFBVTtnQkFDbEUsSUFBSSxFQUFFLDJCQUFxQixDQUFDLE1BQU07Z0JBQ2xDLEdBQUc7Z0JBQ0gsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLG9CQUFVLENBQUMsbUJBQW1CLEVBQUU7YUFDM0QsQ0FBQztZQUVGLE9BQU87WUFDUCxNQUFNLENBQUMsV0FBVyxDQUFDLDBDQUEwQyxDQUFDLENBQUMsb0JBQW9CLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDcEcsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxDQUFvQjtZQUMzQixDQUFDO29CQUNDLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE1BQU07aUJBQzlCLENBQUM7U0FDSCxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsVUFBVSxFQUFFLEVBQUU7WUFDdEIsT0FBTztZQUNQLE1BQU0sR0FBRyxHQUFHLFNBQVMsQ0FBQztnQkFDcEIsVUFBVTthQUNYLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLFlBQVksR0FBNEM7Z0JBQzVELFNBQVMsRUFBRSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxZQUEwQjtnQkFDdEQsa0JBQWtCLEVBQUUseUNBQW1DLENBQUMsVUFBVTtnQkFDbEUsSUFBSSxFQUFFLDJCQUFxQixDQUFDLE1BQU07Z0JBQ2xDLEdBQUc7Z0JBQ0gsVUFBVTthQUNYLENBQUM7WUFFRixPQUFPO1lBQ1AsTUFBTSxDQUFDLFdBQVcsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDLG9CQUFvQixDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3BHLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO1FBQ2xDLE9BQU87UUFDUCxTQUFTLEVBQUUsQ0FBQztRQUVaLE9BQU87UUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxlQUFlLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDcEUsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO1FBQzlCLElBQUksQ0FBQyxVQUFVLEVBQUUsR0FBRyxFQUFFO1lBQ3BCLE9BQU87WUFDUCxTQUFTLEVBQUUsQ0FBQztZQUVaLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxvQ0FBb0MsRUFBRTtnQkFDcEYsT0FBTyxFQUFFLEdBQUc7Z0JBQ1osT0FBTyxFQUFFLEdBQUc7Z0JBQ1osaUJBQWlCLEVBQUU7b0JBQ2pCO3dCQUNFLGlCQUFpQixFQUFFLGtCQUFLLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxlQUFlLENBQUMsU0FBUyw0Q0FBNEMsQ0FBQztxQkFDcEg7b0JBQ0Q7d0JBQ0UsaUJBQWlCLEVBQUUsa0JBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLGVBQWUsQ0FBQyxTQUFTLDRDQUE0QyxDQUFDO3FCQUNwSDtpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLDJCQUEyQixFQUFFLEdBQUcsRUFBRTtZQUNyQyxPQUFPO1lBQ1AsU0FBUyxDQUFDO2dCQUNSLFlBQVksRUFBRSxDQUFDO2FBQ2hCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxvQ0FBb0MsRUFBRTtnQkFDcEYsT0FBTyxFQUFFLEdBQUc7Z0JBQ1osT0FBTyxFQUFFLEdBQUc7YUFDYixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxpQ0FBaUMsRUFBRSxHQUFHLEVBQUU7WUFDM0MsUUFBUTtZQUNSLE1BQU0sR0FBRyxHQUFHLFNBQVMsRUFBRSxDQUFDO1lBRXhCLE9BQU87WUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLG9CQUFvQixVQUFVLENBQWMsQ0FBQztZQUNwRixNQUFNLGdCQUFnQixHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO1lBRWxHLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbEUsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxrQkFBSyxDQUFDLFNBQVMsQ0FBQzt3QkFDekI7NEJBQ0UsTUFBTSxFQUFFLGtCQUFLLENBQUMsU0FBUyxDQUFDO2dDQUN0QixzQkFBc0I7Z0NBQ3RCLG1CQUFtQjs2QkFDcEIsQ0FBQzs0QkFDRixNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO3lCQUM5QztxQkFDRixDQUFDO29CQUNGLE9BQU8sRUFBRSxZQUFZO2lCQUN0QjtnQkFDRCxLQUFLLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3JCLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO2lCQUMxQixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsa0NBQWtDLEVBQUUsR0FBRyxFQUFFO1lBQzVDLFFBQVE7WUFDUixNQUFNLGFBQWEsR0FBRyxJQUFJLHVCQUFhLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO2dCQUNqRSxHQUFHO2FBQ0osQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFNBQVMsQ0FBQyxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7WUFFN0IsT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO2dCQUN2RixjQUFjLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO2FBQ2hGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO1FBQ3RDLElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7WUFDdkMsT0FBTztZQUNQLE1BQU0sR0FBRyxHQUFHLFNBQVMsRUFBRSxDQUFDO1lBRXhCLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtnQkFDbkUsT0FBTyxFQUFFLEVBQUUsR0FBRyxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxFQUFFO2FBQ2xGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7WUFDNUIsSUFBSSxDQUFDLGVBQWUsRUFBRSxHQUFHLEVBQUU7Z0JBQ3pCLE9BQU87Z0JBQ1AsU0FBUyxFQUFFLENBQUM7Z0JBRVosT0FBTztnQkFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtvQkFDbkUsWUFBWSxFQUFFLENBQUM7aUJBQ2hCLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtnQkFDNUIsUUFBUTtnQkFDUixNQUFNLFlBQVksR0FBRyxDQUFDLENBQUM7Z0JBRXZCLE9BQU87Z0JBQ1AsU0FBUyxDQUFDLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztnQkFFNUIsT0FBTztnQkFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtvQkFDbkUsWUFBWSxFQUFFLFlBQVk7aUJBQzNCLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO1lBQ25DLE9BQU87WUFDUCxTQUFTLEVBQUUsQ0FBQztZQUVaLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtnQkFDbkUsVUFBVSxFQUFFLEtBQUs7YUFDbEIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO1lBQ3ZELE9BQU87WUFDUCxTQUFTLEVBQUUsQ0FBQztZQUVaLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtnQkFDbkUsb0JBQW9CLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3BDLEVBQUUsSUFBSSxFQUFFLGtCQUFrQixFQUFFO2lCQUM3QixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO1lBQ3BDLE9BQU87WUFDUCxNQUFNLEdBQUcsR0FBRyxTQUFTLEVBQUUsQ0FBQztZQUV4QixPQUFPO1lBQ1AscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMscUJBQXFCLENBQUMsbUJBQW1CLEVBQUU7Z0JBQ25FLGNBQWMsRUFBRSxFQUFFLEdBQUcsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDLEVBQUU7YUFDeEcsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsMkNBQTJDLEVBQUUsR0FBRyxFQUFFO1lBQ3JELE9BQU87WUFDUCxTQUFTLEVBQUUsQ0FBQztZQUVaLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRTtnQkFDbkUsdUJBQXVCLEVBQUU7b0JBQ3ZCLGNBQWMsRUFBRSxHQUFHO29CQUNuQixxQkFBcUIsRUFBRSxDQUFDO2lCQUN6QjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO1FBQ3pDLElBQUksQ0FBQyw2Q0FBNkMsRUFBRSxHQUFHLEVBQUU7WUFDdkQsT0FBTztZQUNQLFNBQVMsRUFBRSxDQUFDO1lBRVosT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLDBCQUEwQixFQUFFO2dCQUMxRSxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsSUFBSSxFQUFFLDJCQUEyQjtxQkFDbEM7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7WUFDekMsT0FBTztZQUNQLFNBQVMsRUFBRSxDQUFDO1lBRVosT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLDBCQUEwQixFQUFFO2dCQUMxRSxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsU0FBUyxFQUFFLElBQUk7cUJBQ2hCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO1lBQ2xDLE9BQU87WUFDUCxTQUFTLEVBQUUsQ0FBQztZQUVaLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQywwQkFBMEIsRUFBRTtnQkFDMUUsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLE9BQU8sRUFBRTs0QkFDUDtnQ0FDRSxTQUFTLEVBQUUsTUFBTTtnQ0FDakIsSUFBSSxFQUFFLFFBQVE7Z0NBQ2QsU0FBUyxFQUFFLE1BQU07NkJBQ2xCOzRCQUNEO2dDQUNFLFNBQVMsRUFBRSxLQUFLO2dDQUNoQixJQUFJLEVBQUUsT0FBTztnQ0FDYixTQUFTLEVBQUUsS0FBSzs2QkFDakI7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7WUFDbkMsT0FBTztZQUNQLFNBQVMsRUFBRSxDQUFDO1lBRVosT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLDBCQUEwQixFQUFFO2dCQUMxRSxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsZ0JBQWdCLEVBQUU7NEJBQ2hCLFNBQVMsRUFBRSxTQUFTOzRCQUNwQixPQUFPLEVBQUU7Z0NBQ1AsZUFBZSxFQUFFLEVBQUU7Z0NBQ25CLHVCQUF1QixFQUFFLGtCQUFrQjtnQ0FDM0MsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLE1BQU07NkJBQzdCO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO1lBQ3ZDLFFBQVE7WUFDUixNQUFNLEdBQUcsR0FBRyxTQUFTLEVBQUUsQ0FBQztZQUV4QixPQUFPO1lBQ1AsTUFBTSxpQkFBaUIsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxZQUEwQixDQUFDLENBQUM7WUFFMUgsT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLDBCQUEwQixFQUFFO2dCQUMxRSxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsV0FBVyxFQUFFLGtCQUFLLENBQUMsU0FBUyxDQUFDOzRCQUMzQjtnQ0FDRSxJQUFJLEVBQUUsc0JBQXNCO2dDQUM1QixLQUFLLEVBQUUsaUJBQWlCLENBQUMsU0FBUzs2QkFDbkM7eUJBQ0YsQ0FBQztxQkFDSDtpQkFDRjtnQkFDRCxXQUFXLEVBQUU7b0JBQ1gsWUFBWSxFQUFFO3dCQUNaLGlCQUFpQjt3QkFDakIsS0FBSztxQkFDTjtpQkFDRjthQUNGLENBQUMsQ0FBQztZQUVILHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO2dCQUNsRSxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFLGtCQUFLLENBQUMsU0FBUyxDQUFDO3dCQUN6Qjs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sK0JBQStCO2dDQUMvQiwrQkFBK0I7NkJBQ2hDOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxpQkFBaUIsQ0FBQyxTQUFTO3lCQUN0QztxQkFDRixDQUFDO29CQUNGLE9BQU8sRUFBRSxZQUFZO2lCQUN0QjtnQkFDRCxLQUFLLEVBQUU7b0JBQ0wsRUFBRSxHQUFHLEVBQUUsbUJBQUssQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxFQUFFO2lCQUN6RzthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHNCQUFzQixFQUFFLEdBQUcsRUFBRTtZQUNoQyxPQUFPO1lBQ1AsU0FBUyxFQUFFLENBQUM7WUFFWixPQUFPO1lBQ1AscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMscUJBQXFCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQzFFLFdBQVcsRUFBRSxNQUFNO2FBQ3BCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtZQUN6QyxPQUFPO1lBQ1AsU0FBUyxFQUFFLENBQUM7WUFFWixPQUFPO1lBQ1AscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMscUJBQXFCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQzFFLHVCQUF1QixFQUFFLENBQUUsS0FBSyxDQUFFO2FBQ25DLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO1FBQzlDLFFBQVE7UUFDUixNQUFNLGVBQWUsR0FBRyxDQUFDLGVBQWUsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUMzRCxNQUFNLGlCQUFpQixHQUFHLGFBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLEVBQUUsU0FBUyxFQUFFO1lBQzFFLGlCQUFpQixFQUFFLENBQUMsWUFBWSxFQUFFLFlBQVksQ0FBQztZQUMvQyxLQUFLLEVBQUUsT0FBTztZQUNkLGVBQWU7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLFNBQVMsQ0FBQztZQUNSLEdBQUcsRUFBRSxpQkFBaUI7WUFDdEIsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLG9CQUFVLENBQUMsTUFBTSxFQUFFO1NBQzlDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxvQ0FBb0MsRUFBRTtZQUNwRixpQkFBaUIsRUFBRSxlQUFlO1NBQ25DLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztRQUNSLGNBQWM7UUFDZCxFQUFFO0tBQ0gsQ0FBQyxDQUFDLDhEQUE4RCxFQUFFLENBQUMsVUFBa0IsRUFBRSxFQUFFO1FBQ3hGLFFBQVE7UUFDUixNQUFNLEVBQUUsR0FBRyxvQkFBb0IsQ0FBQztRQUVoQyxPQUFPO1FBQ1AsU0FBUyxDQUFDO1lBQ1IsYUFBYSxFQUFFO2dCQUNiLGNBQWMsRUFBRSxVQUFVO2FBQzNCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLHNCQUFzQixFQUFFO1lBQ3RFLFlBQVksRUFBRSxVQUFVLEdBQUcsRUFBRTtTQUM5QixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUU7UUFDOUIsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtZQUN6QyxPQUFPO1lBQ1AsU0FBUyxDQUFDO2dCQUNSLFFBQVEsRUFBRTtvQkFDUix1QkFBaUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3Qix1QkFBaUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2lCQUM5QjthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQywwQkFBMEIsRUFBRTtnQkFDMUUsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLFdBQVcsRUFBRSxrQkFBSyxDQUFDLFNBQVMsQ0FBQzs0QkFDM0I7Z0NBQ0UsSUFBSSxFQUFFLFlBQVk7Z0NBQ2xCLEtBQUssRUFBRSxpQkFBaUI7NkJBQ3pCO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLENBQUM7WUFDUixDQUFDLFFBQVEsRUFBRSx1QkFBaUIsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNwRCxDQUFDLFFBQVEsRUFBRSx1QkFBaUIsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDekQsQ0FBQyxVQUFVLEVBQUUsdUJBQWlCLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQzdELENBQUMsVUFBVSxFQUFFLHVCQUFpQixDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3hELENBQUMsU0FBUyxFQUFFLHVCQUFpQixDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3JELENBQUMsUUFBUSxFQUFFLHVCQUFpQixDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztZQUN6RCxDQUFDLFNBQVMsRUFBRSx1QkFBaUIsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDNUQsQ0FBQyxVQUFVLEVBQUUsdUJBQWlCLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQzlELENBQUMsUUFBUSxFQUFFLHVCQUFpQixDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ25ELENBQUMsU0FBUyxFQUFFLHVCQUFpQixDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztZQUMzRCxDQUFDLE1BQU0sRUFBRSx1QkFBaUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDdEQsQ0FBQyxNQUFNLEVBQUUsdUJBQWlCLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQ3JELENBQUMsVUFBVSxFQUFFLHVCQUFpQixDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztZQUM3RCxDQUFDLFVBQVUsRUFBRSx1QkFBaUIsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDN0QsQ0FBQyxNQUFNLEVBQUUsdUJBQWlCLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDaEQsQ0FBQyxNQUFNLEVBQUUsdUJBQWlCLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQ3RELENBQUMsQ0FBQywrQkFBK0IsRUFBRSxDQUFDLFlBQW9CLEVBQUUsT0FBMEIsRUFBRSxLQUFlLEVBQUUsRUFBRTtZQUN4RyxRQUFRO1lBQ1IsTUFBTSxHQUFHLEdBQUcsU0FBUyxFQUFFLENBQUM7WUFDeEIsTUFBTSxXQUFXLEdBQUcsSUFBSSxtQkFBSyxDQUFDLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBQzNELFdBQVcsR0FBRyxJQUFJLHlCQUFtQixDQUFDLFdBQVcsRUFBRSxhQUFhLEVBQUU7Z0JBQ2hFLEdBQUc7Z0JBQ0gsa0JBQWtCLEVBQUUsSUFBSSw2QkFBbUIsQ0FBQztvQkFDMUMsV0FBVyxFQUFFLFNBQVM7aUJBQ3ZCLENBQUM7Z0JBQ0YsV0FBVztnQkFDWCxhQUFhLEVBQUUsdUJBQWEsQ0FBQyxtQkFBbUIsQ0FBQyxXQUFXLEVBQUUsSUFBSSxFQUFFLGNBQWMsRUFBRTtvQkFDbEYsZ0JBQWdCLEVBQUUsS0FBSztpQkFDeEIsQ0FBQzthQUNILENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxHQUFHLENBQUMsZUFBZSxDQUFDLFdBQVcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFFNUMsT0FBTztZQUNQLEtBQUssQ0FBQyxPQUFPLENBQUUsSUFBSSxDQUFDLEVBQUU7Z0JBQ3BCLHFCQUFRLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxDQUFDLHFCQUFxQixDQUFDLGdDQUFnQyxFQUFFO29CQUN0RixVQUFVLEVBQUUsS0FBSztvQkFDakIsTUFBTSxFQUFFLElBQUk7b0JBQ1osT0FBTyxFQUFFO3dCQUNQLGlCQUFpQixFQUFFLGtCQUFLLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxTQUFTLDREQUE0RCxDQUFDO3FCQUNsSTtvQkFDRCxxQkFBcUIsRUFBRSxjQUFjO2lCQUN0QyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEdBQUcsRUFBRTtZQUM1Qyx1QkFBdUI7WUFDdkIsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDVixTQUFTLENBQUMsRUFBRSxRQUFRLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUM5QixDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsOERBQThELENBQUMsQ0FBQztRQUNsRixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHlCQUF5QixFQUFFLEdBQUcsRUFBRTtRQUN2QyxJQUFJLENBQUMsbUVBQW1FLEVBQUUsR0FBRyxFQUFFO1lBQzdFLFFBQVE7WUFDUixNQUFNLHdCQUF3QixHQUFHLFdBQVcsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTNFLE9BQU87WUFDUCxNQUFNLEdBQUcsR0FBRyxTQUFTLEVBQUUsQ0FBQztZQUN4QixNQUFNLGdCQUFnQixHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTNELHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLGdDQUFnQyxFQUFFO2dCQUNoRixVQUFVLEVBQUUsS0FBSztnQkFDakIsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsTUFBTSxFQUFFLElBQUk7Z0JBQ1osT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsd0JBQXdCLENBQUMsZUFBZSxDQUFDO2dCQUNoRSxxQkFBcUIsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLGVBQWUsQ0FBQzthQUN2RSxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxtRUFBbUUsRUFBRSxHQUFHLEVBQUU7WUFDN0UsUUFBUTtZQUNSLE1BQU0sd0JBQXdCLEdBQUcsV0FBVyxDQUFDLGtCQUFrQixDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUVsRixPQUFPO1lBQ1AsTUFBTSxHQUFHLEdBQUcsU0FBUyxFQUFFLENBQUM7WUFDeEIsTUFBTSxnQkFBZ0IsR0FBRyxHQUFHLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUUzRCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDaEYsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLFFBQVEsRUFBRSxLQUFLO2dCQUNmLE1BQU0sRUFBRSxLQUFLO2dCQUNiLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLGVBQWUsQ0FBQztnQkFDeEQscUJBQXFCLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxlQUFlLENBQUM7YUFDL0UsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsNENBQTRDLEVBQUUsR0FBRyxFQUFFO1lBQ3RELE9BQU87WUFDUCxTQUFTLEVBQUUsQ0FBQztZQUVaLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQywwQkFBMEIsRUFBRTtnQkFDMUUsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLFdBQVcsRUFBRSxrQkFBSyxDQUFDLFNBQVMsQ0FBQzs0QkFDM0I7Z0NBQ0UsSUFBSSxFQUFFLGtCQUFrQjtnQ0FDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxXQUFXLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLFdBQVcsRUFBRSxNQUFNLFdBQVcsQ0FBQyxRQUFRLENBQUMsYUFBYSxFQUFFLENBQUM7NkJBQzFIO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7UUFDdkIsK0JBQWlCLENBQUM7WUFDaEIsYUFBYSxFQUFFLHFCQUFxQjtZQUNwQyxlQUFlLEVBQUUsR0FBRyxFQUFFO2dCQUNwQixTQUFTLEVBQUUsQ0FBQztnQkFDWixPQUFPLEtBQUssQ0FBQztZQUNmLENBQUM7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsbUJBQW1CLEVBQUUsQ0FBQztnQkFDdEIseUJBQXlCLEVBQUUsQ0FBQztnQkFDNUIsZ0JBQWdCLEVBQUUsQ0FBQztnQkFDbkIsb0NBQW9DLEVBQUUsQ0FBQztnQkFDdkMsdUJBQXVCLEVBQUUsQ0FBQztnQkFDMUIsaUJBQWlCLEVBQUUsQ0FBQztnQkFDcEIsMEJBQTBCLEVBQUUsQ0FBQztnQkFDN0IsbUJBQW1CLEVBQUUsQ0FBQzthQUN2QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQge1xuICBBcHAsXG4gIENmbkVsZW1lbnQsXG4gIFN0YWNrLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQge1xuICBBbm5vdGF0aW9ucyxcbiAgTWF0Y2gsXG4gIFRlbXBsYXRlLFxufSBmcm9tICdhd3MtY2RrLWxpYi9hc3NlcnRpb25zJztcbmltcG9ydCB7XG4gIEdlbmVyaWNXaW5kb3dzSW1hZ2UsXG4gIElWcGMsXG4gIFNlY3VyaXR5R3JvdXAsXG4gIFN1Ym5ldFNlbGVjdGlvbixcbiAgU3VibmV0VHlwZSxcbiAgVnBjLFxufSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCB7XG4gIERvY2tlckltYWdlQXNzZXQsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3ItYXNzZXRzJztcbmltcG9ydCB7XG4gIENmblNlcnZpY2UsXG4gIENvbnRhaW5lckltYWdlLFxufSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWNzJztcbmltcG9ydCB7XG4gIElMb2dHcm91cCxcbn0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0IHtcbiAgSVNlY3JldCxcbiAgU2VjcmV0LFxufSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc2VjcmV0c21hbmFnZXInO1xuXG5pbXBvcnQge1xuICB0ZXN0Q29uc3RydWN0VGFncyxcbn0gZnJvbSAnLi4vLi4vY29yZS90ZXN0L3RhZy1oZWxwZXJzJztcbmltcG9ydCB7XG4gIElWZXJzaW9uLFxuICBJV29ya2VyRmxlZXQsXG4gIFJlbmRlclF1ZXVlLFxuICBSZXBvc2l0b3J5LFxuICBTZWNyZXRzTWFuYWdlbWVudFJlZ2lzdHJhdGlvblN0YXR1cyxcbiAgU2VjcmV0c01hbmFnZW1lbnRSb2xlLFxuICBTdWJuZXRJZGVudGl0eVJlZ2lzdHJhdGlvblNldHRpbmdzUHJvcHMsXG4gIFVzYWdlQmFzZWRMaWNlbnNlLFxuICBVc2FnZUJhc2VkTGljZW5zaW5nLFxuICBVc2FnZUJhc2VkTGljZW5zaW5nSW1hZ2VzLFxuICBVc2FnZUJhc2VkTGljZW5zaW5nUHJvcHMsXG4gIFZlcnNpb25RdWVyeSxcbiAgV29ya2VySW5zdGFuY2VGbGVldCxcbn0gZnJvbSAnLi4vbGliJztcblxuY29uc3QgZW52ID0ge1xuICByZWdpb246ICd1cy1lYXN0LTEnLFxufTtcbmxldCBhcHA6IEFwcDtcbmxldCBjZXJ0aWZpY2F0ZVNlY3JldDogSVNlY3JldDtcbmxldCB2ZXJzaW9uZWRJbnN0YWxsZXJzOiBJVmVyc2lvbjtcbmxldCBkZXBlbmRlbmN5U3RhY2s6IFN0YWNrO1xubGV0IGRvY2tlckNvbnRhaW5lcjogRG9ja2VySW1hZ2VBc3NldDtcbmxldCBpbWFnZXM6IFVzYWdlQmFzZWRMaWNlbnNpbmdJbWFnZXM7XG5sZXQgbGljZW5zZXM6IFVzYWdlQmFzZWRMaWNlbnNlW107XG5sZXQgcmNzSW1hZ2U6IENvbnRhaW5lckltYWdlO1xubGV0IHJlbmRlclF1ZXVlOiBSZW5kZXJRdWV1ZTtcbmxldCBzdGFjazogU3RhY2s7XG5sZXQgdnBjOiBJVnBjO1xubGV0IHdvcmtlckZsZWV0OiBJV29ya2VyRmxlZXQ7XG5cbmNvbnN0IERFRkFVTFRfQ09OU1RSVUNUX0lEID0gJ1VCTCc7XG5cbmRlc2NyaWJlKCdVc2FnZUJhc2VkTGljZW5zaW5nJywgKCkgPT4ge1xuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGFwcCA9IG5ldyBBcHAoKTtcblxuICAgIGRlcGVuZGVuY3lTdGFjayA9IG5ldyBTdGFjayhhcHAsICdEZXBlbmRlbmN5U3RhY2snLCB7IGVudiB9KTtcblxuICAgIHZlcnNpb25lZEluc3RhbGxlcnMgPSBuZXcgVmVyc2lvblF1ZXJ5KGRlcGVuZGVuY3lTdGFjaywgJ1ZlcnNpb25RdWVyeScpO1xuXG4gICAgdnBjID0gbmV3IFZwYyhkZXBlbmRlbmN5U3RhY2ssICdWUEMnKTtcbiAgICByY3NJbWFnZSA9IENvbnRhaW5lckltYWdlLmZyb21Eb2NrZXJJbWFnZUFzc2V0KG5ldyBEb2NrZXJJbWFnZUFzc2V0KGRlcGVuZGVuY3lTdGFjaywgJ0ltYWdlJywge1xuICAgICAgZGlyZWN0b3J5OiBfX2Rpcm5hbWUsXG4gICAgfSkpO1xuICAgIHJlbmRlclF1ZXVlID0gbmV3IFJlbmRlclF1ZXVlKGRlcGVuZGVuY3lTdGFjaywgJ1JRLU5vbkRlZmF1bHRQb3J0Jywge1xuICAgICAgdnBjLFxuICAgICAgaW1hZ2VzOiB7IHJlbW90ZUNvbm5lY3Rpb25TZXJ2ZXI6IHJjc0ltYWdlIH0sXG4gICAgICByZXBvc2l0b3J5OiBuZXcgUmVwb3NpdG9yeShkZXBlbmRlbmN5U3RhY2ssICdSZXBvc2l0b3J5Tm9uRGVmYXVsdCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICB2ZXJzaW9uOiB2ZXJzaW9uZWRJbnN0YWxsZXJzLFxuICAgICAgfSksXG4gICAgICB2ZXJzaW9uOiB2ZXJzaW9uZWRJbnN0YWxsZXJzLFxuICAgIH0pO1xuICAgIGplc3Quc3B5T24ocmVuZGVyUXVldWUsICdjb25maWd1cmVTZWNyZXRzTWFuYWdlbWVudEF1dG9SZWdpc3RyYXRpb24nKTtcblxuICAgIHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJywgeyBlbnYgfSk7XG4gICAgY2VydGlmaWNhdGVTZWNyZXQgPSBTZWNyZXQuZnJvbVNlY3JldENvbXBsZXRlQXJuKHN0YWNrLCAnQ2VydFNlY3JldCcsICdhcm46YXdzOnNlY3JldHNtYW5hZ2VyOnVzLXdlc3QtMjo2NzU4NzI3MDAzNTU6c2VjcmV0OkNlcnRTZWNyZXQtajFraUZ6Jyk7XG4gICAgZG9ja2VyQ29udGFpbmVyID0gbmV3ICBEb2NrZXJJbWFnZUFzc2V0KHN0YWNrLCAnbGljZW5zZS1mb3J3YXJkZXInLCB7XG4gICAgICBkaXJlY3Rvcnk6IF9fZGlybmFtZSxcbiAgICB9KTtcbiAgICBpbWFnZXMgPSB7XG4gICAgICBsaWNlbnNlRm9yd2FyZGVyOiBDb250YWluZXJJbWFnZS5mcm9tRG9ja2VySW1hZ2VBc3NldChkb2NrZXJDb250YWluZXIpLFxuICAgIH07XG4gICAgbGljZW5zZXMgPSBbVXNhZ2VCYXNlZExpY2Vuc2UuZm9yTWF5YSgpXTtcbiAgfSk7XG5cbiAgZnVuY3Rpb24gY3JlYXRlVWJsKHByb3BzPzogUGFydGlhbDxVc2FnZUJhc2VkTGljZW5zaW5nUHJvcHM+KTogVXNhZ2VCYXNlZExpY2Vuc2luZyB7XG4gICAgcmV0dXJuIG5ldyBVc2FnZUJhc2VkTGljZW5zaW5nKHN0YWNrLCBERUZBVUxUX0NPTlNUUlVDVF9JRCwge1xuICAgICAgY2VydGlmaWNhdGVTZWNyZXQsXG4gICAgICBpbWFnZXMsXG4gICAgICBsaWNlbnNlcyxcbiAgICAgIHJlbmRlclF1ZXVlLFxuICAgICAgdnBjLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICB0ZXN0KCd2cGNTdWJuZXRzIHNwZWNpZmllZCA9PiBkb2VzIG5vdCBlbWl0IHdhcm5pbmdzJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgdnBjU3VibmV0czogU3VibmV0U2VsZWN0aW9uID0ge1xuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfRUdSRVNTLFxuICAgIH07XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdWJsID0gY3JlYXRlVWJsKHtcbiAgICAgIHZwY1N1Ym5ldHMsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgQW5ub3RhdGlvbnMuZnJvbVN0YWNrKHN0YWNrKS5oYXNOb0luZm8oYC8ke3VibC5ub2RlLnBhdGh9YCwgTWF0Y2guYW55VmFsdWUoKSk7XG4gICAgQW5ub3RhdGlvbnMuZnJvbVN0YWNrKHN0YWNrKS5oYXNOb1dhcm5pbmcoYC8ke3VibC5ub2RlLnBhdGh9YCwgTWF0Y2guYW55VmFsdWUoKSk7XG4gICAgQW5ub3RhdGlvbnMuZnJvbVN0YWNrKHN0YWNrKS5oYXNOb0Vycm9yKGAvJHt1Ymwubm9kZS5wYXRofWAsIE1hdGNoLmFueVZhbHVlKCkpO1xuICB9KTtcblxuICB0ZXN0KCd2cGNTdWJuZXRzIG5vdCBzcGVjaWZpZWQgPT4gZW1pdHMgd2FybmluZyBhYm91dCBkZWRpY2F0ZWQgc3VibmV0cycsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdWJsID0gY3JlYXRlVWJsKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgQW5ub3RhdGlvbnMuZnJvbVN0YWNrKHN0YWNrKS5oYXNXYXJuaW5nKFxuICAgICAgYC8ke3VibC5ub2RlLnBhdGh9YCxcbiAgICAgICdEZWFkbGluZSBTZWNyZXRzIE1hbmFnZW1lbnQgaXMgZW5hYmxlZCBvbiB0aGUgUmVwb3NpdG9yeSBhbmQgVlBDIHN1Ym5ldHMgaGF2ZSBub3QgYmVlbiBzdXBwbGllZC4gVXNpbmcgZGVkaWNhdGVkIHN1Ym5ldHMgaXMgcmVjb21tZW5kZWQuIFNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1yZmRrL2Jsb2JzL3JlbGVhc2UvcGFja2FnZXMvYXdzLXJmZGsvbGliL2RlYWRsaW5lL1JFQURNRS5tZCN1c2luZy1kZWRpY2F0ZWQtc3VibmV0cy1mb3ItZGVhZGxpbmUtY29tcG9uZW50cycsXG4gICAgKTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2NvbmZpZ3VyZXMgYXV0byByZWdpc3RyYXRpb24nLCAoKSA9PiB7XG4gICAgdGVzdCgnZGVmYXVsdCB0byBwcml2YXRlIHN1Ym5ldHMnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCB1YmwgPSBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgY29uc3QgZXhwZWN0ZWRDYWxsOiBTdWJuZXRJZGVudGl0eVJlZ2lzdHJhdGlvblNldHRpbmdzUHJvcHMgPSB7XG4gICAgICAgIGRlcGVuZGVudDogdWJsLnNlcnZpY2Uubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuU2VydmljZSxcbiAgICAgICAgcmVnaXN0cmF0aW9uU3RhdHVzOiBTZWNyZXRzTWFuYWdlbWVudFJlZ2lzdHJhdGlvblN0YXR1cy5SRUdJU1RFUkVELFxuICAgICAgICByb2xlOiBTZWNyZXRzTWFuYWdlbWVudFJvbGUuQ0xJRU5ULFxuICAgICAgICB2cGMsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfRUdSRVNTIH0sXG4gICAgICB9O1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QocmVuZGVyUXVldWUuY29uZmlndXJlU2VjcmV0c01hbmFnZW1lbnRBdXRvUmVnaXN0cmF0aW9uKS50b0hhdmVCZWVuQ2FsbGVkV2l0aChleHBlY3RlZENhbGwpO1xuICAgIH0pO1xuXG4gICAgdGVzdC5lYWNoPFtTdWJuZXRTZWxlY3Rpb25dPihbXG4gICAgICBbe1xuICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyxcbiAgICAgIH1dLFxuICAgIF0pKCclcycsICh2cGNTdWJuZXRzKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCB1YmwgPSBjcmVhdGVVYmwoe1xuICAgICAgICB2cGNTdWJuZXRzLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGNvbnN0IGV4cGVjdGVkQ2FsbDogU3VibmV0SWRlbnRpdHlSZWdpc3RyYXRpb25TZXR0aW5nc1Byb3BzID0ge1xuICAgICAgICBkZXBlbmRlbnQ6IHVibC5zZXJ2aWNlLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmblNlcnZpY2UsXG4gICAgICAgIHJlZ2lzdHJhdGlvblN0YXR1czogU2VjcmV0c01hbmFnZW1lbnRSZWdpc3RyYXRpb25TdGF0dXMuUkVHSVNURVJFRCxcbiAgICAgICAgcm9sZTogU2VjcmV0c01hbmFnZW1lbnRSb2xlLkNMSUVOVCxcbiAgICAgICAgdnBjLFxuICAgICAgICB2cGNTdWJuZXRzLFxuICAgICAgfTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHJlbmRlclF1ZXVlLmNvbmZpZ3VyZVNlY3JldHNNYW5hZ2VtZW50QXV0b1JlZ2lzdHJhdGlvbikudG9IYXZlQmVlbkNhbGxlZFdpdGgoZXhwZWN0ZWRDYWxsKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlcyBhbiBFQ1MgY2x1c3RlcicsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY3JlYXRlVWJsKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5yZXNvdXJjZUNvdW50SXMoJ0FXUzo6RUNTOjpDbHVzdGVyJywgMSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdjcmVhdGVzIGFuIEFTRycsICgpID0+IHtcbiAgICB0ZXN0KCdkZWZhdWx0cycsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNyZWF0ZVVibCgpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICAgICAgTWluU2l6ZTogJzEnLFxuICAgICAgICBNYXhTaXplOiAnMScsXG4gICAgICAgIFZQQ1pvbmVJZGVudGlmaWVyOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0ZuOjpJbXBvcnRWYWx1ZSc6IE1hdGNoLnN0cmluZ0xpa2VSZWdleHAoYCR7ZGVwZW5kZW5jeVN0YWNrLnN0YWNrTmFtZX06RXhwb3J0c091dHB1dFJlZlZQQ1ByaXZhdGVTdWJuZXQxU3VibmV0LipgKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgICdGbjo6SW1wb3J0VmFsdWUnOiBNYXRjaC5zdHJpbmdMaWtlUmVnZXhwKGAke2RlcGVuZGVuY3lTdGFjay5zdGFja05hbWV9OkV4cG9ydHNPdXRwdXRSZWZWUENQcml2YXRlU3VibmV0MlN1Ym5ldC4qYCksXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnY2FwYWNpdHkgY2FuIGJlIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNyZWF0ZVVibCh7XG4gICAgICAgIGRlc2lyZWRDb3VudDogMixcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICAgICAgTWluU2l6ZTogJzInLFxuICAgICAgICBNYXhTaXplOiAnMicsXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2dpdmVzIHdyaXRlIGFjY2VzcyB0byBsb2cgZ3JvdXAnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgdWJsID0gY3JlYXRlVWJsKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IGxvZ0dyb3VwID0gdWJsLm5vZGUuZmluZENoaWxkKGAke0RFRkFVTFRfQ09OU1RSVUNUX0lEfUxvZ0dyb3VwYCkgYXMgSUxvZ0dyb3VwO1xuICAgICAgY29uc3QgYXNnUm9sZUxvZ2ljYWxJZCA9IFN0YWNrLm9mKHVibCkuZ2V0TG9naWNhbElkKHVibC5hc2cucm9sZS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBNYXRjaC5hcnJheVdpdGgoW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBBY3Rpb246IE1hdGNoLmFycmF5V2l0aChbXG4gICAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICAgICBdKSxcbiAgICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgICBSZXNvdXJjZTogc3RhY2sucmVzb2x2ZShsb2dHcm91cC5sb2dHcm91cEFybiksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0pLFxuICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgfSxcbiAgICAgICAgUm9sZXM6IE1hdGNoLmFycmF5V2l0aChbXG4gICAgICAgICAgeyBSZWY6IGFzZ1JvbGVMb2dpY2FsSWQgfSxcbiAgICAgICAgXSksXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgdGhlIHN1cHBsaWVkIHNlY3VyaXR5IGdyb3VwJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXAgPSBuZXcgU2VjdXJpdHlHcm91cChzdGFjaywgJ1VibFNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIHZwYyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoeyBzZWN1cml0eUdyb3VwIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBdXRvU2NhbGluZzo6TGF1bmNoQ29uZmlndXJhdGlvbicsIHtcbiAgICAgICAgU2VjdXJpdHlHcm91cHM6IE1hdGNoLmFycmF5V2l0aChbc3RhY2sucmVzb2x2ZShzZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCldKSxcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnY3JlYXRlcyBhbiBFQ1Mgc2VydmljZScsICgpID0+IHtcbiAgICB0ZXN0KCdhc3NvY2lhdGVkIHdpdGggdGhlIGNsdXN0ZXInLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCB1YmwgPSBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpTZXJ2aWNlJywge1xuICAgICAgICBDbHVzdGVyOiB7IFJlZjogc3RhY2suZ2V0TG9naWNhbElkKHVibC5jbHVzdGVyLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkVsZW1lbnQpIH0sXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCdEZXNpcmVkQ291bnQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdkZWZhdWx0cyB0byAxJywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNyZWF0ZVVibCgpO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpTZXJ2aWNlJywge1xuICAgICAgICAgIERlc2lyZWRDb3VudDogMSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnY2FuIGJlIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgZGVzaXJlZENvdW50ID0gMjtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNyZWF0ZVVibCh7IGRlc2lyZWRDb3VudCB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVDUzo6U2VydmljZScsIHtcbiAgICAgICAgICBEZXNpcmVkQ291bnQ6IGRlc2lyZWRDb3VudCxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3NldHMgbGF1bmNoIHR5cGUgdG8gRUMyJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY3JlYXRlVWJsKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVDUzo6U2VydmljZScsIHtcbiAgICAgICAgTGF1bmNoVHlwZTogJ0VDMicsXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3NldHMgZGlzdGluY3QgaW5zdGFuY2UgcGxhY2VtZW50IGNvbnN0cmFpbnQnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpTZXJ2aWNlJywge1xuICAgICAgICBQbGFjZW1lbnRDb25zdHJhaW50czogTWF0Y2guYXJyYXlXaXRoKFtcbiAgICAgICAgICB7IFR5cGU6ICdkaXN0aW5jdEluc3RhbmNlJyB9LFxuICAgICAgICBdKSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndXNlcyB0aGUgdGFzayBkZWZpbml0aW9uJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgdWJsID0gY3JlYXRlVWJsKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVDUzo6U2VydmljZScsIHtcbiAgICAgICAgVGFza0RlZmluaXRpb246IHsgUmVmOiBzdGFjay5nZXRMb2dpY2FsSWQodWJsLnNlcnZpY2UudGFza0RlZmluaXRpb24ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCkgfSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnd2l0aCB0aGUgY29ycmVjdCBkZXBsb3ltZW50IGNvbmZpZ3VyYXRpb24nLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpTZXJ2aWNlJywge1xuICAgICAgICBEZXBsb3ltZW50Q29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIE1heGltdW1QZXJjZW50OiAxMDAsXG4gICAgICAgICAgTWluaW11bUhlYWx0aHlQZXJjZW50OiAwLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdjcmVhdGVzIGEgdGFzayBkZWZpbml0aW9uJywgKCkgPT4ge1xuICAgIHRlc3QoJ2NvbnRhaW5lciBuYW1lIGlzIExpY2Vuc2VGb3J3YXJkZXJDb250YWluZXInLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnTGljZW5zZUZvcndhcmRlckNvbnRhaW5lcicsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnY29udGFpbmVyIGlzIG1hcmtlZCBlc3NlbnRpYWwnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFc3NlbnRpYWw6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnd2l0aCBpbmNyZWFzZWQgdWxpbWl0cycsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIGNyZWF0ZVVibCgpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFVsaW1pdHM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIEhhcmRMaW1pdDogMjAwMDAwLFxuICAgICAgICAgICAgICAgIE5hbWU6ICdub2ZpbGUnLFxuICAgICAgICAgICAgICAgIFNvZnRMaW1pdDogMjAwMDAwLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgSGFyZExpbWl0OiA2NDAwMCxcbiAgICAgICAgICAgICAgICBOYW1lOiAnbnByb2MnLFxuICAgICAgICAgICAgICAgIFNvZnRMaW1pdDogNjQwMDAsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3dpdGggYXdzbG9ncyBsb2cgZHJpdmVyJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY3JlYXRlVWJsKCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgTG9nQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgICBMb2dEcml2ZXI6ICdhd3Nsb2dzJyxcbiAgICAgICAgICAgICAgT3B0aW9uczoge1xuICAgICAgICAgICAgICAgICdhd3Nsb2dzLWdyb3VwJzoge30sXG4gICAgICAgICAgICAgICAgJ2F3c2xvZ3Mtc3RyZWFtLXByZWZpeCc6ICdMaWNlbnNlRm9yd2FyZGVyJyxcbiAgICAgICAgICAgICAgICAnYXdzbG9ncy1yZWdpb24nOiBlbnYucmVnaW9uLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjb25maWd1cmVzIFVCTCBjZXJ0aWZpY2F0ZXMnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgdWJsID0gY3JlYXRlVWJsKCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHRhc2tSb2xlTG9naWNhbElkID0gU3RhY2sub2YodWJsKS5nZXRMb2dpY2FsSWQodWJsLnNlcnZpY2UudGFza0RlZmluaXRpb24udGFza1JvbGUubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgRW52aXJvbm1lbnQ6IE1hdGNoLmFycmF5V2l0aChbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBOYW1lOiAnVUJMX0NFUlRJRklDQVRFU19VUkknLFxuICAgICAgICAgICAgICAgIFZhbHVlOiBjZXJ0aWZpY2F0ZVNlY3JldC5zZWNyZXRBcm4sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdKSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBUYXNrUm9sZUFybjoge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgdGFza1JvbGVMb2dpY2FsSWQsXG4gICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogTWF0Y2guYXJyYXlXaXRoKFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkdldFNlY3JldFZhbHVlJyxcbiAgICAgICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6RGVzY3JpYmVTZWNyZXQnLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFJlc291cmNlOiBjZXJ0aWZpY2F0ZVNlY3JldC5zZWNyZXRBcm4sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0pLFxuICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgfSxcbiAgICAgICAgUm9sZXM6IFtcbiAgICAgICAgICB7IFJlZjogU3RhY2sub2YodWJsKS5nZXRMb2dpY2FsSWQodWJsLnNlcnZpY2UudGFza0RlZmluaXRpb24udGFza1JvbGUubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCkgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndXNlcyBob3N0IG5ldHdvcmtpbmcnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgTmV0d29ya01vZGU6ICdob3N0JyxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnaXMgbWFya2VkIEVDMiBjb21wYXRpYmxlIG9ubHknLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgUmVxdWlyZXNDb21wYXRpYmlsaXRpZXM6IFsgJ0VDMicgXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdMaWNlbnNlIEZvcndhcmRlciBzdWJuZXQgc2VsZWN0aW9uJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgcHVibGljU3VibmV0SWRzID0gWydQdWJsaWNTdWJuZXQxJywgJ1B1YmxpY1N1Ym5ldDInXTtcbiAgICBjb25zdCB2cGNGcm9tQXR0cmlidXRlcyA9IFZwYy5mcm9tVnBjQXR0cmlidXRlcyhkZXBlbmRlbmN5U3RhY2ssICdBdHRyVnBjJywge1xuICAgICAgYXZhaWxhYmlsaXR5Wm9uZXM6IFsndXMtZWFzdC0xYScsICd1cy1lYXN0LTFiJ10sXG4gICAgICB2cGNJZDogJ3ZwY2lkJyxcbiAgICAgIHB1YmxpY1N1Ym5ldElkcyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjcmVhdGVVYmwoe1xuICAgICAgdnBjOiB2cGNGcm9tQXR0cmlidXRlcyxcbiAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICAgIFZQQ1pvbmVJZGVudGlmaWVyOiBwdWJsaWNTdWJuZXRJZHMsXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QuZWFjaChbXG4gICAgJ3Rlc3QtcHJlZml4LycsXG4gICAgJycsXG4gIF0pKCdMaWNlbnNlIEZvcndhcmRlciBpcyBjcmVhdGVkIHdpdGggY29ycmVjdCBMb2dHcm91cCBwcmVmaXggJXMnLCAodGVzdFByZWZpeDogc3RyaW5nKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBpZCA9IERFRkFVTFRfQ09OU1RSVUNUX0lEO1xuXG4gICAgLy8gV0hFTlxuICAgIGNyZWF0ZVVibCh7XG4gICAgICBsb2dHcm91cFByb3BzOiB7XG4gICAgICAgIGxvZ0dyb3VwUHJlZml4OiB0ZXN0UHJlZml4LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQ3VzdG9tOjpMb2dSZXRlbnRpb24nLCB7XG4gICAgICBMb2dHcm91cE5hbWU6IHRlc3RQcmVmaXggKyBpZCxcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2xpY2Vuc2UgbGltaXRzJywgKCkgPT4ge1xuICAgIHRlc3QoJ211bHRpcGxlIGxpY2Vuc2VzIHdpdGggbGltaXRzJywgKCkgPT4ge1xuICAgICAgLy8gV0hFTlxuICAgICAgY3JlYXRlVWJsKHtcbiAgICAgICAgbGljZW5zZXM6IFtcbiAgICAgICAgICBVc2FnZUJhc2VkTGljZW5zZS5mb3JNYXlhKDEwKSxcbiAgICAgICAgICBVc2FnZUJhc2VkTGljZW5zZS5mb3JWcmF5KDEwKSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEVudmlyb25tZW50OiBNYXRjaC5hcnJheVdpdGgoW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgTmFtZTogJ1VCTF9MSU1JVFMnLFxuICAgICAgICAgICAgICAgIFZhbHVlOiAnbWF5YToxMDt2cmF5OjEwJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0pLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QuZWFjaChbXG4gICAgICBbJzNkc01heCcsIFVzYWdlQmFzZWRMaWNlbnNlLmZvcjNkc01heCgxMCksIFsyNzAwMl1dLFxuICAgICAgWydBcm5vbGQnLCBVc2FnZUJhc2VkTGljZW5zZS5mb3JBcm5vbGQoMTApLCBbNTA1NiwgNzA1Nl1dLFxuICAgICAgWydDaW5lbWE0RCcsIFVzYWdlQmFzZWRMaWNlbnNlLmZvckNpbmVtYTREKDEwKSwgWzUwNTcsIDcwNTddXSxcbiAgICAgIFsnQ2xhcmlzc2UnLCBVc2FnZUJhc2VkTGljZW5zZS5mb3JDbGFyaXNzZSgxMCksIFs0MDUwMF1dLFxuICAgICAgWydIb3VkaW5pJywgVXNhZ2VCYXNlZExpY2Vuc2UuZm9ySG91ZGluaSgxMCksIFsxNzE1XV0sXG4gICAgICBbJ0thdGFuYScsIFVzYWdlQmFzZWRMaWNlbnNlLmZvckthdGFuYSgxMCksIFs0MTUxLCA2MTAxXV0sXG4gICAgICBbJ0tleVNob3QnLCBVc2FnZUJhc2VkTGljZW5zZS5mb3JLZXlTaG90KDEwKSwgWzI3MDAzLCAyNzAzXV0sXG4gICAgICBbJ0tyYWthdG9hJywgVXNhZ2VCYXNlZExpY2Vuc2UuZm9yS3Jha2F0b2EoMTApLCBbMjcwMDAsIDI3MDBdXSxcbiAgICAgIFsnTWFudHJhJywgVXNhZ2VCYXNlZExpY2Vuc2UuZm9yTWFudHJhKDEwKSwgWzE3MTZdXSxcbiAgICAgIFsnTWF4d2VsbCcsIFVzYWdlQmFzZWRMaWNlbnNlLmZvck1heHdlbGwoMTApLCBbNTU1NSwgNzA1NV1dLFxuICAgICAgWydNYXlhJywgVXNhZ2VCYXNlZExpY2Vuc2UuZm9yTWF5YSgxMCksIFsyNzAwMiwgMjcwMl1dLFxuICAgICAgWydOdWtlJywgVXNhZ2VCYXNlZExpY2Vuc2UuZm9yTnVrZSgxMCksIFs0MTAxLCA2MTAxXV0sXG4gICAgICBbJ1JlYWxGbG93JywgVXNhZ2VCYXNlZExpY2Vuc2UuZm9yUmVhbEZsb3coMTApLCBbNTA1NSwgNzA1NV1dLFxuICAgICAgWydSZWRTaGlmdCcsIFVzYWdlQmFzZWRMaWNlbnNlLmZvclJlZFNoaWZ0KDEwKSwgWzUwNTQsIDcwNTRdXSxcbiAgICAgIFsnVnJheScsIFVzYWdlQmFzZWRMaWNlbnNlLmZvclZyYXkoMTApLCBbMzAzMDZdXSxcbiAgICAgIFsnWWV0aScsIFVzYWdlQmFzZWRMaWNlbnNlLmZvcllldGkoMTApLCBbNTA1MywgNzA1M11dLFxuICAgIF0pKCdvcGVuIHBvcnQgZm9yIGxpY2Vuc2UgdHlwZSAlcycsIChfbGljZW5zZU5hbWU6IHN0cmluZywgbGljZW5zZTogVXNhZ2VCYXNlZExpY2Vuc2UsIHBvcnRzOiBudW1iZXJbXSkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHVibCA9IGNyZWF0ZVVibCgpO1xuICAgICAgY29uc3Qgd29ya2VyU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnV29ya2VyU3RhY2snLCB7IGVudiB9KTtcbiAgICAgIHdvcmtlckZsZWV0ID0gbmV3IFdvcmtlckluc3RhbmNlRmxlZXQod29ya2VyU3RhY2ssICd3b3JrZXJGbGVldCcsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICB3b3JrZXJNYWNoaW5lSW1hZ2U6IG5ldyBHZW5lcmljV2luZG93c0ltYWdlKHtcbiAgICAgICAgICAndXMtZWFzdC0xJzogJ2FtaS1hbnknLFxuICAgICAgICB9KSxcbiAgICAgICAgcmVuZGVyUXVldWUsXG4gICAgICAgIHNlY3VyaXR5R3JvdXA6IFNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZCh3b3JrZXJTdGFjaywgJ1NHJywgJ3NnLTEyMzQ1Njc4OScsIHtcbiAgICAgICAgICBhbGxvd0FsbE91dGJvdW5kOiBmYWxzZSxcbiAgICAgICAgfSksXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgdWJsLmdyYW50UG9ydEFjY2Vzcyh3b3JrZXJGbGVldCwgW2xpY2Vuc2VdKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgcG9ydHMuZm9yRWFjaCggcG9ydCA9PiB7XG4gICAgICAgIFRlbXBsYXRlLmZyb21TdGFjayh3b3JrZXJTdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgICAgVG9Qb3J0OiBwb3J0LFxuICAgICAgICAgIEdyb3VwSWQ6IHtcbiAgICAgICAgICAgICdGbjo6SW1wb3J0VmFsdWUnOiBNYXRjaC5zdHJpbmdMaWtlUmVnZXhwKGAke1N0YWNrLm9mKHVibCkuc3RhY2tOYW1lfTpFeHBvcnRzT3V0cHV0Rm5HZXRBdHRVQkxDbHVzdGVyQVNHSW5zdGFuY2VTZWN1cml0eUdyb3VwLipgKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIFNvdXJjZVNlY3VyaXR5R3JvdXBJZDogJ3NnLTEyMzQ1Njc4OScsXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdyZXF1aXJlcyBvbmUgdXNhZ2UgYmFzZWQgbGljZW5zZScsICgpID0+IHtcbiAgICAgIC8vIFdpdGhvdXQgYW55IGxpY2Vuc2VzXG4gICAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgICBjcmVhdGVVYmwoeyBsaWNlbnNlczogW10gfSk7XG4gICAgICB9KS50b1Rocm93RXJyb3IoJ1Nob3VsZCBiZSBzcGVjaWZpZWQgYXQgbGVhc3Qgb25lIGxpY2Vuc2Ugd2l0aCBkZWZpbmVkIGxpbWl0LicpO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnY29uZmlndXJlcyByZW5kZXIgcXVldWUnLCAoKSA9PiB7XG4gICAgdGVzdCgnYWRkcyBpbmdyZXNzIHJ1bGUgZnJvbSBVc2FnZUJhc2VkTGljZW5zaW5nIEFTRyB0byBSZW5kZXJRdWV1ZSBBU0cnLCAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgcmVuZGVyUXVldWVTZWN1cml0eUdyb3VwID0gcmVuZGVyUXVldWUuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHNbMF07XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnN0IHVibCA9IGNyZWF0ZVVibCgpO1xuICAgICAgY29uc3QgdWJsU2VjdXJpdHlHcm91cCA9IHVibC5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXTtcblxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICAgIEZyb21Qb3J0OiA0NDMzLFxuICAgICAgICBUb1BvcnQ6IDQ0MzMsXG4gICAgICAgIEdyb3VwSWQ6IHN0YWNrLnJlc29sdmUocmVuZGVyUXVldWVTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICAgIFNvdXJjZVNlY3VyaXR5R3JvdXBJZDogc3RhY2sucmVzb2x2ZSh1YmxTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2FkZHMgaW5ncmVzcyBydWxlIGZyb20gUmVuZGVyUXVldWUgQVNHIHRvIFVzYWdlQmFzZWRMaWNlbnNpbmcgQVNHJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHJlbmRlclF1ZXVlU2VjdXJpdHlHcm91cCA9IHJlbmRlclF1ZXVlLmJhY2tlbmRDb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwc1swXTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgdWJsID0gY3JlYXRlVWJsKCk7XG4gICAgICBjb25zdCB1YmxTZWN1cml0eUdyb3VwID0gdWJsLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzWzBdO1xuXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICAgICAgRnJvbVBvcnQ6IDE3MDA0LFxuICAgICAgICBUb1BvcnQ6IDE3MDA0LFxuICAgICAgICBHcm91cElkOiBzdGFjay5yZXNvbHZlKHVibFNlY3VyaXR5R3JvdXAuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgICAgU291cmNlU2VjdXJpdHlHcm91cElkOiBzdGFjay5yZXNvbHZlKHJlbmRlclF1ZXVlU2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQpLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzZXRzIFJFTkRFUl9RVUVVRV9VUkkgZW52aXJvbm1lbnQgdmFyaWFibGUnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjcmVhdGVVYmwoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFbnZpcm9ubWVudDogTWF0Y2guYXJyYXlXaXRoKFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIE5hbWU6ICdSRU5ERVJfUVVFVUVfVVJJJyxcbiAgICAgICAgICAgICAgICBWYWx1ZTogc3RhY2sucmVzb2x2ZShgJHtyZW5kZXJRdWV1ZS5lbmRwb2ludC5hcHBsaWNhdGlvblByb3RvY29sLnRvTG93ZXJDYXNlKCl9Oi8vJHtyZW5kZXJRdWV1ZS5lbmRwb2ludC5zb2NrZXRBZGRyZXNzfWApLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSksXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgndGFnZ2luZycsICgpID0+IHtcbiAgICB0ZXN0Q29uc3RydWN0VGFncyh7XG4gICAgICBjb25zdHJ1Y3ROYW1lOiAnVXNhZ2VCYXNlZExpY2Vuc2luZycsXG4gICAgICBjcmVhdGVDb25zdHJ1Y3Q6ICgpID0+IHtcbiAgICAgICAgY3JlYXRlVWJsKCk7XG4gICAgICAgIHJldHVybiBzdGFjaztcbiAgICAgIH0sXG4gICAgICByZXNvdXJjZVR5cGVDb3VudHM6IHtcbiAgICAgICAgJ0FXUzo6RUNTOjpDbHVzdGVyJzogMSxcbiAgICAgICAgJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJzogMSxcbiAgICAgICAgJ0FXUzo6SUFNOjpSb2xlJzogNSxcbiAgICAgICAgJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnOiAxLFxuICAgICAgICAnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJzogMSxcbiAgICAgICAgJ0FXUzo6U05TOjpUb3BpYyc6IDEsXG4gICAgICAgICdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nOiAxLFxuICAgICAgICAnQVdTOjpFQ1M6OlNlcnZpY2UnOiAxLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG59KTtcbiJdfQ==