"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let stage;
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    const MAJOR_VERSION = 10;
    const MINOR_VERSION = 1;
    const RELEASE_VERSION = 9;
    const PATCH_VERSION = 2;
    const RELEASE_VERSION_STRING = `${MAJOR_VERSION}.${MINOR_VERSION}.${RELEASE_VERSION}`;
    const FULL_VERSION_STRING = `${RELEASE_VERSION_STRING}.${PATCH_VERSION}`;
    const RCS_RECIPE_NAME = 'rcs';
    const RCS_RECIPE = {
        description: 'rcs',
        title: 'rcs',
        buildArgs: {
            DL_VERSION: FULL_VERSION_STRING,
            a: 'a',
            b: 'b',
        },
        target: undefined,
    };
    const LICENSE_FORWARDER_RECIPE_NAME = 'license-forwarder';
    const LICENSE_FORWARDER_RECIPE = {
        title: 'license-forwarder',
        description: 'license-forwarder',
        buildArgs: {
            DL_VERSION: FULL_VERSION_STRING,
            c: 'c',
            d: 'd',
        },
        target: 'lf',
    };
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        class TestStage extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        stage = new TestStage({
            path: STAGE_PATH,
            manifest: {
                schema: 1,
                version: FULL_VERSION_STRING,
                recipes: {
                    [RCS_RECIPE_NAME]: RCS_RECIPE,
                    [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
                },
            },
        });
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
    });
    /**
     * Tests that the remote connection server and license forwarder Docker recipes
     * create the correct {@link DockerImageAsset} instances using the build arguments,
     * and target from the supplied manifest.
     *
     * This is done by comparing the `assetHash` property of the expected vs actual
     * DockerImageAsset instances.
     */
    test.each([
        [
            'remoteConnnectionServer',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: RCS_RECIPE.buildArgs,
                    target: RCS_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.remoteConnectionServer;
            },
        ],
        [
            'licenseForwarder',
            () => {
                return new aws_ecr_assets_1.DockerImageAsset(stack, 'SomeID', {
                    directory: STAGE_PATH,
                    buildArgs: LICENSE_FORWARDER_RECIPE.buildArgs,
                    target: LICENSE_FORWARDER_RECIPE.target,
                });
            },
            (recipes) => {
                return recipes.licenseForwarder;
            },
        ],
    ])('has proper %p container image', (_imageName, getExpected, getActual) => {
        // GIVEN
        const expectedImage = getExpected();
        // WHEN
        const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
            stage,
        });
        const actualImage = getActual(recipes);
        // THEN
        expect(actualImage.assetHash).toEqual(expectedImage.assetHash);
    });
    describe('.version', () => {
        test('creates a VersionQuery when referenced', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            recipes.version;
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_DEADLINE_INSTALLERS', {
                forceRun: assertions_1.Match.anyValue(),
                versionString: RELEASE_VERSION_STRING,
            });
        });
        test('does not create a VersionQuery when not referenced', () => {
            // GIVEN
            new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_DEADLINE_INSTALLERS', 0);
        });
        test('.linuxInstallers.client creates an Asset using the client installer', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            const clientInstaller = recipes.version.linuxInstallers.client;
            // THEN
            const asset = recipes.node.findChild('ClientInstallerAsset');
            expect(clientInstaller.s3Bucket).toEqual(asset.bucket);
            expect(clientInstaller.objectKey).toEqual(asset.s3ObjectKey);
        });
        test('.linuxInstallers.client successive accesses return the same bucket/key', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            const firstClientInstaller = recipes.version.linuxInstallers.client;
            const secondClientInstaller = recipes.version.linuxInstallers.client;
            // THEN
            expect(firstClientInstaller.objectKey).toBe(secondClientInstaller.objectKey);
            expect(firstClientInstaller.s3Bucket).toBe(secondClientInstaller.s3Bucket);
        });
        describe('.isLessThan()', () => {
            let recipes;
            beforeEach(() => {
                // GIVEN
                recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                    stage,
                });
            });
            test.each([
                [{ majorOffset: -1 }, false],
                [{ minorOffset: -1 }, false],
                [{ releaseOffset: -1 }, false],
                [{}, false],
                [{ majorOffset: 1 }, true],
                [{ minorOffset: 1 }, true],
                [{ releaseOffset: 1 }, true],
            ])('%s = %s', ({ majorOffset, minorOffset, releaseOffset }, expectedResult) => {
                // GIVEN
                majorOffset = majorOffset ?? 0;
                minorOffset = minorOffset ?? 0;
                releaseOffset = releaseOffset ?? 0;
                const other = new lib_1.Version([
                    MAJOR_VERSION + majorOffset,
                    MINOR_VERSION + minorOffset,
                    RELEASE_VERSION + releaseOffset,
                    0,
                ]);
                // WHEN
                const result = recipes.version.isLessThan(other);
                // THEN
                expect(result).toEqual(expectedResult);
            });
        });
        test('.linuxfullVersionString matches the stage manifest version', () => {
            // GIVEN
            const recipes = new lib_1.ThinkboxDockerRecipes(stack, 'Recipes', {
                stage,
            });
            // WHEN
            const linuxFullVersionString = recipes.version.linuxFullVersionString();
            // THEN
            expect(linuxFullVersionString).toEqual(FULL_VERSION_STRING);
        });
    });
    test.each([
        ['rcs', {
                [LICENSE_FORWARDER_RECIPE_NAME]: LICENSE_FORWARDER_RECIPE,
            }],
        ['license-forwarder', {
                [RCS_RECIPE_NAME]: RCS_RECIPE,
            }],
    ])('manifest missing required recipe %s throws error', (_recipeName, recipes) => {
        // GIVEN
        const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
        class StageWithPublicConstructor extends lib_1.Stage {
            constructor(props) {
                super(props);
            }
        }
        const manifest = {
            recipes,
            schema: 1,
            version: '1.2.3.4',
        };
        // WHEN
        expect(() => {
            new lib_1.ThinkboxDockerRecipes(isolatedStack, 'Recipes', {
                stage: new StageWithPublicConstructor({
                    manifest,
                    path: '/some/path',
                }),
            });
        })
            // THEN
            .toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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