"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const runtime_info_1 = require("../../core/lib/runtime-info");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_1 = require("../lib");
const test_helper_1 = require("./test-helper");
let app;
let stack;
let spotFleetStack;
let vpc;
let renderQueue;
let rcsImage;
const groupName = 'Group_Name';
const deadlineGroups = [
    groupName,
];
const workerMachineImage = new aws_ec2_1.GenericLinuxImage({
    'us-east-1': 'ami-any',
});
const instanceTypes = [
    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
];
const maxCapacity = 1;
describe('SpotEventPluginFleet', () => {
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'infraStack', {
            env: {
                region: 'us-east-1',
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        spotFleetStack = new aws_cdk_lib_1.Stack(app, 'SpotFleetStack', {
            env: {
                region: 'us-east-1',
            },
        });
    });
    describe('created with default values', () => {
        test('creates a security group', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.securityGroups).toBeDefined();
            expect(fleet.securityGroups.length).toBe(1);
            assertions_1.Template.fromStack(spotFleetStack).resourceCountIs('AWS::EC2::SecurityGroup', 1);
        });
        test('allows connection to the render queue', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
                SourceSecurityGroupId: spotFleetStack.resolve(fleet.connections.securityGroups[0].securityGroupId),
            });
        });
        test('creates a spot fleet instance role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBeDefined();
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assertions_1.Match.objectLike({
                    Statement: [assertions_1.Match.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn),
                ]),
            });
        });
        test('creates an instance profile', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.instanceProfile).toBeDefined();
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::IAM::InstanceProfile', {
                Roles: assertions_1.Match.arrayWith([{
                        Ref: spotFleetStack.getLogicalId(fleet.fleetInstanceRole.node.defaultChild),
                    }]),
            });
        });
        test('creates a spot fleet role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBeDefined();
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assertions_1.Match.objectLike({
                    Statement: [assertions_1.Match.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn),
                ]),
            });
        });
        test('adds group names to user data', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBeDefined();
            expect(renderedUserData).toMatch(groupName.toLocaleLowerCase());
        });
        test('adds RFDK tags', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            expect(fleet.tags).toBeDefined();
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::SecurityGroup', {
                Tags: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({
                        Key: rfdkTag.name,
                        Value: rfdkTag.value,
                    }),
                ]),
            });
        });
        test('uses default LogGroup prefix %s', () => {
            // GIVEN
            const id = 'SpotFleet';
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: '/renderfarm/' + id,
            });
        });
        test('sets default allocation strategy', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(lib_1.SpotFleetAllocationStrategy.LOWEST_PRICE);
        });
        test('does not set valid until property', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.validUntil).toBeUndefined();
        });
        test('does not set valid block devices', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.blockDevices).toBeUndefined();
        });
        test('does not set ssh key', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.keyName).toBeUndefined();
        });
        test('creates launch template configs for each instance type', () => {
            // WHEN
            const moreInstanceTypes = [
                new aws_ec2_1.InstanceType('t2.small'),
                new aws_ec2_1.InstanceType('c5.large'),
            ];
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                instanceTypes: moreInstanceTypes,
            });
            // THEN
            expect(fleet._launchTemplateConfigs.length).toBeGreaterThanOrEqual(moreInstanceTypes.length);
            moreInstanceTypes.forEach(instanceType => {
                expect(fleet._launchTemplateConfigs.some(ltc => {
                    return ltc.Overrides.some(override => override.InstanceType === instanceType.toString());
                })).toBeTruthy();
            });
        });
        test('creates launch template configs for each subnet id', () => {
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS });
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: subnets,
            });
            // THEN
            expect(fleet._launchTemplateConfigs.length).toBeGreaterThanOrEqual(subnets.subnets.length);
            subnets.subnetIds.forEach(subnetId => {
                expect(fleet._launchTemplateConfigs.some(ltc => {
                    return ltc.Overrides.some(override => override.SubnetId === subnetId);
                })).toBeTruthy();
            });
        });
        test('add tag indicating resource tracker is enabled', () => {
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assertions_1.Match.objectLike({
                    TagSpecifications: assertions_1.Match.arrayWith([
                        {
                            ResourceType: 'instance',
                            Tags: assertions_1.Match.arrayWith([
                                {
                                    Key: 'DeadlineTrackedAWSResource',
                                    Value: 'SpotEventPlugin',
                                },
                            ]),
                        },
                    ]),
                }),
            });
        });
        test('adds multiple fleet security groups to launch template', () => {
            // GIVEN
            const securityGroups = [
                new aws_ec2_1.SecurityGroup(stack, 'NewFleetSecurityGroup1', { vpc }),
                new aws_ec2_1.SecurityGroup(stack, 'NewFleetSecurityGroup2', { vpc }),
            ];
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet2', {
                vpc,
                renderQueue,
                deadlineGroups: ['group2'],
                instanceTypes: [new aws_ec2_1.InstanceType('t2.micro')],
                workerMachineImage,
                maxCapacity: 1,
                securityGroups,
            });
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assertions_1.Match.objectLike({
                    SecurityGroupIds: securityGroups.map(sg => spotFleetStack.resolve(sg.securityGroupId)),
                }),
            });
        });
        test('adds fleet tags to launch template', () => {
            // GIVEN
            const tag = {
                key: 'mykey',
                value: 'myvalue',
            };
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
            });
            // WHEN
            aws_cdk_lib_1.Tags.of(fleet).add(tag.key, tag.value);
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assertions_1.Match.objectLike({
                    TagSpecifications: assertions_1.Match.arrayWith([{
                            ResourceType: lib_1.SpotFleetResourceType.INSTANCE.toString(),
                            Tags: assertions_1.Match.arrayWith([
                                {
                                    Key: tag.key,
                                    Value: tag.value,
                                },
                            ]),
                        }]),
                }),
            });
        });
    });
    describe('created with custom values', () => {
        test('uses provided required properties', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const imageConfig = workerMachineImage.getImage(fleet);
            // THEN
            expect(fleet.deadlineGroups).toStrictEqual(deadlineGroups.map(group => group.toLocaleLowerCase()));
            expect(fleet.instanceTypes).toBe(instanceTypes);
            expect(fleet.osType).toBe(imageConfig.osType);
            expect(fleet.maxCapacity).toBe(maxCapacity);
        });
        test('uses provided security group', () => {
            // GIVEN
            const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).resourceCountIs('AWS::EC2::SecurityGroup', 0);
            expect(fleet.securityGroups.length).toBe(1);
            expect(fleet.securityGroups).toContainEqual(sg);
        });
        test('uses multiple provided security groups', () => {
            // GIVEN
            const sg1 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-123456789', {
                allowAllOutbound: false,
            });
            const sg2 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-987654321', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg1,
                    sg2,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).resourceCountIs('AWS::EC2::SecurityGroup', 0);
            expect(fleet.securityGroups.length).toBe(2);
            expect(fleet.securityGroups).toContainEqual(sg1);
            expect(fleet.securityGroups).toContainEqual(sg2);
        });
        test('adds to provided user data', () => {
            // GIVEN
            const originalCommands = 'original commands';
            const originalUserData = workerMachineImage.getImage(spotFleetStack).userData;
            originalUserData.addCommands(originalCommands);
            const renderedOriginalUser = originalUserData.render();
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userData: originalUserData,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBe(originalUserData);
            expect(renderedUserData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(originalCommands)));
            expect(renderedUserData).not.toEqual(renderedOriginalUser);
        });
        test('uses provided spot fleet instance role from the same stack', () => {
            // GIVEN
            const spotFleetInstanceRole = new aws_iam_1.Role(spotFleetStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                fleetInstanceRole: spotFleetInstanceRole,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBe(spotFleetInstanceRole);
            test_helper_1.resourcePropertiesCountIs(spotFleetStack, 'AWS::IAM::Role', {
                AssumeRolePolicyDocument: assertions_1.Match.objectLike({
                    Statement: assertions_1.Match.arrayWith([
                        assertions_1.Match.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        }),
                    ]),
                }),
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn),
                ]),
            }, 1);
        });
        test('throws if provided spot fleet instance role is not from the same stack', () => {
            // GIVEN
            const otherStack = new aws_cdk_lib_1.Stack(app, 'OtherStack', {
                env: { region: 'us-east-1' },
            });
            const spotFleetInstanceRole = new aws_iam_1.Role(otherStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            function createSpotEventPluginFleet() {
                new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    fleetInstanceRole: spotFleetInstanceRole,
                });
            }
            // THEN
            expect(createSpotEventPluginFleet).toThrowError('Fleet instance role should be created on the same stack as SpotEventPluginFleet to avoid circular dependencies.');
        });
        test('uses provided spot fleet role', () => {
            // GIVEN
            const fleetRole = new aws_iam_1.Role(stack, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                fleetRole: fleetRole,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBe(fleetRole);
            test_helper_1.resourcePropertiesCountIs(spotFleetStack, 'AWS::IAM::Role', {
                AssumeRolePolicyDocument: assertions_1.Match.objectLike({
                    Statement: assertions_1.Match.arrayWith([
                        assertions_1.Match.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        }),
                    ]),
                }),
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn),
                ]),
            }, 0);
        });
        test('tags resources deployed by CDK', () => {
            // GIVEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const tagName = 'name';
            const tagValue = 'tagValue';
            // WHEN
            aws_cdk_lib_1.Tags.of(fleet).add(tagName, tagValue);
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::SecurityGroup', {
                Tags: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({
                        Key: tagName,
                        Value: tagValue,
                    }),
                ]),
            });
        });
        test('uses provided subnets', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            const expectedSubnetId = stack.resolve(vpc.privateSubnets[0].subnetId);
            // THEN
            expect(stack.resolve(fleet.subnets.subnetIds)).toContainEqual(expectedSubnetId);
        });
        test('.defaultSubnets is false when subnets provided', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            // THEN
            expect(fleet.defaultSubnets).toBeFalsy();
        });
        test('uses provided allocation strategy', () => {
            // GIVEN
            const allocationStartegy = lib_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                allocationStrategy: allocationStartegy,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(allocationStartegy);
        });
        test('adds deadline region to user data', () => {
            // GIVEN
            const deadlineRegion = 'someregion';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlineRegion: deadlineRegion,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(deadlineRegion);
        });
        test('adds deadline pools to user data', () => {
            // GIVEN
            const pool1 = 'Pool1';
            const pool2 = 'pool2';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlinePools: [
                    pool1,
                    pool2,
                ],
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(pool1.toLocaleLowerCase());
            expect(renderedUserData).toMatch(pool2.toLocaleLowerCase());
        });
        test('uses provided ssh key name', () => {
            // GIVEN
            const keyName = 'test-key-name';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                keyName: keyName,
            });
            // THEN
            expect(fleet.keyName).toEqual(keyName);
        });
        test('UserData is added by UserDataProvider', () => {
            // WHEN
            class UserDataProvider extends lib_1.InstanceUserDataProvider {
                preCloudWatchAgent(host) {
                    host.userData.addCommands('echo preCloudWatchAgent');
                }
                preRenderQueueConfiguration(host) {
                    host.userData.addCommands('echo preRenderQueueConfiguration');
                }
                preWorkerConfiguration(host) {
                    host.userData.addCommands('echo preWorkerConfiguration');
                }
                postWorkerLaunch(host) {
                    host.userData.addCommands('echo postWorkerLaunch');
                }
            }
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userDataProvider: new UserDataProvider(spotFleetStack, 'UserDataProvider'),
            });
            const userData = fleet.userData.render();
            // THEN
            expect(userData).toContain('echo preCloudWatchAgent');
            expect(userData).toContain('echo preRenderQueueConfiguration');
            expect(userData).toContain('echo preWorkerConfiguration');
            expect(userData).toContain('echo postWorkerLaunch');
        });
        test.each([
            'test-prefix/',
            '',
        ])('uses custom LogGroup prefix %s', (testPrefix) => {
            // GIVEN
            const id = 'SpotFleet';
            const logGroupProps = {
                logGroupPrefix: testPrefix,
            };
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                logGroupProps: logGroupProps,
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: testPrefix + id,
            });
        });
        test('adds tag indicating resource tracker is not enabled', () => {
            // WHEN
            new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                instanceTypes,
                deadlineGroups,
                workerMachineImage,
                maxCapacity,
                trackInstancesWithResourceTracker: false,
            });
            // THEN
            assertions_1.Template.fromStack(spotFleetStack).hasResourceProperties('AWS::EC2::LaunchTemplate', {
                LaunchTemplateData: assertions_1.Match.objectLike({
                    TagSpecifications: assertions_1.Match.arrayWith([{
                            ResourceType: 'instance',
                            Tags: assertions_1.Match.arrayWith([
                                {
                                    Key: 'DeadlineResourceTracker',
                                    Value: 'SpotEventPlugin',
                                },
                            ]),
                        }]),
                }),
            });
        });
    });
    describe('allowing remote control', () => {
        test('from CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            });
        });
        test('to CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            });
        });
        test('from SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            });
        });
        test('to SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            });
        });
        test('from other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new aws_cdk_lib_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            });
        });
        test('to other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new aws_cdk_lib_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assertions_1.Template.fromStack(otherStack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            });
        });
    });
    describe('validation with', () => {
        describe('instance types', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyInstanceTypes = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: emptyInstanceTypes,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one instance type is required for a Spot Fleet Request Configuration/);
            });
            test('passes with at least one', () => {
                // GIVEN
                const oneInstanceType = [aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL)];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: oneInstanceType,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('subnets', () => {
            test('error if no subnets provided', () => {
                // GIVEN
                const invalidSubnets = {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
                    availabilityZones: ['dummy zone'],
                };
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    vpcSubnets: invalidSubnets,
                });
                // THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasError(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('Did not find any subnets matching.*'));
            });
        });
        describe('groups', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyDeadlineGroups = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: emptyDeadlineGroups,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one Deadline Group is required for a Spot Fleet Request Configuration/);
            });
            test.each([
                'none',
                'with space',
                'group_*',
            ])('throws with %s', (invalidGroupName) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [invalidGroupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineGroups'/);
            });
            test('passes with valid group name', () => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [groupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('region', () => {
            test.each([
                'none',
                'all',
                'unrecognized',
                'none@123',
                'None',
            ])('throws with %s', (deadlineRegion) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineRegion'/);
            });
            test('passes with reserved name as substring', () => {
                // GIVEN
                const deadlineRegion = 'none123';
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('Block Device Tests', () => {
            test('Warning if no BlockDevices provided', () => {
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                });
                // THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.'));
            });
            test('No Warnings if Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: '/dev/xvda',
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                        }],
                });
                //THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoInfo(`/${fleet.node.path}`, assertions_1.Match.anyValue());
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.anyValue());
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoError(`/${fleet.node.path}`, assertions_1.Match.anyValue());
            });
            test('Warnings if non-Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                        }],
                });
                //THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasWarning(`/${fleet.node.path}`, `The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('Warnings for BlockDevices without encryption specified', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                        }],
                });
                //THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasWarning(`/${fleet.node.path}`, `The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('No warnings for Ephemeral blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                        }],
                });
                //THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoInfo(`/${fleet.node.path}`, assertions_1.Match.anyValue());
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.anyValue());
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoError(`/${fleet.node.path}`, assertions_1.Match.anyValue());
            });
            test('No warnings for Suppressed blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                        }],
                });
                //THEN
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoInfo(`/${fleet.node.path}`, assertions_1.Match.anyValue());
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.anyValue());
                assertions_1.Annotations.fromStack(spotFleetStack).hasNoError(`/${fleet.node.path}`, assertions_1.Match.anyValue());
            });
            test('throws if block devices without iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.IO1;
                // WHEN
                function createSpotEventPluginFleet() {
                    return new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups: [
                            groupName,
                        ],
                        instanceTypes: [
                            aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                        ],
                        workerMachineImage,
                        maxCapacity: 1,
                        blockDevices: [{
                                deviceName,
                                volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                    volumeType,
                                }),
                            }],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/iops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            });
            test('warning if block devices with iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const iops = 100;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups: [
                        groupName,
                    ],
                    instanceTypes: [
                        aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                    ],
                    workerMachineImage,
                    maxCapacity: 1,
                    blockDevices: [{
                            deviceName,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                iops,
                                volumeType,
                            }),
                        }],
                });
                // THEN
                assertions_1.Annotations.fromStack(stack).hasWarning(`/${fleet.node.path}`, 'iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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