"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_docdb_1 = require("aws-cdk-lib/aws-docdb");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const core_1 = require("../../core");
const asset_constants_1 = require("../../core/test/asset-constants");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
const test_helper_1 = require("./test-helper");
let app;
let stack;
let vpc;
let version;
let installers;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
function create_version(version_array) {
    class MockVersion extends lib_1.Version {
        constructor() {
            super(...arguments);
            this.linuxInstallers = installers;
        }
        linuxFullVersionString() {
            return this.toString();
        }
    }
    return new MockVersion(version_array);
}
beforeEach(() => {
    app = new aws_cdk_lib_1.App();
    stack = new aws_cdk_lib_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC', {
        subnetConfiguration: [
            {
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
            {
                name: 'Isolated',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
            },
        ],
    });
    installers = {
        patchVersion: 0,
        repository: {
            objectKey: 'testInstaller',
            s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxInstallerBucket'),
        },
        client: {
            objectKey: 'testClientInstaller',
            s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxClientInstallerBucket'),
        },
    };
    version = create_version([10, 1, 19, 4]);
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: assertions_1.Match.arrayWith([
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ]),
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    });
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const isolatedSubnetIds = ['IsolatedSubnet1', 'IsolatedSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        isolatedSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: isolatedSubnetIds,
    });
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    });
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // gpg get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([{
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                    },
                }]),
        },
        PolicyName: assertions_1.Match.stringLikeRegexp('^repositoryInstallerInstanceRoleDefaultPolicy.*'),
    });
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: assertions_1.Match.arrayWith([
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ]),
                },
            ]),
        },
        PolicyName: assertions_1.Match.stringLikeRegexp('^repositoryInstallerInstanceRoleDefaultPolicy.*'),
    });
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    });
});
test.each([
    [[10, 1, 19, 4]],
    [[10, 2, 0, 9]],
])('repository installer logs all required files', (version_array) => {
    // GIVEN
    const repository_version = create_version(version_array);
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version: repository_version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/' +
                        (repository_version.isLessThan(lib_1.Version.MINIMUM_VERSION_USING_NEW_INSTALLBUILDER_LOG) ? 'bitrock' : 'installbuilder') +
                        '_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    });
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
});
test.each([
    [aws_cdk_lib_1.RemovalPolicy.DESTROY, 'Delete'],
    [aws_cdk_lib_1.RemovalPolicy.RETAIN, 'Retain'],
    [aws_cdk_lib_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    });
});
test.each([
    [aws_cdk_lib_1.RemovalPolicy.DESTROY, 'Delete'],
    [aws_cdk_lib_1.RemovalPolicy.RETAIN, 'Retain'],
    [aws_cdk_lib_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    });
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_1.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, assertions_1.Match.stringLikeRegexp('RemovalPolicy for database will not be applied since a database is not being created by this construct.*'));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_1.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DocDB::DBCluster', 1);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DocDB::DBInstance', 1);
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    });
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_1.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DocDB::DBCluster', 1);
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: assertions_1.Match.arrayWith(['audit']),
    }, 0);
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, 0);
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, warningMsg);
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new aws_cdk_lib_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    });
    assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' });
    assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' });
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }, instanceCount);
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: aws_cdk_lib_1.Duration.days(period),
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    });
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: aws_cdk_lib_1.Duration.days(20),
        },
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, 'Backup retention for database will not be applied since a database is not being created by this construct');
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        },
        backup: {
            retention: aws_cdk_lib_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EFS::FileSystem', 1);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EFS::MountTarget', 2);
    expect(repo.node.tryFindChild('PadEfsStorage')).toBeDefined();
    expect(repo.node.findChild('FileSystem').node.tryFindChild('PaddingAccessPoint')).toBeDefined();
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT30M',
            },
        },
    });
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toEqual(expect.stringContaining(version.versionString));
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    });
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    // THEN
    const regionToken = escapeTokenRegex('${Token[AWS.Region.\\d+]}');
    const asgLogicalIdToken = escapeTokenRegex('${Token[Stack.repositoryInstaller.Installer.ASG.LogicalID.\\d+]}');
    const expectedString = `function exitTrap\\(\\)\\{\nexitCode=\\$\\?\nsleep 1m\nTOKEN=\\$\\(curl -X PUT "http:\\/\\/169\\.254\\.169\\.254\\/latest\\/api\\/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> \\/dev\\/null\\)\nINSTANCE="\\$\\(curl -s -H "X-aws-ec2-metadata-token: \\$TOKEN" http:\\/\\/169\\.254\\.169\\.254\\/latest\\/meta-data\\/instance-id  2> \\/dev\\/null\\)"\nASG="\\$\\(aws --region ${regionToken} ec2 describe-tags --filters "Name=resource-id,Values=\\$\\{INSTANCE\\}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags\\[0\\]\\.Value" --output text\\)"\naws --region ${regionToken} autoscaling update-auto-scaling-group --auto-scaling-group-name \\$\\{ASG\\} --min-size 0 --max-size 0 --desired-capacity 0\n\\/opt\\/aws\\/bin\\/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalIdToken} --region ${regionToken} -e \\$exitCode \\|\\| echo 'Failed to send Cloudformation Signal'\n\\}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(new RegExp(expectedString));
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                {
                    Action: 'autoscaling:UpdateAutoScalingGroup',
                    Condition: {
                        StringEquals: {
                            'autoscaling:ResourceTag/resourceLogicalId': aws_cdk_lib_1.Names.uniqueId(repo),
                        },
                    },
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'ec2:DescribeTags',
                    Effect: 'Allow',
                    Resource: '*',
                },
            ]),
        },
    });
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instanceRole = (instance
        .node.findChild('InstanceRole')
        .node.defaultChild);
    const db = (repo
        .node.findChild('DocumentDatabase'));
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository false rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    expect(userData).toContain(`'/tmp/${asset_constants_2.CONFIG_REPO_DIRECT_CONNECT_LINUX.Key}.sh' "/mnt/repository/DeadlineRepository"`);
    // Assert the IAM instance profile is given read access to the database credentials secret
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([{
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: stack.resolve(db.secret.secretArn),
                }]),
        },
        Roles: [
            stack.resolve(instanceRole.ref),
        ],
    });
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Policy', {
        PolicyDocument: assertions_1.Match.objectLike({
            Statement: assertions_1.Match.arrayWith([
                {
                    Effect: 'Allow',
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_2.REPO_DC_ASSET.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_2.REPO_DC_ASSET.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
            ]),
        }),
        PolicyName: assertions_1.Match.stringLikeRegexp('Instance[1-2]InstanceRoleDefaultPolicy.*'),
    }, 2);
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 2,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
describe('Security Groups', () => {
    let repositorySecurityGroup;
    beforeEach(() => {
        repositorySecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'AdditionalSecurityGroup', { vpc });
    });
    describe('DocDB', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    database: repositorySecurityGroup,
                },
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBCluster', {
                VpcSecurityGroupIds: assertions_1.Match.arrayWith([stack.resolve(repositorySecurityGroup.securityGroupId)]),
            });
        });
    });
    describe('EFS', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    fileSystem: repositorySecurityGroup,
                },
            });
            // THEN
            // The EFS construct adds the security group to each mount target, and one mount target is generated per subnet.
            const numMountTargets = vpc.selectSubnets().subnets.length;
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::EFS::MountTarget', {
                SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(repositorySecurityGroup.securityGroupId)]),
            }, numMountTargets);
        });
    });
    describe('Installer', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    installer: repositorySecurityGroup,
                },
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(repositorySecurityGroup.securityGroupId)]),
            });
        });
    });
});
test('validates VersionQuery is not in a different stack', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    version = new lib_1.VersionQuery(stack, 'Version');
    new lib_1.Repository(newStack, 'Repository', {
        vpc,
        version,
    });
    // WHEN
    function synth() {
        app.synth();
    }
    // THEN
    expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
});
test('creates an EFS AccessPoint when no filesystem is supplied', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
    });
    // THEN
    const efsResource = repo.node.findChild('FileSystem').node.defaultChild;
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EFS::AccessPoint', {
        FileSystemId: stack.resolve(efsResource.ref),
        PosixUser: {
            Gid: '0',
            Uid: '0',
        },
        RootDirectory: {},
    });
});
test('throws an error if supplied a MountableEfs with no Access Point', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    const fs = new aws_efs_1.FileSystem(newStack, 'FileSystem', {
        vpc,
    });
    const mountableFs = new core_1.MountableEfs(newStack, {
        filesystem: fs,
    });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version,
            vpc,
            fileSystem: mountableFs,
        });
    }
    // THEN
    expect(when).toThrow('When using EFS with the Repository, you must provide an EFS Access Point');
});
test('disable Secrets Management by default when Deadline version is old', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    const repository = new lib_1.Repository(newStack, 'Repo', {
        vpc,
        version: oldVersion,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeFalsy();
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws when Secrets Management is enabled but deadline version is too low', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version: oldVersion,
            vpc,
            secretsManagementSettings: {
                enabled: true,
            },
        });
    }
    // THEN
    expect(when).toThrow(`The supplied Deadline version (${oldVersion.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${lib_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
});
test('imports repository settings', () => {
    // GIVEN
    const repositorySettings = new aws_s3_assets_1.Asset(stack, 'RepositorySettingsAsset', {
        path: __filename,
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        repositorySettings,
    });
    // THEN
    const installerGroup = repository.node.tryFindChild('Installer');
    // Note: The repostory settings is the js file that this compiles in to. The hash for that is fragile in that any
    // change to this file will change it. So, we search for an s3 cp of a js file (this is the only one).
    expect(installerGroup.userData.render()).toMatch(new RegExp('aws s3 cp \'s3://[^\']+/[a-f0-9]+.js\''));
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = true changes ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => true,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).toMatch('-o 1000:1000');
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = false does not change ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => false,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).not.toMatch('-o 1000:1000');
});
test('secret manager enabled', () => {
    // GIVEN
    const expectedCredentials = new aws_secretsmanager_1.Secret(stack, 'CustomSMAdminUser', {
        description: 'Custom admin credentials for the Secret Management',
        generateSecretString: {
            excludeCharacters: '\"$&\'()-/<>[\\]\`{|}',
            includeSpace: false,
            passwordLength: 24,
            requireEachIncludedType: true,
            generateStringKey: 'password',
            secretStringTemplate: JSON.stringify({ username: 'admin' }),
        },
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: true,
            credentials: expectedCredentials,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBe(expectedCredentials);
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`-r ${stack.region} -c ${expectedCredentials.secretArn}`);
});
test('secret manager is enabled by default', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeTruthy();
    expect(repository.secretsManagementSettings.credentials).toBeDefined();
});
test('credentials are undefined when secrets management is disabled', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: false,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws an error if credentials are undefined and database is imported', () => {
    // GIVEN
    const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
        vpc,
    });
    const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
    const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
        clusterEndpointAddress: '1.2.3.4',
        clusterIdentifier: 'foo',
        instanceEndpointAddresses: ['1.2.3.5'],
        instanceIdentifiers: ['i0'],
        port: 27001,
        readerEndpointAddress: '1.2.3.6',
        securityGroup: sg,
    });
    const databaseConnection = lib_1.DatabaseConnection.forDocDB({ database, login: secret });
    // WHEN
    function when() {
        new lib_1.Repository(stack, 'Repository', {
            vpc,
            version,
            database: databaseConnection,
        });
    }
    // THEN
    expect(when).toThrow('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
});
//# sourceMappingURL=data:application/json;base64,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