"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const core_1 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const secrets_management_ref_1 = require("./secrets-management-ref");
const version_1 = require("./version");
const worker_configuration_1 = require("./worker-configuration");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends constructs_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * ![architecture diagram](/diagrams/deadline/WorkerInstanceFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An EC2 Auto Scaling Group to maintain the number of instances.
 * - An Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *   in the fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *   that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *   the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *   patching strategy to keep this software current with the latest security patches. Please see
 *   https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps(props);
        const minCapacity = props.minCapacity ?? 1;
        const signals = minCapacity > 0 ? aws_autoscaling_1.Signals.waitForMinCapacity({
            timeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
        }) : undefined;
        if (signals === undefined) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Deploying with 0 minimum capacity. If there is an error in the EC2 UserData for this fleet, then your stack deployment will not fail. Watch for errors in your CloudWatch logs.');
        }
        const vpcSubnets = props.vpcSubnets ? props.vpcSubnets : {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        };
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets,
            securityGroup: props.securityGroup,
            minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            signals,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: props.spotPrice?.toString(),
            blockDevices: props.blockDevices,
            userData: props.userData,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensionsMap: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        // Configure the health monitoring if provided.
        // Note: This must be done *BEFORE* configuring the worker. We rely on the worker configuration
        // script restarting the launcher.
        this.configureHealthMonitor(props);
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this.fleet,
            cloudWatchLogSettings: {
                logGroupPrefix: WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: props,
            userDataProvider: props.userDataProvider,
        });
        this.listeningPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + WorkerInstanceFleet.MAX_WORKERS_PER_HOST);
        if (props.renderQueue.repository.secretsManagementSettings.enabled) {
            if (!props.vpcSubnets) {
                aws_cdk_lib_1.Annotations.of(this).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
            }
            props.renderQueue.configureSecretsManagementAutoRegistration({
                vpc: props.vpc,
                vpcSubnets,
                role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                dependent: this.fleet,
            });
        }
        // Updating the user data with successful cfn-signal commands.
        if (signals) {
            this.fleet.userData.addSignalOnExitCommand(this.fleet);
        }
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Add the security group to all workers
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    /**
     * @inheritdoc
     */
    allowListenerPortFrom(other) {
        this.connections.allowFrom(other.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    /**
     * @inheritdoc
     */
    allowListenerPortTo(other) {
        other.connections.allowTo(this.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. 'All', 'none' and 'unrecognized' are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    aws_cdk_lib_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props) {
        if (props.healthMonitor) {
            const healthCheckPort = props.healthCheckConfig?.port ?? core_1.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
            const configureHealthMonitorScriptAsset = core_1.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
                osType: this.fleet.osType,
                baseName: 'configureWorkerHealthCheck',
                rootDir: path.join(__dirname, '..', 'scripts/'),
            });
            configureHealthMonitorScriptAsset.executeOn({
                host: this.fleet,
                args: [
                    `'${healthCheckPort}'`,
                    `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                ],
            });
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
_a = JSII_RTTI_SYMBOL_1;
WorkerInstanceFleet[_a] = { fqn: "aws-rfdk.deadline.WorkerInstanceFleet", version: "1.1.0" };
/**
 * The min limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = aws_cdk_lib_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
WorkerInstanceFleet.MAX_WORKERS_PER_HOST = 8;
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = aws_cdk_lib_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya2VyLWZsZWV0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsid29ya2VyLWZsZWV0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7OztHQUdHO0FBRUgsNkJBQTZCO0FBQzdCLDZDQUdxQjtBQUNyQixpRUFNcUM7QUFDckMsK0RBQTJEO0FBQzNELGlEQWE2QjtBQUU3QixpREFPNkI7QUFDN0IsMkNBQW1EO0FBRW5ELHFDQU9vQjtBQUNwQiw4REFFcUM7QUFJckMscUVBR2tDO0FBQ2xDLHVDQUFvQztBQUNwQyxpRUFJZ0M7QUFnTGhDOztHQUVHO0FBQ0gsTUFBZSx1QkFBd0IsU0FBUSxzQkFBUztDQTZEdkQ7QUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQStCRztBQUNILE1BQWEsbUJBQW9CLFNBQVEsdUJBQXVCO0lBNEY5RCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCO1FBQ3ZFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLENBQUMsQ0FBQztRQUMzQyxNQUFNLE9BQU8sR0FBRyxXQUFXLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyx5QkFBTyxDQUFDLGtCQUFrQixDQUFDO1lBQzNELE9BQU8sRUFBRSxtQkFBbUIsQ0FBQyx1QkFBdUI7U0FDckQsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDZixJQUFJLE9BQU8sS0FBSyxTQUFTLEVBQUU7WUFDekIseUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUFDLGlMQUFpTCxDQUFDLENBQUM7U0FDcE47UUFFRCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUN2RCxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxtQkFBbUI7U0FDM0MsQ0FBQztRQUVGLDZDQUE2QztRQUM3QyxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksa0NBQWdCLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNqRCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxZQUFZLEVBQUUsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxzQkFBWSxDQUFDLEVBQUUsQ0FBQyx1QkFBYSxDQUFDLEVBQUUsRUFBRSxzQkFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQy9HLFlBQVksRUFBRSxLQUFLLENBQUMsa0JBQWtCO1lBQ3RDLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixVQUFVO1lBQ1YsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLFdBQVc7WUFDWCxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQ3RDLE9BQU87WUFDUCxXQUFXLEVBQUUsNkJBQVcsQ0FBQyxHQUFHLENBQUM7Z0JBQzNCLEtBQUssRUFBRSxtQkFBbUIsQ0FBQyw2QkFBNkI7YUFDekQsQ0FBQztZQUNGLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVMsRUFBRSxRQUFRLEVBQUU7WUFDdEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtTQUN6QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLFFBQVEsQ0FBRSxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFvQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNsRyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQztRQUN4QixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUM7UUFDbEMsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksdUJBQU0sQ0FBQztZQUNyQyxTQUFTLEVBQUUsaUJBQWlCO1lBQzVCLFVBQVUsRUFBRSxzQkFBc0I7WUFDbEMsYUFBYSxFQUFFO2dCQUNiLG9CQUFvQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CO2FBQ3REO1lBQ0QsS0FBSyxFQUFFLHNCQUFzQjtTQUM5QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxnQkFBTSxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUM1RCxVQUFVLEVBQUUsQ0FBQyxJQUFJLHlCQUFlLENBQUM7b0JBQy9CLE9BQU8sRUFBRSxDQUFDLG9DQUFvQyxDQUFDO29CQUMvQyxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDO2lCQUM1QyxDQUFDLENBQUM7U0FDSixDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFvQyxDQUFDLGlCQUFpQixHQUFHLENBQUM7Z0JBQ3pFLFdBQVcsRUFBRSxTQUFTO2dCQUN0QixPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQzthQUNsQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1FBQ2hELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFFMUMsK0NBQStDO1FBQy9DLCtGQUErRjtRQUMvRixrQ0FBa0M7UUFDbEMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRW5DLE1BQU0sWUFBWSxHQUFHLElBQUksa0RBQTJCLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUM3RCxNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUs7WUFDbEIscUJBQXFCLEVBQUU7Z0JBQ3JCLGNBQWMsRUFBRSxtQkFBbUIsQ0FBQyx3QkFBd0I7Z0JBQzVELEdBQUcsS0FBSyxDQUFDLGFBQWE7YUFDdkI7WUFDRCxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsY0FBYyxFQUFFLEtBQUs7WUFDckIsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtTQUN6QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsY0FBYyxHQUFHLGNBQUksQ0FBQyxRQUFRLENBQ2pDLFlBQVksQ0FBQyxZQUFZLEVBQ3pCLFlBQVksQ0FBQyxZQUFZLEdBQUcsbUJBQW1CLENBQUMsb0JBQW9CLENBQ3JFLENBQUM7UUFFRixJQUFJLEtBQUssQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLHlCQUF5QixDQUFDLE9BQU8sRUFBRTtZQUNsRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsRUFBRTtnQkFDckIseUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUM3QixxUkFBcVIsQ0FDdFIsQ0FBQzthQUNIO1lBQ0QsS0FBSyxDQUFDLFdBQVcsQ0FBQywwQ0FBMEMsQ0FBQztnQkFDM0QsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLFVBQVU7Z0JBQ1YsSUFBSSxFQUFFLDhDQUFxQixDQUFDLE1BQU07Z0JBQ2xDLGtCQUFrQixFQUFFLDREQUFtQyxDQUFDLFVBQVU7Z0JBQ2xFLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSzthQUN0QixDQUFDLENBQUM7U0FDSjtRQUVELDhEQUE4RDtRQUM5RCxJQUFJLE9BQU8sRUFBRTtZQUNYLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUN4RDtRQUVELDZDQUE2QztRQUM3QywyQkFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3JCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksZ0JBQWdCLENBQUMsYUFBNkI7UUFDbkQsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUFtQjtRQUM5QyxJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsc0NBQXNDLENBQUMsQ0FBQztJQUM3RyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxtQkFBbUIsQ0FBQyxLQUFtQjtRQUM1QyxLQUFLLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsc0NBQXNDLENBQUMsQ0FBQztJQUMzRyxDQUFDO0lBRU8sYUFBYSxDQUFDLEtBQStCO1FBQ25ELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDeEMsSUFBSSxDQUFDLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsNEJBQTRCLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDMUYsSUFBSSxDQUFDLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsNEJBQTRCLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDeEYsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLCtDQUErQyxDQUFDLENBQUM7UUFDbkYsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRU8saUJBQWlCLENBQUMsU0FBNkI7UUFDckQsSUFBSSxTQUFTLElBQUksQ0FBQyxDQUFDLFNBQVMsSUFBSSxtQkFBbUIsQ0FBQyxvQkFBb0IsSUFBSSxTQUFTLElBQUksbUJBQW1CLENBQUMsb0JBQW9CLENBQUMsRUFBRTtZQUNsSSxNQUFNLElBQUksS0FBSyxDQUFDLGtCQUFrQixTQUFTLHNFQUFzRSxtQkFBbUIsQ0FBQyxvQkFBb0IsUUFBUSxtQkFBbUIsQ0FBQyxvQkFBb0IsR0FBRyxDQUFDLENBQUM7U0FDL007SUFDSCxDQUFDO0lBRU8sY0FBYyxDQUFDLE1BQTBCLEVBQUUsS0FBYTtRQUM5RCxJQUFJLE1BQU0sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsTUFBTSwrSUFBK0ksQ0FBQyxDQUFDO1NBQzFMO0lBQ0gsQ0FBQztJQUVPLDhCQUE4QixDQUFDLEtBQTJCLEVBQUUsS0FBYSxFQUFFLFFBQWdCO1FBQ2pHLElBQUksS0FBSyxFQUFFO1lBQ1QsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDcEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUU7b0JBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0JBQWtCLEtBQUssa0JBQWtCLFFBQVEsc0dBQXNHLENBQUMsQ0FBQztpQkFDMUs7WUFDSCxDQUFDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztJQUVPLG9CQUFvQixDQUFDLFlBQXVDO1FBQ2xFLElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUM5Qix5QkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsb0JBQW9CLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSx1R0FBdUc7Z0JBQ3JLLGtMQUFrTCxDQUFDLENBQUM7U0FDdkw7YUFBTTtZQUNMLFlBQVksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQzVCLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEtBQUssU0FBUyxFQUFFO29CQUN6Qyx1Q0FBdUM7b0JBQ3ZDLE9BQU87aUJBQ1I7Z0JBRUQsNEdBQTRHO2dCQUM1Ryx3Q0FBd0M7Z0JBQ3hDLElBQUssQ0FBQyxXQUFXLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLEtBQUssS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBRSxFQUFHO29CQUM5SSx5QkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsb0JBQW9CLE1BQU0sQ0FBQyxVQUFVLHlCQUF5QixJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUscUJBQXFCO3dCQUMzSCw0R0FBNEcsQ0FBQyxDQUFDO2lCQUNuSDtZQUNILENBQUMsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDO0lBRU8sc0JBQXNCLENBQUMsS0FBK0I7UUFDNUQsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxJQUFJLElBQUksb0JBQWEsQ0FBQyx5QkFBeUIsQ0FBQztZQUVqRyxNQUFNLGlDQUFpQyxHQUFHLGtCQUFXLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO2dCQUMxRyxNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNO2dCQUN6QixRQUFRLEVBQUUsNEJBQTRCO2dCQUN0QyxPQUFPLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FDaEIsU0FBUyxFQUNULElBQUksRUFDSixVQUFVLENBQ1g7YUFDRixDQUFDLENBQUM7WUFFSCxpQ0FBaUMsQ0FBQyxTQUFTLENBQUM7Z0JBQzFDLElBQUksRUFBRSxJQUFJLENBQUMsS0FBSztnQkFDaEIsSUFBSSxFQUFFO29CQUNKLElBQUksZUFBZSxHQUFHO29CQUN0QixJQUFJLGlCQUFPLENBQUMsa0NBQWtDLENBQUMsUUFBUSxFQUFFLEdBQUc7aUJBQzdEO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsS0FBSyxDQUFDLGFBQWEsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsSUFBSTtnQkFDakUsSUFBSSxFQUFFLGVBQWU7YUFDdEIsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLHlCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQyxvQkFBb0IsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLHlKQUF5SixDQUFDLENBQUM7U0FDNU47SUFDSCxDQUFDOztBQTlTSCxrREErU0M7OztBQTlTQzs7R0FFRztBQUNvQix3Q0FBb0IsR0FBRyxLQUFLLENBQUM7QUFFcEQ7O0dBRUc7QUFDb0Isd0NBQW9CLEdBQUcsR0FBRyxDQUFDO0FBRWxEOzs7O0dBSUc7QUFDcUIsaURBQTZCLEdBQUcsc0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFFNUU7O0dBRUc7QUFDcUIsNENBQXdCLEdBQVcsY0FBYyxDQUFDO0FBRTFFOzs7O0dBSUc7QUFDcUIsd0NBQW9CLEdBQUcsQ0FBQyxDQUFDO0FBRWpEOzs7O0dBSUc7QUFDcUIsMkNBQXVCLEdBQUcsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHtcbiAgQW5ub3RhdGlvbnMsXG4gIER1cmF0aW9uLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQge1xuICBBdXRvU2NhbGluZ0dyb3VwLFxuICBCbG9ja0RldmljZSxcbiAgQ2ZuQXV0b1NjYWxpbmdHcm91cCxcbiAgSGVhbHRoQ2hlY2ssXG4gIFNpZ25hbHMsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1hdXRvc2NhbGluZyc7XG5pbXBvcnQge0lNZXRyaWMsIE1ldHJpY30gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0IHtcbiAgQ29ubmVjdGlvbnMsXG4gIElDb25uZWN0YWJsZSxcbiAgSU1hY2hpbmVJbWFnZSxcbiAgSW5zdGFuY2VDbGFzcyxcbiAgSW5zdGFuY2VTaXplLFxuICBJbnN0YW5jZVR5cGUsXG4gIElTZWN1cml0eUdyb3VwLFxuICBJVnBjLFxuICBQb3J0LFxuICBTdWJuZXRTZWxlY3Rpb24sXG4gIFN1Ym5ldFR5cGUsXG4gIFVzZXJEYXRhLFxufSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCB7SUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0fSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQge1xuICBJR3JhbnRhYmxlLFxuICBJUG9saWN5LFxuICBJUHJpbmNpcGFsLFxuICBJUm9sZSxcbiAgUG9saWN5LFxuICBQb2xpY3lTdGF0ZW1lbnQsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbmltcG9ydCB7XG4gIEhlYWx0aENoZWNrQ29uZmlnLFxuICBIZWFsdGhNb25pdG9yLFxuICBJSGVhbHRoTW9uaXRvcixcbiAgSU1vbml0b3JhYmxlRmxlZXQsXG4gIExvZ0dyb3VwRmFjdG9yeVByb3BzLFxuICBTY3JpcHRBc3NldCxcbn0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQge1xuICB0YWdDb25zdHJ1Y3QsXG59IGZyb20gJy4uLy4uL2NvcmUvbGliL3J1bnRpbWUtaW5mbyc7XG5pbXBvcnQge1xuICBJUmVuZGVyUXVldWUsXG59IGZyb20gJy4vcmVuZGVyLXF1ZXVlJztcbmltcG9ydCB7XG4gIFNlY3JldHNNYW5hZ2VtZW50UmVnaXN0cmF0aW9uU3RhdHVzLFxuICBTZWNyZXRzTWFuYWdlbWVudFJvbGUsXG59IGZyb20gJy4vc2VjcmV0cy1tYW5hZ2VtZW50LXJlZic7XG5pbXBvcnQgeyBWZXJzaW9uIH0gZnJvbSAnLi92ZXJzaW9uJztcbmltcG9ydCB7XG4gIElJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXIsXG4gIFdvcmtlckluc3RhbmNlQ29uZmlndXJhdGlvbixcbiAgV29ya2VyU2V0dGluZ3MsXG59IGZyb20gJy4vd29ya2VyLWNvbmZpZ3VyYXRpb24nO1xuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgRGVhZGxpbmUgV29ya2VyIEZsZWV0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElXb3JrZXJGbGVldCBleHRlbmRzIElDb25uZWN0YWJsZSwgSUNvbnN0cnVjdCwgSUdyYW50YWJsZSB7XG4gIC8qKlxuICAgKiBBbGxvdyBhY2Nlc3MgdG8gdGhlIHdvcmtlcidzIHJlbW90ZSBjb21tYW5kIGxpc3RlbmVyIHBvcnQgKGNvbmZpZ3VyZWQgYXMgYSBwYXJ0IG9mIHRoZVxuICAgKiBXb3JrZXJDb25maWd1cmF0aW9uKSBmb3IgYW4gSUNvbm5lY3RhYmxlIHRoYXQgaXMgZWl0aGVyIGluIHRoaXMgc3RhY2ssIG9yIGluIGEgc3RhY2sgdGhhdFxuICAgKiBkZXBlbmRzIG9uIHRoaXMgc3RhY2suIElmIHRoaXMgc3RhY2sgZGVwZW5kcyBvbiB0aGUgb3RoZXIgc3RhY2ssIHVzZSBhbGxvd0xpc3RlbmVyUG9ydFRvKCkuXG4gICAqXG4gICAqIENvbW1vbiB1c2VzIGFyZTpcbiAgICpcbiAgICogICBBZGRpbmcgYSBTZWN1cml0eUdyb3VwOlxuICAgKiAgICAgYHdvcmtlckZsZWV0LmFsbG93TGlzdGVuZXJQb3J0RnJvbShzZWN1cml0eUdyb3VwKWBcbiAgICpcbiAgICogICBBZGRpbmcgYSBDSURSOlxuICAgKiAgICAgYHdvcmtlckZsZWV0LmFsbG93TGlzdGVuZXJQb3J0RnJvbShQZWVyLmlwdjQoJzEwLjAuMC4wLzI0JykuY29ubmVjdGlvbnMpYFxuICAgKi9cbiAgYWxsb3dMaXN0ZW5lclBvcnRGcm9tKG90aGVyOiBJQ29ubmVjdGFibGUpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBBbGxvdyBhY2Nlc3MgdG8gdGhlIHdvcmtlcidzIHJlbW90ZSBjb21tYW5kIGxpc3RlbmVyIHBvcnQgKGNvbmZpZ3VyZWQgYXMgYSBwYXJ0IG9mIHRoZVxuICAgKiBXb3JrZXJDb25maWd1cmF0aW9uKSBmb3IgYW4gSUNvbm5lY3RhYmxlIHRoYXQgaXMgZWl0aGVyIGluIHRoaXMgc3RhY2ssIG9yIGluIGEgc3RhY2sgdGhhdCB0aGlzXG4gICAqIHN0YWNrIGRlcGVuZHMgb24uIElmIHRoZSBvdGhlciBzdGFjayBkZXBlbmRzIG9uIHRoaXMgc3RhY2ssIHVzZSBhbGxvd0xpc3RlbmVyUG9ydEZyb20oKS5cbiAgICpcbiAgICogQ29tbW9uIHVzZXMgYXJlOlxuICAgKlxuICAgKiAgIEFkZGluZyBhIFNlY3VyaXR5R3JvdXA6XG4gICAqICAgICBgd29ya2VyRmxlZXQuYWxsb3dMaXN0ZW5lclBvcnRUbyhzZWN1cml0eUdyb3VwKWBcbiAgICpcbiAgICogICBBZGRpbmcgYSBDSURSOlxuICAgKiAgICAgYHdvcmtlckZsZWV0LmFsbG93TGlzdGVuZXJQb3J0VG8oUGVlci5pcHY0KCcxMC4wLjAuMC8yNCcpLmNvbm5lY3Rpb25zKWBcbiAgICovXG4gIGFsbG93TGlzdGVuZXJQb3J0VG8ob3RoZXI6IElDb25uZWN0YWJsZSk6IHZvaWQ7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgdGhlIERlYWRsaW5lIFdvcmtlciBGbGVldC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXb3JrZXJJbnN0YW5jZUZsZWV0UHJvcHMgZXh0ZW5kcyBXb3JrZXJTZXR0aW5ncyB7XG4gIC8qKlxuICAgKiBWUEMgdG8gbGF1bmNoIHRoZSB3b3JrZXIgZmxlZXQgaW4uXG4gICAqL1xuICByZWFkb25seSB2cGM6IElWcGM7XG5cbiAgLyoqXG4gICAqIFNlY3VyaXR5IEdyb3VwIHRvIGFzc2lnbiB0byB0aGlzIGZsZWV0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGNyZWF0ZSBuZXcgc2VjdXJpdHkgZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBJU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogQW4gSUFNIHJvbGUgdG8gYXNzb2NpYXRlIHdpdGggdGhlIGluc3RhbmNlIHByb2ZpbGUgYXNzaWduZWQgdG8gaXRzIHJlc291cmNlcy5cbiAgICpcbiAgICogVGhlIHJvbGUgbXVzdCBiZSBhc3N1bWFibGUgYnkgdGhlIHNlcnZpY2UgcHJpbmNpcGFsIGBlYzIuYW1hem9uYXdzLmNvbWA6XG4gICAqXG4gICAqICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ015Um9sZScsIHtcbiAgICogICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKVxuICAgKiAgICB9KTtcbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIHJvbGUgd2lsbCBhdXRvbWF0aWNhbGx5IGJlIGNyZWF0ZWQsIGl0IGNhbiBiZSBhY2Nlc3NlZCB2aWEgdGhlIGByb2xlYCBwcm9wZXJ0eVxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IElSb2xlO1xuXG4gIC8qKlxuICAgKiBBTUkgb2YgdGhlIGRlYWRsaW5lIHdvcmtlciB0byBsYXVuY2guXG4gICAqL1xuICByZWFkb25seSB3b3JrZXJNYWNoaW5lSW1hZ2U6IElNYWNoaW5lSW1hZ2U7XG5cbiAgLyoqXG4gICAqIFR5cGUgb2YgaW5zdGFuY2UgdG8gbGF1bmNoIGZvciBleGVjdXRpbmcgcmVwb3NpdG9yeSBpbnN0YWxsZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYSBUMi1MYXJnZSB0eXBlIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IEluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogV2hlcmUgdG8gcGxhY2UgdGhlIGluc3RhbmNlIHdpdGhpbiB0aGUgVlBDLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFByaXZhdGUgc3VibmV0cy5cbiAgICovXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBTdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgU1NIIGtleXBhaXIgdG8gZ3JhbnQgYWNjZXNzIHRvIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIFNTSCBhY2Nlc3Mgd2lsbCBiZSBwb3NzaWJsZS5cbiAgICovXG4gIHJlYWRvbmx5IGtleU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluaXRpYWwgYW1vdW50IG9mIHdvcmtlcnMgaW4gdGhlIGZsZWV0LlxuICAgKlxuICAgKiBJZiB0aGlzIGlzIHNldCB0byBhIG51bWJlciwgZXZlcnkgZGVwbG95bWVudCB3aWxsIHJlc2V0IHRoZSBhbW91bnQgb2ZcbiAgICogd29ya2VycyB0byB0aGlzIG51bWJlci4gSXQgaXMgcmVjb21tZW5kZWQgdG8gbGVhdmUgdGhpcyB2YWx1ZSBibGFuay5cbiAgICpcbiAgICogQGRlZmF1bHQgbWluQ2FwYWNpdHksIGFuZCBsZWF2ZSB1bmNoYW5nZWQgZHVyaW5nIGRlcGxveW1lbnRcbiAgICovXG4gIHJlYWRvbmx5IGRlc2lyZWRDYXBhY2l0eT86IG51bWJlcjtcblxuICAvKipcbiAgICogTWluaW11bSBudW1iZXIgb2YgaW5zdGFuY2VzIGluIHRoZSBmbGVldC5cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgbWluQ2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE1heGltdW0gbnVtYmVyIG9mIGluc3RhbmNlcyBpbiB0aGUgZmxlZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGRlc2lyZWRDYXBhY2l0eSwgb3IgbWluQ2FwYWNpdHkgaWYgZGVzaXJlZENhcGFjaXR5IGlzIG5vdCBzZXRcbiAgICovXG4gIHJlYWRvbmx5IG1heENhcGFjaXR5PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBFbmRwb2ludCBmb3IgdGhlIFJlbmRlclF1ZXVlLCB0byB3aGljaCB0aGUgd29ya2VyIGZsZWV0IG5lZWRzIHRvIGJlIGNvbm5lY3RlZC5cbiAgICovXG4gIHJlYWRvbmx5IHJlbmRlclF1ZXVlOiBJUmVuZGVyUXVldWU7XG5cbiAgLyoqXG4gICAqIFByb3BlcnRpZXMgZm9yIHNldHRpbmcgdXAgdGhlIERlYWRsaW5lIFdvcmtlcidzIExvZ0dyb3VwXG4gICAqIEBkZWZhdWx0IC0gTG9nR3JvdXAgd2lsbCBiZSBjcmVhdGVkIHdpdGggYWxsIHByb3BlcnRpZXMnIGRlZmF1bHQgdmFsdWVzIGFuZCBhIHByZWZpeCBvZiBcIi9yZW5kZXJmYXJtL1wiLlxuICAgKi9cbiAgcmVhZG9ubHkgbG9nR3JvdXBQcm9wcz86IExvZ0dyb3VwRmFjdG9yeVByb3BzO1xuXG4gIC8qKlxuICAgKiBIZWFsdGggTW9uaXRvciBjb21wb25lbnQgdG8gbW9uaXRvciB0aGUgaGVhbHRoIG9mIGluc3RhbmNlcy5cbiAgICpcbiAgICogTm90ZTogVGhlIGhlYWx0aC1jaGVjayBmZWF0dXJlIGlzIHN1cHBvcnRlZCB3aXRoIERlYWRsaW5lIENsaWVudCB2MTAuMS45IGFuZCBsYXRlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBIZWFsdGggTW9uaXRvcmluZyBpcyB0dXJuZWQtb2ZmXG4gICAqL1xuICByZWFkb25seSBoZWFsdGhNb25pdG9yPzogSUhlYWx0aE1vbml0b3I7XG5cbiAgLyoqXG4gICAqIFByb3BlcnRpZXMgZm9yIGNvbmZpZ3VyaW5nIGEgaGVhbHRoIGNoZWNrLlxuICAgKlxuICAgKiBOb3RlOiBUaGUgaGVhbHRoLWNoZWNrIGZlYXR1cmUgaXMgc3VwcG9ydGVkIHdpdGggRGVhZGxpbmUgQ2xpZW50IHYxMC4xLjkgYW5kIGxhdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBwcm9wZXJ0aWVzIG9mIEhlYWx0aENoZWNrQ29uZmlnIGFwcGxpZXNcbiAgICovXG4gIHJlYWRvbmx5IGhlYWx0aENoZWNrQ29uZmlnPzogSGVhbHRoQ2hlY2tDb25maWc7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIGhvdXJseSBwcmljZSgkKSB0byBiZSBwYWlkIGZvciBlYWNoIFNwb3QgaW5zdGFuY2UuXG4gICAqIG1pbiAtIDAuMDAxOyBtYXggLSAyNTVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBsYXVuY2hlcyBvbi1kZW1hbmQgRUMyIGluc3RhbmNlcy5cbiAgICovXG4gIHJlYWRvbmx5IHNwb3RQcmljZT86IG51bWJlcjtcblxuICAvKlxuICAgKiBUaGUgQmxvY2sgZGV2aWNlcyB0aGF0IHdpbGwgYmUgYXR0YWNoZWQgdG8geW91ciB3b3JrZXJzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBUaGUgZGVmYXVsdCBkZXZpY2VzIG9mIHRoZSBwcm92aWRlZCBhbWkgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgYmxvY2tEZXZpY2VzPzogQmxvY2tEZXZpY2VbXTtcblxuICAvKipcbiAgICogVGhlIHNwZWNpZmljIFVzZXJEYXRhIHRvIHVzZS5cbiAgICpcbiAgICogVGhlIFVzZXJEYXRhIHdpbGwgYmUgbXV0YXRlZCBieSB0aGlzIGNvbnN0cnVjdCBhbmQgbWF5IGJlIG11dGF0ZWQgYWZ0ZXJ3YXJkcyBhcyB3ZWxsLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBIFVzZXJEYXRhIG9iamVjdCBhcHByb3ByaWF0ZSBmb3IgdGhlIE1hY2hpbmVJbWFnZSdzIE9wZXJhdGluZyBTeXN0ZW0gaXMgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHVzZXJEYXRhPzogVXNlckRhdGE7XG5cbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIHByb3ZpZGVyIG9mIHVzZXIgZGF0YSBjb21tYW5kcyB0byBiZSBpbmplY3RlZCBhdCB2YXJpb3VzIHBvaW50cyBkdXJpbmcgdGhlIFdvcmtlciBjb25maWd1cmF0aW9uIGxpZmVjeWNsZS5cbiAgICogWW91IGNhbiBwcm92aWRlIGEgc3ViY2xhc3Mgb2YgSW5zdGFuY2VVc2VyRGF0YVByb3ZpZGVyIHdpdGggdGhlIG1ldGhvZHMgb3ZlcnJpZGRlbiBhcyBkZXNpcmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdXNlckRhdGFQcm92aWRlcj86IElJbnN0YW5jZVVzZXJEYXRhUHJvdmlkZXI7XG59XG5cbi8qKlxuICogIEEgbmV3IG9yIGltcG9ydGVkIERlYWRsaW5lIFdvcmtlciBGbGVldC5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgV29ya2VySW5zdGFuY2VGbGVldEJhc2UgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJV29ya2VyRmxlZXQsIElNb25pdG9yYWJsZUZsZWV0IHtcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGdyb3Vwcy9ydWxlcyB1c2VkIHRvIGFsbG93IG5ldHdvcmsgY29ubmVjdGlvbnMgdG8gdGhlIGZpbGUgc3lzdGVtLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBDb25uZWN0aW9ucztcblxuICAvKipcbiAgICogVGhlIHByaW5jaXBhbCB0byBncmFudCBwZXJtaXNzaW9ucyB0by5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBncmFudFByaW5jaXBhbDogSVByaW5jaXBhbDtcblxuICAvKipcbiAgICogVGhlIEFTRyBvYmplY3QgY3JlYXRlZCBieSB0aGUgY29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGZsZWV0OiBBdXRvU2NhbGluZ0dyb3VwO1xuXG4gIC8qKlxuICAgKiBUaGlzIGZpZWxkIGV4cGVjdHMgdGhlIGJhc2UgY2FwYWNpdHkgbWV0cmljIG9mIHRoZSBmbGVldCBhZ2FpbnN0XG4gICAqIHdoaWNoLCB0aGUgaGVhbHRoeSBwZXJjZW50IHdpbGwgYmUgY2FsY3VsYXRlZC5cbiAgICpcbiAgICogZWcuOiBHcm91cERlc2lyZWRDYXBhY2l0eSBmb3IgYW4gQVNHXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdGFyZ2V0Q2FwYWNpdHlNZXRyaWM6IElNZXRyaWM7XG5cbiAgLyoqXG4gICAqIFRoaXMgZmllbGQgZXhwZWN0cyB0aGUgY29tcG9uZW50IG9mIHR5cGUgSU5ldHdvcmtMb2FkQmFsYW5jZXJUYXJnZXRcbiAgICogd2hpY2ggY2FuIGJlIGF0dGFjaGVkIHRvIE5ldHdvcmsgTG9hZCBCYWxhbmNlciBmb3IgbW9uaXRvcmluZy5cbiAgICpcbiAgICogZWcuIEFuIEF1dG9TY2FsaW5nR3JvdXBcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB0YXJnZXRUb01vbml0b3I6IElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlclRhcmdldDtcblxuICAvKipcbiAgICogVGhpcyBmaWVsZCBleHBlY3RzIGEgcG9saWN5IHdoaWNoIGNhbiBiZSBhdHRhY2hlZCB0byB0aGUgbGFtYmRhXG4gICAqIGV4ZWN1dGlvbiByb2xlIHNvIHRoYXQgaXQgaXMgY2FwYWJsZSBvZiBzdXNwZW5kaW5nIHRoZSBmbGVldC5cbiAgICpcbiAgICogZWcuOiBhdXRvc2NhbGluZzpVcGRhdGVBdXRvU2NhbGluZ0dyb3VwIHBlcm1pc3Npb24gZm9yIGFuIEFTR1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHRhcmdldFVwZGF0ZVBvbGljeTogSVBvbGljeTtcblxuICAvKipcbiAgICogVGhpcyBmaWVsZCBleHBlY3RzIHRoZSBtYXhpbXVtIGluc3RhbmNlIGNvdW50IHRoaXMgZmxlZXQgY2FuIGhhdmUuXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdGFyZ2V0Q2FwYWNpdHk6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhpcyBmaWVsZCBleHBlY3RzIHRoZSBzY29wZSBpbiB3aGljaCB0byBjcmVhdGUgdGhlIG1vbml0b3JpbmcgcmVzb3VyY2VcbiAgICogbGlrZSBUYXJnZXRHcm91cHMsIExpc3RlbmVyIGV0Yy5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB0YXJnZXRTY29wZTogQ29uc3RydWN0O1xuXG4gIC8qKlxuICAgKiBAaW5oZXJpdGRvY1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IGFsbG93TGlzdGVuZXJQb3J0RnJvbShvdGhlcjogSUNvbm5lY3RhYmxlKTogdm9pZDtcblxuICAvKipcbiAgICogQGluaGVyaXRkb2NcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBhbGxvd0xpc3RlbmVyUG9ydFRvKG90aGVyOiBJQ29ubmVjdGFibGUpOiB2b2lkO1xufVxuXG4vKipcbiAqIFRoaXMgY29uc3RydWN0IHJlcGVyZXNlbnRzIGEgZmxlZXQgb2YgRGVhZGxpbmUgV29ya2Vycy5cbiAqXG4gKiBUaGUgY29uc3RydWN0IGNvbnNpc3RzIG9mIGFuIEF1dG8gU2NhbGluZyBHcm91cCAoQVNHKSBvZiBpbnN0YW5jZXMgdXNpbmcgYSBwcm92aWRlZCBBTUkgd2hpY2ggaGFzIERlYWRsaW5lIGFuZCBhbnkgbnVtYmVyXG4gKiBvZiByZW5kZXIgYXBwbGljYXRpb25zIGluc3RhbGxlZC4gIFdoZW5ldmVyIGFuIGluc3RhbmNlIGluIHRoZSBBU0cgc3RhcnQgaXQgd2lsbCBjb25uZWN0IERlYWRsaW5lIHRvIHRoZSBkZXNpcmVkIHJlbmRlciBxdWV1ZS5cbiAqXG4gKiBXaGVuIHRoZSB3b3JrZXIgZmxlZXQgaXMgZGVwbG95ZWQgaWYgaXQgaGFzIGJlZW4gcHJvdmlkZWQgYSBIZWFsdGhNb25pdG9yIHRoZSBXb3JrZXIgZmxlZXQgd2lsbCByZWdpc3RlciBpdHNlbGYgYWdhaW5zdCB0aGUgTW9uaXRvclxuICogdG8gZW5zdXJlIHRoYXQgdGhlIGZsZWV0IHJlbWFpbnMgaGVhbHRoeS5cbiAqXG4gKiAhW2FyY2hpdGVjdHVyZSBkaWFncmFtXSgvZGlhZ3JhbXMvZGVhZGxpbmUvV29ya2VySW5zdGFuY2VGbGVldC5zdmcpXG4gKlxuICogUmVzb3VyY2VzIERlcGxveWVkXG4gKiAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqIC0gQW4gRUMyIEF1dG8gU2NhbGluZyBHcm91cCB0byBtYWludGFpbiB0aGUgbnVtYmVyIG9mIGluc3RhbmNlcy5cbiAqIC0gQW4gSW5zdGFuY2UgUm9sZSBhbmQgY29ycmVzcG9uZGluZyBJQU0gUG9saWN5LlxuICogLSBBbiBBbWF6b24gQ2xvdWRXYXRjaCBsb2cgZ3JvdXAgdGhhdCBjb250YWlucyB0aGUgRGVhZGxpbmUgV29ya2VyLCBEZWFkbGluZSBMYXVuY2hlciwgYW5kIGluc3RhbmNlLXN0YXJ0dXAgbG9ncyBmb3IgdGhlIGluc3RhbmNlc1xuICogICBpbiB0aGUgZmxlZXQuXG4gKlxuICogU2VjdXJpdHkgQ29uc2lkZXJhdGlvbnNcbiAqIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICogLSBUaGUgaW5zdGFuY2VzIGRlcGxveWVkIGJ5IHRoaXMgY29uc3RydWN0IGRvd25sb2FkIGFuZCBydW4gc2NyaXB0cyBmcm9tIHlvdXIgQ0RLIGJvb3RzdHJhcCBidWNrZXQgd2hlbiB0aGF0IGluc3RhbmNlXG4gKiAgIGlzIGxhdW5jaGVkLiBZb3UgbXVzdCBsaW1pdCB3cml0ZSBhY2Nlc3MgdG8geW91ciBDREsgYm9vdHN0cmFwIGJ1Y2tldCB0byBwcmV2ZW50IGFuIGF0dGFja2VyIGZyb20gbW9kaWZ5aW5nIHRoZSBhY3Rpb25zXG4gKiAgIHBlcmZvcm1lZCBieSB0aGVzZSBzY3JpcHRzLiBXZSBzdHJvbmdseSByZWNvbW1lbmQgdGhhdCB5b3UgZWl0aGVyIGVuYWJsZSBBbWF6b24gUzMgc2VydmVyIGFjY2VzcyBsb2dnaW5nIG9uIHlvdXIgQ0RLXG4gKiAgIGJvb3RzdHJhcCBidWNrZXQsIG9yIGVuYWJsZSBBV1MgQ2xvdWRUcmFpbCBvbiB5b3VyIGFjY291bnQgdG8gYXNzaXN0IGluIHBvc3QtaW5jaWRlbnQgYW5hbHlzaXMgb2YgY29tcHJvbWlzZWQgcHJvZHVjdGlvblxuICogICBlbnZpcm9ubWVudHMuXG4gKiAtIFRoZSBkYXRhIHRoYXQgaXMgc3RvcmVkIG9uIHlvdXIgV29ya2VyJ3MgbG9jYWwgRUJTIHZvbHVtZSBjYW4gaW5jbHVkZSB0ZW1wb3Jhcnkgd29ya2luZyBmaWxlcyBmcm9tIHRoZSBhcHBsaWNhdGlvbnNcbiAqICAgdGhhdCBhcmUgcmVuZGVyaW5nIHlvdXIgam9icyBhbmQgdGFza3MuIFRoYXQgZGF0YSBjYW4gYmUgc2Vuc2l0aXZlIG9yIHByaXZpbGVnZWQsIHNvIHdlIHJlY29tbWVuZCB0aGF0IHlvdSBlbmNyeXB0XG4gKiAgIHRoZSBkYXRhIHZvbHVtZXMgb2YgdGhlc2UgaW5zdGFuY2VzIHVzaW5nIGVpdGhlciB0aGUgcHJvdmlkZWQgb3B0aW9uIG9yIGJ5IHVzaW5nIGFuIGVuY3J5cHRlZCBBTUkgYXMgeW91ciBzb3VyY2UuXG4gKiAtIFRoZSBzb2Z0d2FyZSBvbiB0aGUgQU1JIHRoYXQgaXMgYmVpbmcgdXNlZCBieSB0aGlzIGNvbnN0cnVjdCBtYXkgcG9zZSBhIHNlY3VyaXR5IHJpc2suIFdlIHJlY29tbWVuZCB0aGF0IHlvdSBhZG9wdCBhXG4gKiAgIHBhdGNoaW5nIHN0cmF0ZWd5IHRvIGtlZXAgdGhpcyBzb2Z0d2FyZSBjdXJyZW50IHdpdGggdGhlIGxhdGVzdCBzZWN1cml0eSBwYXRjaGVzLiBQbGVhc2Ugc2VlXG4gKiAgIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9yZmRrL2xhdGVzdC9ndWlkZS9wYXRjaGluZy1zb2Z0d2FyZS5odG1sIGZvciBtb3JlIGluZm9ybWF0aW9uLlxuICovXG5leHBvcnQgY2xhc3MgV29ya2VySW5zdGFuY2VGbGVldCBleHRlbmRzIFdvcmtlckluc3RhbmNlRmxlZXRCYXNlIHtcbiAgLyoqXG4gICAqIFRoZSBtaW4gbGltaXQgZm9yIHNwb3QgcHJpY2UuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNQT1RfUFJJQ0VfTUlOX0xJTUlUID0gMC4wMDE7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXggbGltaXQgZm9yIHNwb3QgcHJpY2UuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNQT1RfUFJJQ0VfTUFYX0xJTUlUID0gMjU1O1xuXG4gIC8qKlxuICAgKiBUaGlzIGRldGVybWluZXMgd29ya2VyJ3MgaGVhbHRoIGJhc2VkIG9uIGFueSBoYXJkd2FyZSBvciBzb2Z0d2FyZSBpc3N1ZXMgb2YgRUMyIGluc3RhbmNlLlxuICAgKiBSZXNvdXJjZSBUcmFja2VyIGRvZXMgZGVlcCBwaW5nIGV2ZXJ5IDUgbWludXRlcy4gVGhlc2UgY2hlY2tzIHNob3VsZCBiZSBtb3JlIGZyZXF1ZW50IHNvXG4gICAqIHRoYXQgYW55IEVDMiBsZXZlbCBpc3N1ZXMgYXJlIGlkZW50aWZpZWQgQVNBUC4gSGVuY2Ugc2V0dGluZyBpdCB0byAxIG1pbnV0ZS5cbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfSEVBTFRIX0NIRUNLX0lOVEVSVkFMID0gRHVyYXRpb24ubWludXRlcygxKTtcblxuICAvKipcbiAgICogRGVmYXVsdCBwcmVmaXggZm9yIGEgTG9nR3JvdXAgaWYgb25lIGlzbid0IHByb3ZpZGVkIGluIHRoZSBwcm9wcy5cbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfTE9HX0dST1VQX1BSRUZJWDogc3RyaW5nID0gJy9yZW5kZXJmYXJtLyc7XG5cbiAgLyoqXG4gICAqIFRoaXMgaXMgdGhlIGN1cnJlbnQgbWF4aW11bSBmb3IgbnVtYmVyIG9mIHdvcmtlcnMgdGhhdCBjYW4gYmUgc3RhcnRlZCBvbiBhIHNpbmdsZSBob3N0LiBDdXJyZW50bHkgdGhlXG4gICAqIG9ubHkgdGhpbmcgdXNpbmcgdGhpcyBsaW1pdCBpcyB0aGUgY29uZmlndXJhdGlvbiBvZiB0aGUgbGlzdGVuZXIgcG9ydHMuIE1vcmUgdGhhbiA4IHdvcmtlcnMgY2FuIGJlIHN0YXJ0ZWQsXG4gICAqIGJ1dCBvbmx5IHRoZSBmaXJzdCA4IHdpbGwgaGF2ZSB0aGVpciBwb3J0cyBvcGVuZWQgaW4gdGhlIHdvcmtlcnMnIHNlY3VyaXR5IGdyb3VwLlxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgTUFYX1dPUktFUlNfUEVSX0hPU1QgPSA4O1xuXG4gIC8qKlxuICAgKiBTZXR0aW5nIHRoZSBkZWZhdWx0IHNpZ25hbCB0aW1lb3V0IHRvIDE1IG1pbi4gVGhpcyBpcyB0aGUgbWF4IHRpbWUsIGEgc2luZ2xlIGluc3RhbmNlIGlzIGV4cGVjdGVkXG4gICAqIHRvIHRha2UgZm9yIGxhdW5jaCBhbmQgZXhlY3V0ZSB0aGUgdXNlci1kYXRhIGZvciBkZWFkbGluZSB3b3JrZXIgY29uZmlndXJhdGlvbi4gQXMgd2UgYXJlIHNldHRpbmdcbiAgICogZmFpbHVyZSBzaWduYWxzIGluIHRoZSB1c2VyLWRhdGEsIGFueSBmYWlsdXJlIHdpbGwgdGVybWluYXRlIGRlcGxveW1lbnQgaW1tZWRpYXRlbHkuXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBSRVNPVVJDRV9TSUdOQUxfVElNRU9VVCA9IER1cmF0aW9uLm1pbnV0ZXMoMTUpO1xuXG4gIC8qKlxuICAgKiBUaGUgQVNHIG9iamVjdCBjcmVhdGVkIGJ5IHRoZSBjb25zdHJ1Y3QuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZmxlZXQ6IEF1dG9TY2FsaW5nR3JvdXA7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWN1cml0eSBncm91cHMvcnVsZXMgdXNlZCB0byBhbGxvdyBuZXR3b3JrIGNvbm5lY3Rpb25zIHRvIHRoZSBmaWxlIHN5c3RlbS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIFRoZSBwcmluY2lwYWwgdG8gZ3JhbnQgcGVybWlzc2lvbnMgdG8uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IElQcmluY2lwYWw7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0IHdvcmtlcnMgbGlzdGVuIG9uIHRvIHNoYXJlIHRoZWlyIGxvZ3MuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbGlzdGVuaW5nUG9ydHM6IFBvcnQ7XG5cbiAgLyoqXG4gICAqIFRoaXMgZmllbGQgaW1wbGVtZW50cyB0aGUgYmFzZSBjYXBhY2l0eSBtZXRyaWMgb2YgdGhlIGZsZWV0IGFnYWluc3RcbiAgICogd2hpY2gsIHRoZSBoZWFsdGh5IHBlcmNlbnQgd2lsbCBiZSBjYWxjdWxhdGVkLlxuICAgKlxuICAgKiBlZy46IEdyb3VwRGVzaXJlZENhcGFjaXR5IGZvciBhbiBBU0dcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0YXJnZXRDYXBhY2l0eU1ldHJpYzogSU1ldHJpYztcblxuICAvKipcbiAgICogVGhpcyBmaWVsZCBpbXBsZW1lbnRzIHRoZSBjb21wb25lbnQgb2YgdHlwZSBJTmV0d29ya0xvYWRCYWxhbmNlclRhcmdldFxuICAgKiB3aGljaCBjYW4gYmUgYXR0YWNoZWQgdG8gTmV0d29yayBMb2FkIEJhbGFuY2VyIGZvciBtb25pdG9yaW5nLlxuICAgKlxuICAgKiBlZy4gQW4gQXV0b1NjYWxpbmdHcm91cFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhcmdldFRvTW9uaXRvcjogSUFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyVGFyZ2V0O1xuXG4gIC8qKlxuICAgKiBUaGlzIGZpZWxkIGltcGxlbWVudHMgYSBwb2xpY3kgd2hpY2ggY2FuIGJlIGF0dGFjaGVkIHRvIHRoZSBsYW1iZGFcbiAgICogZXhlY3V0aW9uIHJvbGUgc28gdGhhdCBpdCBpcyBjYXBhYmxlIG9mIHN1c3BlbmRpbmcgdGhlIGZsZWV0LlxuICAgKlxuICAgKiBlZy46IGF1dG9zY2FsaW5nOlVwZGF0ZUF1dG9TY2FsaW5nR3JvdXAgcGVybWlzc2lvbiBmb3IgYW4gQVNHXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdGFyZ2V0VXBkYXRlUG9saWN5OiBJUG9saWN5O1xuXG4gIC8qKlxuICAgKiBUaGlzIGZpZWxkIGltcGxlbWVudHMgdGhlIG1heGltdW0gaW5zdGFuY2UgY291bnQgdGhpcyBmbGVldCBjYW4gaGF2ZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0YXJnZXRDYXBhY2l0eTogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGlzIGZpZWxkIGltcGxlbWVudHMgdGhlIHNjb3BlIGluIHdoaWNoIHRvIGNyZWF0ZSB0aGUgbW9uaXRvcmluZyByZXNvdXJjZVxuICAgKiBsaWtlIFRhcmdldEdyb3VwcywgTGlzdGVuZXIgZXRjLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhcmdldFNjb3BlOiBDb25zdHJ1Y3Q7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFdvcmtlckluc3RhbmNlRmxlZXRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnZhbGlkYXRlUHJvcHMocHJvcHMpO1xuXG4gICAgY29uc3QgbWluQ2FwYWNpdHkgPSBwcm9wcy5taW5DYXBhY2l0eSA/PyAxO1xuICAgIGNvbnN0IHNpZ25hbHMgPSBtaW5DYXBhY2l0eSA+IDAgPyBTaWduYWxzLndhaXRGb3JNaW5DYXBhY2l0eSh7XG4gICAgICB0aW1lb3V0OiBXb3JrZXJJbnN0YW5jZUZsZWV0LlJFU09VUkNFX1NJR05BTF9USU1FT1VULFxuICAgIH0pIDogdW5kZWZpbmVkO1xuICAgIGlmIChzaWduYWxzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIEFubm90YXRpb25zLm9mKHRoaXMpLmFkZFdhcm5pbmcoJ0RlcGxveWluZyB3aXRoIDAgbWluaW11bSBjYXBhY2l0eS4gSWYgdGhlcmUgaXMgYW4gZXJyb3IgaW4gdGhlIEVDMiBVc2VyRGF0YSBmb3IgdGhpcyBmbGVldCwgdGhlbiB5b3VyIHN0YWNrIGRlcGxveW1lbnQgd2lsbCBub3QgZmFpbC4gV2F0Y2ggZm9yIGVycm9ycyBpbiB5b3VyIENsb3VkV2F0Y2ggbG9ncy4nKTtcbiAgICB9XG5cbiAgICBjb25zdCB2cGNTdWJuZXRzID0gcHJvcHMudnBjU3VibmV0cyA/IHByb3BzLnZwY1N1Ym5ldHMgOiB7XG4gICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9FR1JFU1MsXG4gICAgfTtcblxuICAgIC8vIExhdW5jaGluZyB0aGUgZmxlZXQgd2l0aCBkZWFkbGluZSB3b3JrZXJzLlxuICAgIHRoaXMuZmxlZXQgPSBuZXcgQXV0b1NjYWxpbmdHcm91cCh0aGlzLCAnRGVmYXVsdCcsIHtcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgaW5zdGFuY2VUeXBlOiAocHJvcHMuaW5zdGFuY2VUeXBlID8gcHJvcHMuaW5zdGFuY2VUeXBlIDogSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDIsIEluc3RhbmNlU2l6ZS5MQVJHRSkpLFxuICAgICAgbWFjaGluZUltYWdlOiBwcm9wcy53b3JrZXJNYWNoaW5lSW1hZ2UsXG4gICAgICBrZXlOYW1lOiBwcm9wcy5rZXlOYW1lLFxuICAgICAgdnBjU3VibmV0cyxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IHByb3BzLnNlY3VyaXR5R3JvdXAsXG4gICAgICBtaW5DYXBhY2l0eSxcbiAgICAgIG1heENhcGFjaXR5OiBwcm9wcy5tYXhDYXBhY2l0eSxcbiAgICAgIGRlc2lyZWRDYXBhY2l0eTogcHJvcHMuZGVzaXJlZENhcGFjaXR5LFxuICAgICAgc2lnbmFscyxcbiAgICAgIGhlYWx0aENoZWNrOiBIZWFsdGhDaGVjay5lbGIoe1xuICAgICAgICBncmFjZTogV29ya2VySW5zdGFuY2VGbGVldC5ERUZBVUxUX0hFQUxUSF9DSEVDS19JTlRFUlZBTCxcbiAgICAgIH0pLFxuICAgICAgcm9sZTogcHJvcHMucm9sZSxcbiAgICAgIHNwb3RQcmljZTogcHJvcHMuc3BvdFByaWNlPy50b1N0cmluZygpLFxuICAgICAgYmxvY2tEZXZpY2VzOiBwcm9wcy5ibG9ja0RldmljZXMsXG4gICAgICB1c2VyRGF0YTogcHJvcHMudXNlckRhdGEsXG4gICAgfSk7XG5cbiAgICB0aGlzLnRhcmdldENhcGFjaXR5ID0gcGFyc2VJbnQoKHRoaXMuZmxlZXQubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuQXV0b1NjYWxpbmdHcm91cCkubWF4U2l6ZSwgMTApO1xuICAgIHRoaXMudGFyZ2V0U2NvcGUgPSB0aGlzO1xuICAgIHRoaXMudGFyZ2V0VG9Nb25pdG9yID0gdGhpcy5mbGVldDtcbiAgICB0aGlzLnRhcmdldENhcGFjaXR5TWV0cmljID0gbmV3IE1ldHJpYyh7XG4gICAgICBuYW1lc3BhY2U6ICdBV1MvQXV0b1NjYWxpbmcnLFxuICAgICAgbWV0cmljTmFtZTogJ0dyb3VwRGVzaXJlZENhcGFjaXR5JyxcbiAgICAgIGRpbWVuc2lvbnNNYXA6IHtcbiAgICAgICAgQXV0b1NjYWxpbmdHcm91cE5hbWU6IHRoaXMuZmxlZXQuYXV0b1NjYWxpbmdHcm91cE5hbWUsXG4gICAgICB9LFxuICAgICAgbGFiZWw6ICdHcm91cERlc2lyZWRDYXBhY2l0eScsXG4gICAgfSk7XG4gICAgdGhpcy50YXJnZXRVcGRhdGVQb2xpY3kgPSBuZXcgUG9saWN5KHRoaXMsICdBU0dVcGRhdGVQb2xpY3knLCB7XG4gICAgICBzdGF0ZW1lbnRzOiBbbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnYXV0b3NjYWxpbmc6VXBkYXRlQXV0b1NjYWxpbmdHcm91cCddLFxuICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmZsZWV0LmF1dG9TY2FsaW5nR3JvdXBBcm5dLFxuICAgICAgfSldLFxuICAgIH0pO1xuXG4gICAgKHRoaXMuZmxlZXQubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuQXV0b1NjYWxpbmdHcm91cCkubWV0cmljc0NvbGxlY3Rpb24gPSBbe1xuICAgICAgZ3JhbnVsYXJpdHk6ICcxTWludXRlJyxcbiAgICAgIG1ldHJpY3M6IFsnR3JvdXBEZXNpcmVkQ2FwYWNpdHknXSxcbiAgICB9XTtcblxuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLmZsZWV0LmdyYW50UHJpbmNpcGFsO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSB0aGlzLmZsZWV0LmNvbm5lY3Rpb25zO1xuXG4gICAgLy8gQ29uZmlndXJlIHRoZSBoZWFsdGggbW9uaXRvcmluZyBpZiBwcm92aWRlZC5cbiAgICAvLyBOb3RlOiBUaGlzIG11c3QgYmUgZG9uZSAqQkVGT1JFKiBjb25maWd1cmluZyB0aGUgd29ya2VyLiBXZSByZWx5IG9uIHRoZSB3b3JrZXIgY29uZmlndXJhdGlvblxuICAgIC8vIHNjcmlwdCByZXN0YXJ0aW5nIHRoZSBsYXVuY2hlci5cbiAgICB0aGlzLmNvbmZpZ3VyZUhlYWx0aE1vbml0b3IocHJvcHMpO1xuXG4gICAgY29uc3Qgd29ya2VyQ29uZmlnID0gbmV3IFdvcmtlckluc3RhbmNlQ29uZmlndXJhdGlvbih0aGlzLCBpZCwge1xuICAgICAgd29ya2VyOiB0aGlzLmZsZWV0LFxuICAgICAgY2xvdWRXYXRjaExvZ1NldHRpbmdzOiB7XG4gICAgICAgIGxvZ0dyb3VwUHJlZml4OiBXb3JrZXJJbnN0YW5jZUZsZWV0LkRFRkFVTFRfTE9HX0dST1VQX1BSRUZJWCxcbiAgICAgICAgLi4ucHJvcHMubG9nR3JvdXBQcm9wcyxcbiAgICAgIH0sXG4gICAgICByZW5kZXJRdWV1ZTogcHJvcHMucmVuZGVyUXVldWUsXG4gICAgICB3b3JrZXJTZXR0aW5nczogcHJvcHMsXG4gICAgICB1c2VyRGF0YVByb3ZpZGVyOiBwcm9wcy51c2VyRGF0YVByb3ZpZGVyLFxuICAgIH0pO1xuICAgIHRoaXMubGlzdGVuaW5nUG9ydHMgPSBQb3J0LnRjcFJhbmdlKFxuICAgICAgd29ya2VyQ29uZmlnLmxpc3RlbmVyUG9ydCxcbiAgICAgIHdvcmtlckNvbmZpZy5saXN0ZW5lclBvcnQgKyBXb3JrZXJJbnN0YW5jZUZsZWV0Lk1BWF9XT1JLRVJTX1BFUl9IT1NULFxuICAgICk7XG5cbiAgICBpZiAocHJvcHMucmVuZGVyUXVldWUucmVwb3NpdG9yeS5zZWNyZXRzTWFuYWdlbWVudFNldHRpbmdzLmVuYWJsZWQpIHtcbiAgICAgIGlmICghcHJvcHMudnBjU3VibmV0cykge1xuICAgICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKFxuICAgICAgICAgICdEZWFkbGluZSBTZWNyZXRzIE1hbmFnZW1lbnQgaXMgZW5hYmxlZCBvbiB0aGUgUmVwb3NpdG9yeSBhbmQgVlBDIHN1Ym5ldHMgaGF2ZSBub3QgYmVlbiBzdXBwbGllZC4gVXNpbmcgZGVkaWNhdGVkIHN1Ym5ldHMgaXMgcmVjb21tZW5kZWQuIFNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1yZmRrL2Jsb2JzL3JlbGVhc2UvcGFja2FnZXMvYXdzLXJmZGsvbGliL2RlYWRsaW5lL1JFQURNRS5tZCN1c2luZy1kZWRpY2F0ZWQtc3VibmV0cy1mb3ItZGVhZGxpbmUtY29tcG9uZW50cycsXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICBwcm9wcy5yZW5kZXJRdWV1ZS5jb25maWd1cmVTZWNyZXRzTWFuYWdlbWVudEF1dG9SZWdpc3RyYXRpb24oe1xuICAgICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgICAgdnBjU3VibmV0cyxcbiAgICAgICAgcm9sZTogU2VjcmV0c01hbmFnZW1lbnRSb2xlLkNMSUVOVCxcbiAgICAgICAgcmVnaXN0cmF0aW9uU3RhdHVzOiBTZWNyZXRzTWFuYWdlbWVudFJlZ2lzdHJhdGlvblN0YXR1cy5SRUdJU1RFUkVELFxuICAgICAgICBkZXBlbmRlbnQ6IHRoaXMuZmxlZXQsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBVcGRhdGluZyB0aGUgdXNlciBkYXRhIHdpdGggc3VjY2Vzc2Z1bCBjZm4tc2lnbmFsIGNvbW1hbmRzLlxuICAgIGlmIChzaWduYWxzKSB7XG4gICAgICB0aGlzLmZsZWV0LnVzZXJEYXRhLmFkZFNpZ25hbE9uRXhpdENvbW1hbmQodGhpcy5mbGVldCk7XG4gICAgfVxuXG4gICAgLy8gVGFnIGRlcGxveWVkIHJlc291cmNlcyB3aXRoIFJGREsgbWV0YS1kYXRhXG4gICAgdGFnQ29uc3RydWN0KHRoaXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCB0aGUgc2VjdXJpdHkgZ3JvdXAgdG8gYWxsIHdvcmtlcnNcbiAgICpcbiAgICogQHBhcmFtIHNlY3VyaXR5R3JvdXA6IFRoZSBzZWN1cml0eSBncm91cCB0byBhZGRcbiAgICovXG4gIHB1YmxpYyBhZGRTZWN1cml0eUdyb3VwKHNlY3VyaXR5R3JvdXA6IElTZWN1cml0eUdyb3VwKTogdm9pZCB7XG4gICAgdGhpcy5mbGVldC5hZGRTZWN1cml0eUdyb3VwKHNlY3VyaXR5R3JvdXApO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwdWJsaWMgYWxsb3dMaXN0ZW5lclBvcnRGcm9tKG90aGVyOiBJQ29ubmVjdGFibGUpOiB2b2lkIHtcbiAgICB0aGlzLmNvbm5lY3Rpb25zLmFsbG93RnJvbShvdGhlci5jb25uZWN0aW9ucywgdGhpcy5saXN0ZW5pbmdQb3J0cywgJ1dvcmtlciByZW1vdGUgY29tbWFuZCBsaXN0ZW5pbmcgcG9ydCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0ZG9jXG4gICAqL1xuICBwdWJsaWMgYWxsb3dMaXN0ZW5lclBvcnRUbyhvdGhlcjogSUNvbm5lY3RhYmxlKTogdm9pZCB7XG4gICAgb3RoZXIuY29ubmVjdGlvbnMuYWxsb3dUbyh0aGlzLmNvbm5lY3Rpb25zLCB0aGlzLmxpc3RlbmluZ1BvcnRzLCAnV29ya2VyIHJlbW90ZSBjb21tYW5kIGxpc3RlbmluZyBwb3J0Jyk7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlUHJvcHMocHJvcHM6IFdvcmtlckluc3RhbmNlRmxlZXRQcm9wcykge1xuICAgIHRoaXMudmFsaWRhdGVTcG90UHJpY2UocHJvcHMuc3BvdFByaWNlKTtcbiAgICB0aGlzLnZhbGlkYXRlQXJyYXlHcm91cHNQb29sc1N5bnRheChwcm9wcy5ncm91cHMsIC9eKD8hbm9uZSQpW2EtekEtWjAtOS1fXSskL2ksICdncm91cHMnKTtcbiAgICB0aGlzLnZhbGlkYXRlQXJyYXlHcm91cHNQb29sc1N5bnRheChwcm9wcy5wb29scywgL14oPyFub25lJClbYS16QS1aMC05LV9dKyQvaSwgJ3Bvb2xzJyk7XG4gICAgdGhpcy52YWxpZGF0ZVJlZ2lvbihwcm9wcy5yZWdpb24sIC9eKD8hbm9uZSR8YWxsJHx1bnJlY29nbml6ZWQkKVthLXpBLVowLTktX10rJC9pKTtcbiAgICB0aGlzLnZhbGlkYXRlQmxvY2tEZXZpY2VzKHByb3BzLmJsb2NrRGV2aWNlcyk7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlU3BvdFByaWNlKHNwb3RQcmljZTogbnVtYmVyIHwgdW5kZWZpbmVkKSB7XG4gICAgaWYgKHNwb3RQcmljZSAmJiAhKHNwb3RQcmljZSA+PSBXb3JrZXJJbnN0YW5jZUZsZWV0LlNQT1RfUFJJQ0VfTUlOX0xJTUlUICYmIHNwb3RQcmljZSA8PSBXb3JrZXJJbnN0YW5jZUZsZWV0LlNQT1RfUFJJQ0VfTUFYX0xJTUlUKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIHZhbHVlOiAke3Nwb3RQcmljZX0gZm9yIHByb3BlcnR5ICdzcG90UHJpY2UnLiBWYWxpZCB2YWx1ZXMgY2FuIGJlIGFueSBkZWNpbWFsIGJldHdlZW4gJHtXb3JrZXJJbnN0YW5jZUZsZWV0LlNQT1RfUFJJQ0VfTUlOX0xJTUlUfSBhbmQgJHtXb3JrZXJJbnN0YW5jZUZsZWV0LlNQT1RfUFJJQ0VfTUFYX0xJTUlUfS5gKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlUmVnaW9uKHJlZ2lvbjogc3RyaW5nIHwgdW5kZWZpbmVkLCByZWdleDogUmVnRXhwKSB7XG4gICAgaWYgKHJlZ2lvbiAmJiAhcmVnZXgudGVzdChyZWdpb24pKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgdmFsdWU6ICR7cmVnaW9ufSBmb3IgcHJvcGVydHkgJ3JlZ2lvbicuIFZhbGlkIGNoYXJhY3RlcnMgYXJlIEEtWiwgYS16LCAwLTksIC0gYW5kIF8uICdBbGwnLCAnbm9uZScgYW5kICd1bnJlY29nbml6ZWQnIGFyZSByZXNlcnZlZCBuYW1lcyB0aGF0IGNhbm5vdCBiZSB1c2VkLmApO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVBcnJheUdyb3Vwc1Bvb2xzU3ludGF4KGFycmF5OiBzdHJpbmdbXSB8IHVuZGVmaW5lZCwgcmVnZXg6IFJlZ0V4cCwgcHJvcGVydHk6IHN0cmluZykge1xuICAgIGlmIChhcnJheSkge1xuICAgICAgYXJyYXkuZm9yRWFjaCh2YWx1ZSA9PiB7XG4gICAgICAgIGlmICghcmVnZXgudGVzdCh2YWx1ZSkpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgdmFsdWU6ICR7dmFsdWV9IGZvciBwcm9wZXJ0eSAnJHtwcm9wZXJ0eX0nLiBWYWxpZCBjaGFyYWN0ZXJzIGFyZSBBLVosIGEteiwgMC05LCAtIGFuZCBfLiBBbHNvLCBncm91cCAnbm9uZScgaXMgcmVzZXJ2ZWQgYXMgdGhlIGRlZmF1bHQgZ3JvdXAuYCk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVCbG9ja0RldmljZXMoYmxvY2tEZXZpY2VzOiBCbG9ja0RldmljZVtdIHwgdW5kZWZpbmVkKSB7XG4gICAgaWYgKGJsb2NrRGV2aWNlcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKGBUaGUgd29ya2VyLWZsZWV0ICR7dGhpcy5ub2RlLmlkfSBpcyBiZWluZyBjcmVhdGVkIHdpdGhvdXQgYmVpbmcgcHJvdmlkZWQgYW55IGJsb2NrIGRldmljZXMgc28gdGhlIFNvdXJjZSBBTUkncyBkZXZpY2VzIHdpbGwgYmUgdXNlZC4gYCArXG4gICAgICAgICdXb3JrZXJzIGNhbiBoYXZlIGFjY2VzcyB0byBzZW5zaXRpdmUgZGF0YSBzbyBpdCBpcyByZWNvbW1lbmRlZCB0byBlaXRoZXIgZXhwbGljaXRseSBlbmNyeXB0IHRoZSBkZXZpY2VzIG9uIHRoZSB3b3JrZXIgZmxlZXQgb3IgdG8gZW5zdXJlIHRoZSBzb3VyY2UgQU1JXFwncyBEcml2ZXMgYXJlIGVuY3J5cHRlZC4nKTtcbiAgICB9IGVsc2Uge1xuICAgICAgYmxvY2tEZXZpY2VzLmZvckVhY2goZGV2aWNlID0+IHtcbiAgICAgICAgaWYgKGRldmljZS52b2x1bWUuZWJzRGV2aWNlID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAvLyBTdXBwcmVzc2VkIG9yIEVwaGVtZXJhbCBCbG9jayBEZXZpY2VcbiAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cblxuICAgICAgICAvLyBlbmNyeXB0ZWQgaXMgbm90IGV4cG9zZWQgYXMgcGFydCBvZiBlYnNEZXZpY2VQcm9wcyBzbyB3ZSBuZWVkIHRvIGNvbmZpcm0gaXQgZXhpc3RzIHRoZW4gYWNjZXNzIGl0IHZpYSBbXS5cbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGRvdC1ub3RhdGlvblxuICAgICAgICBpZiAoICgnZW5jcnlwdGVkJyBpbiBkZXZpY2Uudm9sdW1lLmVic0RldmljZSA9PT0gZmFsc2UpIHx8ICgnZW5jcnlwdGVkJyBpbiBkZXZpY2Uudm9sdW1lLmVic0RldmljZSAmJiAhZGV2aWNlLnZvbHVtZS5lYnNEZXZpY2VbJ2VuY3J5cHRlZCddICkgKSB7XG4gICAgICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkV2FybmluZyhgVGhlIEJsb2NrRGV2aWNlIFwiJHtkZXZpY2UuZGV2aWNlTmFtZX1cIiBvbiB0aGUgd29ya2VyLWZsZWV0ICR7dGhpcy5ub2RlLmlkfSBpcyBub3QgZW5jcnlwdGVkLiBgICtcbiAgICAgICAgICAgICAgJ1dvcmtlcnMgY2FuIGhhdmUgYWNjZXNzIHRvIHNlbnNpdGl2ZSBkYXRhIHNvIGl0IGlzIHJlY29tbWVuZGVkIHRvIGVuY3J5cHQgdGhlIGRldmljZXMgb24gdGhlIHdvcmtlciBmbGVldC4nKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVIZWFsdGhNb25pdG9yKHByb3BzOiBXb3JrZXJJbnN0YW5jZUZsZWV0UHJvcHMpIHtcbiAgICBpZiAocHJvcHMuaGVhbHRoTW9uaXRvcikge1xuICAgICAgY29uc3QgaGVhbHRoQ2hlY2tQb3J0ID0gcHJvcHMuaGVhbHRoQ2hlY2tDb25maWc/LnBvcnQgPz8gSGVhbHRoTW9uaXRvci5ERUZBVUxUX0hFQUxUSF9DSEVDS19QT1JUO1xuXG4gICAgICBjb25zdCBjb25maWd1cmVIZWFsdGhNb25pdG9yU2NyaXB0QXNzZXQgPSBTY3JpcHRBc3NldC5mcm9tUGF0aENvbnZlbnRpb24odGhpcywgJ1dvcmtlckNvbmZpZ3VyYXRpb25TY3JpcHQnLCB7XG4gICAgICAgIG9zVHlwZTogdGhpcy5mbGVldC5vc1R5cGUsXG4gICAgICAgIGJhc2VOYW1lOiAnY29uZmlndXJlV29ya2VySGVhbHRoQ2hlY2snLFxuICAgICAgICByb290RGlyOiBwYXRoLmpvaW4oXG4gICAgICAgICAgX19kaXJuYW1lLFxuICAgICAgICAgICcuLicsXG4gICAgICAgICAgJ3NjcmlwdHMvJyxcbiAgICAgICAgKSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25maWd1cmVIZWFsdGhNb25pdG9yU2NyaXB0QXNzZXQuZXhlY3V0ZU9uKHtcbiAgICAgICAgaG9zdDogdGhpcy5mbGVldCxcbiAgICAgICAgYXJnczogW1xuICAgICAgICAgIGAnJHtoZWFsdGhDaGVja1BvcnR9J2AsXG4gICAgICAgICAgYCcke1ZlcnNpb24uTUlOSU1VTV9TVVBQT1JURURfREVBRExJTkVfVkVSU0lPTi50b1N0cmluZygpfSdgLFxuICAgICAgICBdLFxuICAgICAgfSk7XG5cbiAgICAgIHByb3BzLmhlYWx0aE1vbml0b3IucmVnaXN0ZXJGbGVldCh0aGlzLCBwcm9wcy5oZWFsdGhDaGVja0NvbmZpZyB8fCB7XG4gICAgICAgIHBvcnQ6IGhlYWx0aENoZWNrUG9ydCxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKGBUaGUgd29ya2VyLWZsZWV0ICR7dGhpcy5ub2RlLmlkfSBpcyBiZWluZyBjcmVhdGVkIHdpdGhvdXQgYSBoZWFsdGggbW9uaXRvciBhdHRhY2hlZCB0byBpdC4gVGhpcyBtZWFucyB0aGF0IHRoZSBmbGVldCB3aWxsIG5vdCBhdXRvbWF0aWNhbGx5IHNjYWxlLWluIHRvIDAgaWYgdGhlIHdvcmtlcnMgYXJlIHVuaGVhbHRoeS5gKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==