"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Version = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
  * This class is reposonsible to do basic operations on version format.
  */
class Version {
    constructor(components) {
        if (components.length != 4) {
            throw new Error('Invalid version format. Version should contain exactly 4 components.');
        }
        components.forEach((component) => {
            if (component < 0) {
                throw new RangeError('Invalid version format. None of the version components can be negative.');
            }
            if (!Number.isInteger(component)) {
                throw new RangeError('Invalid version format. None of the version components can contain decimal values.');
            }
        });
        this.components = components;
    }
    /**
     * This method parses the input string and returns the version object.
     *
     * @param version version string to parse
     */
    static parse(version) {
        if (!Version.validateVersionFormat(version)) {
            throw new TypeError(`Invalid version format. Expected format 'a.b.c.d', found '${version}'`);
        }
        return new Version(version.split('.').map(x => parseInt(x)));
    }
    /**
     * This method validates the given string for a sequence '.' separated numbers.
     *
     * @param version the string to be validated.
     *
     * @returns true if the format is correct, else false.
     */
    static validateVersionFormat(version) {
        /**
         * Regex: ^\d+(?:\.\d+){3}$
         * Matches a sequence of '.' separated numbers with exactly 4 digits.
         * - ^ asserts position at start of a line.
         * - \d+ Matches one or more digits.
         * - (?:\.\d+) Matches a dot and the following one or more digits.
         * - {3} Matches previous pattern exactly 3 times.
         * - $ asserts position at the end of a line
         */
        if (version.match(/^\d+(?:\.\d+){3}$/)) {
            return true;
        }
        return false;
    }
    /**
     * @inheritdoc
     */
    get majorVersion() {
        return this.components[0];
    }
    /**
     * @inheritdoc
     */
    get minorVersion() {
        return this.components[1];
    }
    /**
     * @inheritdoc
     */
    get releaseVersion() {
        return this.components[2];
    }
    /**
     * @inheritdoc
     */
    get patchVersion() {
        return this.components[3];
    }
    /**
     * This method compares two version strings
     *
     * @param version
     *
     * @returns true if this version is greater than the provided version;
     * false if this version is less than or equal to the provided verison.
     */
    isGreaterThan(version) {
        return this.compare(version) > 0;
    }
    /**
     * This method compares two version strings
     *
     * @param version
     *
     * @returns true if this version is less than the provided version;
     * false if this version is greater than or equal to the provided verison.
     */
    isLessThan(version) {
        return this.compare(version) < 0;
    }
    /**
     * This method compares two version strings
     *
     * @param version
     *
     * @returns true if this version is equal to the provided version;
     * false otherwise.
     */
    isEqual(version) {
        return this.compare(version) == 0;
    }
    /**
     * The method returns the version components in dot separated string format.
     */
    toString() {
        return this.components.join('.');
    }
    /**
     * @inheritdoc
     */
    get versionString() {
        return this.toString();
    }
    /**
     * This method compares 2 versions.
     *
     * @param version version to compare
     *
     * @returns negative value if this version is less than the provided version;
     * 0 if both the versions are equal;
     * positive value if this version is greater than the provided verison.
     */
    compare(version) {
        if (this.components.length != version.components.length) {
            throw new TypeError('Component count in both the versions should be same.');
        }
        for (let i = 0; i < version.components.length; i++) {
            const diff = this.components[i] - version.components[i];
            if (diff != 0) {
                return diff;
            }
        }
        return 0;
    }
}
exports.Version = Version;
_a = JSII_RTTI_SYMBOL_1;
Version[_a] = { fqn: "aws-rfdk.deadline.Version", version: "1.1.0" };
/**
 * This variable holds the value for minimum supported deadline version.
 */
Version.MINIMUM_SUPPORTED_DEADLINE_VERSION = new Version([10, 1, 9, 2]);
/**
 * The minimum Deadline version required to enable Deadline Secrets Management.
 */
Version.MINIMUM_SECRETS_MANAGEMENT_VERSION = new Version([10, 1, 19, 0]);
/**
 * The minimum Deadline version which uses Install Builder instead of Bitrock Installer.
 */
Version.MINIMUM_VERSION_USING_NEW_INSTALLBUILDER_LOG = new Version([10, 2, 0, 0]);
//# sourceMappingURL=data:application/json;base64,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