"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsageBasedLicensing = exports.UsageBasedLicense = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const constructs_1 = require("constructs");
const _1 = require(".");
const core_1 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 * Instances of this class represent a usage-based license for a particular product.
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 */
class UsageBasedLicense {
    constructor(props) {
        this.licenseName = props.licenseName;
        this.ports = props.ports;
        this.limit = props.limit;
    }
    /**
     * Method for 3dsMax license limit.
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static for3dsMax(limit) {
        return new UsageBasedLicense({
            licenseName: 'max',
            ports: [aws_ec2_1.Port.tcp(27002)],
            limit,
        });
    }
    /**
     * Method for Arnold license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forArnold(limit) {
        return new UsageBasedLicense({
            licenseName: 'arnold',
            ports: [aws_ec2_1.Port.tcp(5056), aws_ec2_1.Port.tcp(7056)],
            limit,
        });
    }
    /**
     * Method for Cinema 4D license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forCinema4D(limit) {
        return new UsageBasedLicense({
            licenseName: 'cinema4d',
            ports: [aws_ec2_1.Port.tcp(5057), aws_ec2_1.Port.tcp(7057)],
            limit,
        });
    }
    /**
     * Method for Clarisse license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forClarisse(limit) {
        return new UsageBasedLicense({
            licenseName: 'clarisse',
            ports: [aws_ec2_1.Port.tcp(40500)],
            limit,
        });
    }
    /**
     * Method for Houdini license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forHoudini(limit) {
        return new UsageBasedLicense({
            licenseName: 'houdini',
            ports: [aws_ec2_1.Port.tcp(1715)],
            limit,
        });
    }
    /**
     * Method for Katana license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKatana(limit) {
        return new UsageBasedLicense({
            licenseName: 'katana',
            ports: [aws_ec2_1.Port.tcp(4151), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for KeyShot license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKeyShot(limit) {
        return new UsageBasedLicense({
            licenseName: 'keyshot',
            ports: [aws_ec2_1.Port.tcp(27003), aws_ec2_1.Port.tcp(2703)],
            limit,
        });
    }
    /**
     * Method for krakatoa license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKrakatoa(limit) {
        return new UsageBasedLicense({
            licenseName: 'krakatoa',
            ports: [aws_ec2_1.Port.tcp(27000), aws_ec2_1.Port.tcp(2700)],
            limit,
        });
    }
    /**
     * Method for Mantra license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMantra(limit) {
        return new UsageBasedLicense({
            licenseName: 'mantra',
            ports: [aws_ec2_1.Port.tcp(1716)],
            limit,
        });
    }
    /**
     * Method for maxwell license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaxwell(limit) {
        return new UsageBasedLicense({
            licenseName: 'maxwell',
            ports: [aws_ec2_1.Port.tcp(5555), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for Maya license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaya(limit) {
        return new UsageBasedLicense({
            licenseName: 'maya',
            ports: [aws_ec2_1.Port.tcp(27002), aws_ec2_1.Port.tcp(2702)],
            limit,
        });
    }
    /**
     * Method for Nuke license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forNuke(limit) {
        return new UsageBasedLicense({
            licenseName: 'nuke',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for RealFlow license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRealFlow(limit) {
        return new UsageBasedLicense({
            licenseName: 'realflow',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for RedShift license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRedShift(limit) {
        return new UsageBasedLicense({
            licenseName: 'redshift',
            ports: [aws_ec2_1.Port.tcp(5054), aws_ec2_1.Port.tcp(7054)],
            limit,
        });
    }
    /**
     * Method for V-Ray license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forVray(limit) {
        return new UsageBasedLicense({
            licenseName: 'vray',
            ports: [aws_ec2_1.Port.tcp(30306)],
            limit,
        });
    }
    /**
     * Method for Yeti license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forYeti(limit) {
        return new UsageBasedLicense({
            licenseName: 'yeti',
            ports: [aws_ec2_1.Port.tcp(5053), aws_ec2_1.Port.tcp(7053)],
            limit,
        });
    }
}
exports.UsageBasedLicense = UsageBasedLicense;
_a = JSII_RTTI_SYMBOL_1;
UsageBasedLicense[_a] = { fqn: "aws-rfdk.deadline.UsageBasedLicense", version: "1.1.0" };
/**
 * Constant used to signify unlimited overage.
 */
UsageBasedLicense.UNLIMITED = 2147483647;
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL)
 * (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html )
 * in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, only allows ingress from the
 * Render Queue (in order to register Workers for license forwarding).
 *
 * When a Deadline Worker requires access to licensing via `UsageBasedLicensing.grantPortAccess(...)`, then the RFDK
 * constructs will grant that worker’s security group ingress on TCP port 17004 as well as other ports as required by
 * the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * ![architecture diagram](/diagrams/deadline/UsageBasedLicensing.svg)
 *
 * Resources Deployed
 * ------------------------
 * - The Auto Scaling Group (ASG) added to the Amazon Elastic Container Service cluster that is hosting the Deadline
 *   License Forwarder for UBL. This creates one C5 Large instance by default.
 * - Amazon Elastic Block Store (EBS) device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * - An Amazon CloudWatch log group that contains the logs from the Deadline License Forwarder application.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The Deadline License Forwarder is designed to be secured by restricting network access to it. For security, only the Deadline
 *   Workers that require access to Usage-based Licenses should be granted network access to the instances deployed by this construct.
 *   Futhermore, you should restrict that access to only the product(s) that those workers require when deploying this construct.
 */
class UsageBasedLicensing extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const usageBasedLicenses = new Array();
        props.licenses.forEach(license => {
            usageBasedLicenses.push(`${license.licenseName}:${license.limit ? license.limit : UsageBasedLicense.UNLIMITED}`);
        });
        if (usageBasedLicenses.length < 1) {
            throw new Error('Should be specified at least one license with defined limit.');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', { vpc: props.vpc });
        if (!props.vpcSubnets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
        }
        const vpcSubnets = props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS };
        this.asg = this.cluster.addCapacity('ASG', {
            vpcSubnets,
            instanceType: props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: props.desiredCount ?? 1,
            maxCapacity: props.desiredCount ?? 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: props.securityGroup,
        });
        const taskDefinition = new aws_ecs_1.TaskDefinition(this, 'TaskDefinition', {
            compatibility: aws_ecs_1.Compatibility.EC2,
            networkMode: aws_ecs_1.NetworkMode.HOST,
        });
        this.grantPrincipal = taskDefinition.taskRole;
        const containerEnv = {
            UBL_CERTIFICATES_URI: '',
            UBL_LIMITS: usageBasedLicenses.join(';'),
            ...props.renderQueue.configureClientECS({
                hosts: [this.asg],
                grantee: this,
            }),
        };
        containerEnv.UBL_CERTIFICATES_URI = props.certificateSecret.secretArn;
        props.certificateSecret.grantRead(taskDefinition.taskRole);
        const prefix = props.logGroupProps?.logGroupPrefix ?? UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...props.logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_1.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, defaultedLogGroupProps);
        logGroup.grantWrite(this.asg);
        const container = taskDefinition.addContainer('LicenseForwarderContainer', {
            image: props.images.licenseForwarder,
            environment: containerEnv,
            memoryReservationMiB: 1024,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'LicenseForwarder',
            }),
        });
        // Increase ulimits
        container.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition,
            desiredCount: props.desiredCount ?? 1,
            placementConstraints: [aws_ecs_1.PlacementConstraint.distinctInstances()],
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
        });
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.service.node.addDependency(this.asg);
        this.node.defaultChild = this.service;
        if (props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.renderQueue.configureSecretsManagementAutoRegistration({
                dependent: this.service.node.defaultChild,
                registrationStatus: _1.SecretsManagementRegistrationStatus.REGISTERED,
                role: _1.SecretsManagementRole.CLIENT,
                vpc: props.vpc,
                vpcSubnets,
            });
        }
        // Grant the render queue the ability to connect to the license forwarder to register workers
        this.asg.connections.allowFrom(props.renderQueue.backendConnections, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * This method grant access of worker fleet to ports that required
     *
     * @param workerFleet - worker fleet
     * @param licenses - UBL licenses
     */
    grantPortAccess(workerFleet, licenses) {
        licenses.forEach(license => {
            license.ports.forEach(port => {
                workerFleet.connections.allowTo(this, port);
            });
        });
        workerFleet.connections.allowTo(this, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
    }
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     */
    get connections() {
        return this.service.connections;
    }
}
exports.UsageBasedLicensing = UsageBasedLicensing;
_b = JSII_RTTI_SYMBOL_1;
UsageBasedLicensing[_b] = { fqn: "aws-rfdk.deadline.UsageBasedLicensing", version: "1.1.0" };
/**
 * The port that the License Forwarder listens on
 */
UsageBasedLicensing.LF_PORT = 17004;
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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