"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runtime_info_1 = require("../../core/lib/runtime-info");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const worker_configuration_1 = require("./worker-configuration");
/**
 * This construct reperesents a fleet from the Spot Fleet Request created by the Spot Event Plugin.
 * This fleet is intended to be used as input for the {@link @aws-rfdk/deadline#ConfigureSpotEventPlugin} construct.
 *
 * The construct itself doesn't create the Spot Fleet Request, but deploys all the resources
 * required for the Spot Fleet Request and generates the Spot Fleet Configuration setting:
 * a one to one mapping between a Deadline Group and Spot Fleet Request Configurations.
 *
 * ![architecture diagram](/diagrams/deadline/SpotEventPluginFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Instance Role, corresponding IAM Policy and an Instance Profile.
 * - A Fleet Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *   in the fleet.
 * - A security Group if security groups are not provided.
 * - An EC2 Launch Template for the Spot Fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *   that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *   the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *   patching strategy to keep this software current with the latest security patches. Please see
 *   https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 */
class SpotEventPluginFleet extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.defaultSubnets = !props.vpcSubnets;
        this.deadlineGroups = props.deadlineGroups.map(group => group.toLocaleLowerCase());
        this.deadlinePools = props.deadlinePools?.map(pool => pool.toLocaleLowerCase());
        this.validateProps(props);
        this.securityGroups = props.securityGroups ?? [new aws_ec2_1.SecurityGroup(this, 'SpotFleetSecurityGroup', { vpc: props.vpc })];
        this.connections = new aws_ec2_1.Connections({ securityGroups: this.securityGroups });
        this.connections.allowToDefaultPort(props.renderQueue.endpoint);
        this.fleetInstanceRole = props.fleetInstanceRole ?? new aws_iam_1.Role(this, 'SpotFleetInstanceRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
            ],
            description: `Spot Fleet instance role for ${id} in region ${aws_cdk_lib_1.Stack.of(scope).region}`,
        });
        this.instanceProfile = new aws_iam_1.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.fleetInstanceRole.roleName],
        });
        this.grantPrincipal = this.fleetInstanceRole;
        this.fleetRole = props.fleetRole ?? new aws_iam_1.Role(this, 'SpotFleetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
            description: `Spot Fleet role for ${id} in region ${aws_cdk_lib_1.Stack.of(scope).region}`,
        });
        this.blockDevices = props.blockDevices;
        this.subnets = props.vpc.selectSubnets(props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS });
        this.instanceTypes = props.instanceTypes;
        this.allocationStrategy = props.allocationStrategy ?? spot_event_plugin_fleet_ref_1.SpotFleetAllocationStrategy.LOWEST_PRICE;
        this.maxCapacity = props.maxCapacity;
        this.validUntil = props.validUntil;
        this.keyName = props.keyName;
        const imageConfig = props.workerMachineImage.getImage(this);
        this.osType = imageConfig.osType;
        this.userData = props.userData ?? imageConfig.userData;
        this.machineImage = props.workerMachineImage;
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this,
            cloudWatchLogSettings: {
                logGroupPrefix: SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: {
                groups: this.deadlineGroups,
                pools: this.deadlinePools,
                region: props.deadlineRegion,
            },
            userDataProvider: props.userDataProvider,
        });
        this.remoteControlPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + SpotEventPluginFleet.MAX_WORKERS_PER_HOST);
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.KEY_VALUE, 'RFDK::SpotEventPluginFleet');
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
        this.launchTemplate = this.createLaunchTemplate(props.trackInstancesWithResourceTracker ?? true);
        this._launchTemplateConfigs = this.createLaunchTemplateConfigs();
    }
    /**
     * @inheritdoc
     */
    allowRemoteControlFrom(other) {
        this.connections.allowFrom(other.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    /**
     * @inheritdoc
     */
    allowRemoteControlTo(other) {
        other.connections.allowTo(this.connections, this.remoteControlPorts, 'Worker remote command listening port');
    }
    createLaunchTemplate(resourceTrackerEnabled) {
        const launchTemplate = new aws_ec2_1.LaunchTemplate(this, 'LaunchTemplate', {
            blockDevices: this.blockDevices,
            role: this.fleetInstanceRole,
            machineImage: this.machineImage,
            keyName: this.keyName,
            securityGroup: this.securityGroups[0],
            userData: this.userData,
        });
        if (this.securityGroups.length > 1) {
            launchTemplate.connections.addSecurityGroup(...this.securityGroups.slice(1));
        }
        aws_cdk_lib_1.Tags.of(launchTemplate).add(resourceTrackerEnabled ? 'DeadlineTrackedAWSResource' : 'DeadlineResourceTracker', 'SpotEventPlugin');
        return launchTemplate;
    }
    createLaunchTemplateConfigs() {
        const launchTemplateConfigs = [];
        // Create a launch template config for each instance type + subnet pair
        this.instanceTypes.forEach(instanceType => {
            this.subnets.subnetIds.forEach(subnetId => {
                launchTemplateConfigs.push({
                    LaunchTemplateSpecification: {
                        Version: aws_ec2_1.LaunchTemplateSpecialVersions.LATEST_VERSION,
                        LaunchTemplateId: this.launchTemplate.launchTemplateId,
                        LaunchTemplateName: this.launchTemplate.launchTemplateName,
                    },
                    Overrides: [{
                            InstanceType: instanceType.toString(),
                            SubnetId: subnetId,
                        }],
                });
            });
        });
        return launchTemplateConfigs;
    }
    validateProps(props) {
        this.validateFleetInstanceRole(props.fleetInstanceRole);
        this.validateInstanceTypes(props.instanceTypes);
        this.validateSubnets(props.vpc, props.vpcSubnets);
        this.validateGroups('deadlineGroups', this.deadlineGroups);
        this.validateRegion('deadlineRegion', props.deadlineRegion);
        this.validateBlockDevices(props.blockDevices);
    }
    validateFleetInstanceRole(fleetInstanceRole) {
        if (fleetInstanceRole) {
            if (aws_cdk_lib_1.Stack.of(fleetInstanceRole) !== aws_cdk_lib_1.Stack.of(this)) {
                throw new Error(`Fleet instance role should be created on the same stack as ${this.constructor.name} to avoid circular dependencies.`);
            }
        }
    }
    validateInstanceTypes(array) {
        if (array.length === 0) {
            throw new Error('At least one instance type is required for a Spot Fleet Request Configuration');
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const { subnets } = vpc.selectSubnets(vpcSubnets);
        if (subnets.length === 0) {
            aws_cdk_lib_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(vpcSubnets)}', please use a different selection.`);
        }
    }
    validateGroups(property, array) {
        const regex = /^(?!none$)[a-z0-9-_]+$/g;
        if (array.length === 0) {
            throw new Error('At least one Deadline Group is required for a Spot Fleet Request Configuration');
        }
        array.forEach(value => {
            if (!regex.test(value)) {
                throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
            }
        });
    }
    validateRegion(property, region) {
        const regex = /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i;
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`The spot-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                const { iops, volumeType } = device.volume.ebsDevice;
                if (!iops) {
                    if (volumeType === aws_ec2_1.EbsDeviceVolumeType.IO1) {
                        throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                    }
                }
                else if (volumeType !== aws_ec2_1.EbsDeviceVolumeType.IO1) {
                    aws_cdk_lib_1.Annotations.of(this).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    aws_cdk_lib_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the spot-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
}
exports.SpotEventPluginFleet = SpotEventPluginFleet;
_a = JSII_RTTI_SYMBOL_1;
SpotEventPluginFleet[_a] = { fqn: "aws-rfdk.deadline.SpotEventPluginFleet", version: "1.1.0" };
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
SpotEventPluginFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
SpotEventPluginFleet.MAX_WORKERS_PER_HOST = 8;
//# sourceMappingURL=data:application/json;base64,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