"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretsManagementIdentityRegistration = void 0;
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const _1 = require(".");
const core_1 = require("../../core");
/**
 * Construct that configures desired Deadline Secrets Management identity registration settings.
 *
 * Resources Deployed
 * ------------------------
 * - IAM policy statements are added to the IAM policy that is attached to the IAM role of the DeploymentInstance.
 *   These statements grant the DeploymentInstance the ability to fetch the Deadline Client installer, get the value of
 *   the AWS Secrets Manager secert containing the Deadline Secrets Management administrator credentials, get the value
 *   of the AWS Secrets Manager secert containing the Deadline Repository's database credentials,
 * - Security group ingress rule to allow the DeploymentInstance to connect to the Repository's database
 * - Security group ingress rule to allow the DeploymentInstance to connect to the Repository's file-system
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The instance deployed by this construct has read/write access to the Deadline Repository (database and
 *   file-system), the AWS Secrets Manager secrets containing credentials for the Database and the Deadline Secrets
 *   Management administrator. Access to the instance permits command and control of the render farm and should be
 *   restricted.
 */
class SecretsManagementIdentityRegistration extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.subnetRegistrations = new Map();
        if (!props.repository.secretsManagementSettings.enabled) {
            throw new Error('Secrets management is not enabled on repository');
        }
        /* istanbul ignore next */
        if (!props.repository.secretsManagementSettings.credentials) {
            throw new Error('Repository does not contain secrets management credentials');
        }
        this.adminCredentials = props.repository.secretsManagementSettings.credentials;
        this.deploymentInstance = props.deploymentInstance;
        this.renderQueueSubnets = props.renderQueueSubnets;
        // Download and install the Deadline Client
        this.installDeadlineClient(props);
        // Configure the Deadline Client to direct-connect to the repository
        props.repository.configureClientInstance({
            host: props.deploymentInstance,
            mountPoint: '/mnt/repository',
        });
        // Install python dependencies
        const localScriptFile = this.preparePythonScript(props);
        this.runPythonScript(props, localScriptFile);
        props.deploymentInstance.addExecutionDependency(props.repository);
    }
    addSubnetIdentityRegistrationSetting(addSubnetProps) {
        if (addSubnetProps.role === _1.SecretsManagementRole.ADMINISTRATOR) {
            throw new Error('The Administrator role cannot be set using a Deadline identity registration setting');
        }
        const { vpc, vpcSubnets } = addSubnetProps;
        const selectedSubnets = vpc.selectSubnets(vpcSubnets);
        selectedSubnets.subnets.forEach(subnet => {
            if (this.renderQueueSubnets.subnets.some(rqSubnet => subnet == rqSubnet)) {
                aws_cdk_lib_1.Annotations.of(addSubnetProps.dependent).addWarning(`Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${addSubnetProps.dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`);
            }
            const observedSubnet = this.subnetRegistrations.get(subnet.subnetId);
            if (observedSubnet) {
                if (observedSubnet.registrationStatus !== addSubnetProps.registrationStatus) {
                    throw new Error(`Subnet is already registered with registrationStatus "${observedSubnet.registrationStatus}" but another caller requested "${addSubnetProps.registrationStatus}"`);
                }
                else if (observedSubnet.role !== addSubnetProps.role) {
                    throw new Error(`Subnet is already registered with role "${observedSubnet.role}" but another caller requested "${addSubnetProps.role}"`);
                }
            }
            else {
                this.subnetRegistrations.set(subnet.subnetId, {
                    registrationStatus: addSubnetProps.registrationStatus,
                    role: addSubnetProps.role,
                });
            }
        });
        addSubnetProps.dependent.node.addDependency(this.deploymentInstance);
    }
    runPythonScript(props, localScriptFile) {
        // The command-line arguments to be passed to the script that configures the Deadline identity registration
        // settings
        const scriptArgs = aws_cdk_lib_1.Lazy.list({
            produce: () => {
                return [].concat([
                    // Region
                    '--region',
                    aws_cdk_lib_1.Stack.of(this).region,
                    // Admin credentials
                    '--credentials',
                    `"${this.adminCredentials.secretArn}"`,
                ], 
                // Subnets of the load balancer
                (props.renderQueueSubnets
                    .subnetIds
                    .map(subnetID => `--connection-subnet "${subnetID}"`)), 
                // Subnets of RFDK Deadline Client constructs
                (Array.from(this.subnetRegistrations.entries())
                    // Each setting becomes a comma (,) separated string of fields
                    //   <SUBNET_ID>,<ROLE>,<REGISTRATION_STATUS>
                    .map(subnetRegistrationEntry => {
                    const [subnetID, registrationSettingEffect] = subnetRegistrationEntry;
                    return [
                        subnetID,
                        registrationSettingEffect.role.toString(),
                        (registrationSettingEffect.registrationStatus).toString(),
                    ].join(',');
                })
                    // convert into argument key/value pair
                    .map(joinedSubnetArgValue => `--source-subnet "${joinedSubnetArgValue}"`)));
            },
        });
        // We can't use ScriptAsset.executeOn(...) because we need to run as "ec2-user".
        // This is because Repository.configureClientInstance(...) used above will store the credentials
        // in a per-user credential store that is only available to "ec2-user".
        props.deploymentInstance.userData.addCommands(`sudo --login -u ec2-user ${localScriptFile} ` + aws_cdk_lib_1.Fn.join(' ', scriptArgs));
    }
    preparePythonScript(props) {
        const script = new core_1.ScriptAsset(this, 'ConfigureIdentityRegistrationSettingScript', {
            path: path.join(__dirname, '..', 'scripts', 'python', 'configure_identity_registration_settings.py'),
        });
        // Grant access to ec2:DescribeSubnets. Subnet IPv4 CIDR ranges are not exposed through
        // CloudFormation attributes. Instead, we must query them using the EC2 API on the deployment instance
        props.deploymentInstance.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['ec2:DescribeSubnets'],
            // ec2:DescribeSubnets does not support resource level permissions. See
            // https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonec2.html
            resources: ['*'],
        }));
        this.adminCredentials.grantRead(props.deploymentInstance);
        script.grantRead(props.deploymentInstance);
        const localScriptFile = props.deploymentInstance.userData.addS3DownloadCommand({
            bucket: script.bucket,
            bucketKey: script.s3ObjectKey,
            localFile: '/home/ec2-user/configure_identity_registration_settings.py',
        });
        props.deploymentInstance.userData.addCommands(`chmod +x ${localScriptFile}`, `chown ec2-user:ec2-user ${localScriptFile}`);
        return localScriptFile;
    }
    installDeadlineClient(props) {
        props.version.linuxInstallers.client.s3Bucket.grantRead(props.deploymentInstance, props.version.linuxInstallers.client.objectKey);
        const clientInstallerPath = props.deploymentInstance.userData.addS3DownloadCommand({
            bucket: props.version.linuxInstallers.client.s3Bucket,
            bucketKey: props.version.linuxInstallers.client.objectKey,
        });
        props.deploymentInstance.userData.addCommands('set -x');
        props.deploymentInstance.userData.addCommands(`chmod +x "${clientInstallerPath}"`);
        props.deploymentInstance.userData.addCommands([
            // This is required b/c USER and HOME environment variables are not defined when running
            // user-data
            'sudo', '--login',
            // Run the Deadline Client installer
            `"${clientInstallerPath}"`,
            '--mode', 'unattended',
            '--connectiontype', 'Remote',
            '--proxyrootdir', '127.0.0.1:8080',
            '--noguimode', 'true',
            '--slavestartup', 'false',
            '--launcherdaemon', 'false',
            '--restartstalled', 'true',
            '--autoupdateoverride', 'False',
        ].join(' '));
    }
}
exports.SecretsManagementIdentityRegistration = SecretsManagementIdentityRegistration;
//# sourceMappingURL=data:application/json;base64,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