"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueueConnection = void 0;
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
/**
 * Helper class for connecting Clients to a render queue
 */
class RenderQueueConnection {
    /**
     * Create a RQ Connection for http traffic
     */
    static forHttp(options) {
        return new HTTPConnection(options);
    }
    /**
     * Create a RQ Connection for https traffic
     */
    static forHttps(options) {
        return new HTTPSConnection(options);
    }
    /**
     * Fetch the instance configuration python script
     * @param stack The stack that the asset should be created in
     */
    connectionAssetSingleton(stack) {
        const uuid = '3be2203f-fea1-4143-bc09-7ef079b4299c';
        const uniqueId = 'RenderQueueConnectionAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'client-rq-connection.py'),
        });
    }
    /**
     * Executes connection asset singleton wth a given list of arguments
     */
    ExecuteConnectionAsset(host, args) {
        const hostStack = aws_cdk_lib_1.Stack.of(host);
        const connectionAsset = this.connectionAssetSingleton(hostStack);
        connectionAsset.grantRead(host);
        const configureScriptPath = host.userData.addS3DownloadCommand({
            bucket: connectionAsset.bucket,
            bucketKey: connectionAsset.s3ObjectKey,
        });
        const dlExtraCommands = [];
        if (args.tlsCaArn) {
            dlExtraCommands.push('--tls-ca', `"${args.tlsCaArn}"`);
        }
        if (host.osType === aws_ec2_1.OperatingSystemType.LINUX) {
            host.userData.addCommands('if [ -f "/etc/profile.d/deadlineclient.sh" ]; then', '  source "/etc/profile.d/deadlineclient.sh"', 'fi', `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "${configureScriptPath}" --render-queue "${args.address}" ${dlExtraCommands.join(' ')}`, 
            // Cleanup
            `rm -f "${configureScriptPath}"`);
            if (args.restartLauncher ?? true) {
                host.userData.addCommands('if service --status-all | grep -q "Deadline 10 Launcher"; then', '  service deadline10launcher restart', 'fi');
            }
        }
        else if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            host.userData.addCommands('$ErrorActionPreference = "Stop"', '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value', `& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "${configureScriptPath}" --render-queue "${args.address}" ${dlExtraCommands.join(' ')} 2>&1`, `Remove-Item -Path "${configureScriptPath}"`);
            if (args.restartLauncher ?? true) {
                host.userData.addCommands('If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {', '  Restart-Service "deadline10launcherservice"', '} Else {', '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1', '  & "$DEADLINE_PATH/deadlinelauncher.exe" -nogui 2>&1', '}');
            }
        }
    }
}
exports.RenderQueueConnection = RenderQueueConnection;
/**
 * Specialization of {@link RenderQueueConnection} for HTTP Connections
 */
class HTTPConnection extends RenderQueueConnection {
    constructor(config) {
        super();
        this.config = config;
    }
    configureClientECS(params) {
        params.hosts.forEach(host => {
            host.connections.allowToDefaultPort(this.config.endpoint);
        });
        return {
            RENDER_QUEUE_URI: `http://${this.config.endpoint.socketAddress}`,
        };
    }
    configureClientInstance(params) {
        params.host.connections.allowToDefaultPort(this.config.endpoint);
        this.ExecuteConnectionAsset(params.host, {
            address: `http://${this.config.endpoint.socketAddress}`,
            restartLauncher: params.restartLauncher,
        });
    }
}
/**
 * Specialization of {@link RenderQueueConnection} for HTTPS Connections
 */
class HTTPSConnection extends RenderQueueConnection {
    constructor(config) {
        super();
        this.config = config;
    }
    configureClientECS(params) {
        params.hosts.forEach(host => {
            host.connections.allowToDefaultPort(this.config.endpoint);
        });
        this.config.caCert.grantRead(params.grantee);
        return {
            RENDER_QUEUE_URI: `https://${this.config.endpoint.socketAddress}`,
            RENDER_QUEUE_TLS_CA_CERT_URI: this.config.caCert.secretArn,
        };
    }
    configureClientInstance(params) {
        params.host.connections.allowToDefaultPort(this.config.endpoint);
        this.config.caCert.grantRead(params.host);
        this.ExecuteConnectionAsset(params.host, {
            address: `https://${this.config.endpoint.socketAddress}`,
            tlsCaArn: this.config.caCert.secretArn,
            restartLauncher: params.restartLauncher,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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