"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const url_1 = require("url");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_docdb_1 = require("aws-cdk-lib/aws-docdb");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const core_1 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
const version_1 = require("./version");
const version_query_1 = require("./version-query");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system
 * that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * ![architecture diagram](/diagrams/deadline/Repository.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon EFS Point - If no filesystem is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 * - An RFDK PadEfsStorage - If no filesystem is provided.
 * - An AWS Secrets Manager Secret - If no Secret with admin credentials for Deadline Secrets Management is provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *   system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *   Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *   Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *   require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *   submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *   that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *   to only those who require it.
 * - If no file-system is provided to the Repository, then the Repository creates an EFS access point with unrestricted
 *   access to the entire EFS file-system. If you would like a single EFS file-system that is used by the Deadline
 *   Repository and other agents, you should supply the file-system and a access-restricted EFS access point to the
 *   Repository construct instead.
 */
class Repository extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.database && props.backupOptions?.databaseRetention) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && props.removalPolicy?.filesystem) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && props.removalPolicy?.database) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem instanceof core_1.MountableEfs && !props.fileSystem.accessPoint) {
            throw new Error('When using EFS with the Repository, you must provide an EFS Access Point');
        }
        if ((props.secretsManagementSettings?.enabled ?? true) && props.database && !props.database.databaseConstruct) {
            throw new Error('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
        }
        this.version = props.version;
        const meetsMinSecretsVersion = !this.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION);
        const secretsManagementIsEnabled = props.secretsManagementSettings?.enabled ?? meetsMinSecretsVersion;
        if (secretsManagementIsEnabled && !meetsMinSecretsVersion) {
            throw new Error(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        }
        this.secretsManagementSettings = {
            enabled: secretsManagementIsEnabled,
            credentials: props.secretsManagementSettings?.credentials ??
                (secretsManagementIsEnabled ? new aws_secretsmanager_1.Secret(props.database?.databaseConstruct ? aws_cdk_lib_1.Stack.of(props.database?.databaseConstruct) : this, 'SMAdminUser', {
                    description: 'Admin credentials for Deadline Secrets Management',
                    generateSecretString: {
                        excludeCharacters: '\"$&\'()/<>[\\]\`{|}',
                        includeSpace: false,
                        passwordLength: 24,
                        requireEachIncludedType: true,
                        generateStringKey: 'password',
                        secretStringTemplate: JSON.stringify({ username: Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME }),
                    },
                    removalPolicy: props.secretsManagementSettings?.credentialsRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN,
                }) : undefined),
        };
        this.fileSystem = props.fileSystem ?? (() => {
            const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: props.removalPolicy?.filesystem ?? aws_cdk_lib_1.RemovalPolicy.RETAIN,
                securityGroup: props.securityGroupsOptions?.fileSystem,
            });
            const paddingAccess = fs.addAccessPoint('PaddingAccessPoint', {
                createAcl: {
                    ownerGid: '0',
                    ownerUid: '0',
                    permissions: '744',
                },
                path: '/RFDK_PaddingFiles',
            });
            new core_1.PadEfsStorage(this, 'PadEfsStorage', {
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS },
                accessPoint: paddingAccess,
                desiredPadding: aws_cdk_lib_1.Size.gibibytes(40),
            });
            const accessPoint = fs.addAccessPoint('AccessPoint', {
                posixUser: {
                    uid: '0',
                    gid: '0',
                },
            });
            return new core_1.MountableEfs(this, {
                filesystem: fs,
                accessPoint,
            });
        })();
        // Set up the Database of the repository
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                aws_cdk_lib_1.Annotations.of(this).addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = props.databaseAuditLogging ?? true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = props.documentDbInstanceCount ?? Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                vpc: props.vpc,
                vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS, onePerAz: true },
                securityGroup: props.securityGroupsOptions?.database,
                instances,
                backup: {
                    retention: props.backupOptions?.databaseRetention ?? Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: props.removalPolicy?.database ?? aws_cdk_lib_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
            },
            minCapacity: 1,
            maxCapacity: 1,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (props.repositoryInstallationTimeout || aws_cdk_lib_1.Duration.minutes(15)),
            }),
            securityGroup: props.securityGroupsOptions?.installer,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version, props.repositorySettings, 
        // Change ownership of the Deadline repository files if-and-only-if the mounted file-system
        // uses the POSIX permissions based on the process' user UID/GID
        this.fileSystem.usesUserPosixPermissions() ? Repository.REPOSITORY_OWNER : undefined);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
        const thisConstruct = this;
        this.node.addValidation({
            validate() {
                const validationErrors = [];
                // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
                // a resource in another stack is referencing it.
                if (thisConstruct.version instanceof version_query_1.VersionQuery) {
                    const versionStack = aws_cdk_lib_1.Stack.of(thisConstruct.version);
                    const thisStack = aws_cdk_lib_1.Stack.of(thisConstruct);
                    if (versionStack != thisStack) {
                        validationErrors.push('A VersionQuery can not be supplied from a different stack');
                    }
                }
                return validationErrors;
            },
        });
    }
    /**
     * @inheritdoc
     */
    configureClientECS(props) {
        const hostMountPoint = props.containerInstances.filesystemMountPoint ?? '/mnt/repo';
        const majorVersion = aws_cdk_lib_1.Token.isUnresolved(this.version.majorVersion) ?
            aws_cdk_lib_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = props.containers.filesystemMountPoint ?? `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this.installerGroup);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(props) {
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this.installerGroup);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = stack.node.tryFindChild(uniqueId) ?? core_1.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        const prefix = logGroupProps?.logGroupPrefix ?? Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_1.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_1.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', this.version.isLessThan(version_1.Version.MINIMUM_VERSION_USING_NEW_INSTALLBUILDER_LOG) ? '/tmp/bitrock_installer.log' : '/tmp/installbuilder_installer.log');
        new core_1.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = aws_cdk_lib_1.Names.uniqueId(this);
        aws_cdk_lib_1.Tags.of(this.installerGroup).add(tagKey, aws_cdk_lib_1.Names.uniqueId(this));
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)');
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + aws_cdk_lib_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + aws_cdk_lib_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version, settings, owner) {
        const installerScriptAsset = core_1.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        const installerArgs = [
            '-i', `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
            '-p', `"${installPath}"`,
            '-v', version.linuxFullVersionString(),
        ];
        if (this.secretsManagementSettings.enabled) {
            installerArgs.push('-r', aws_cdk_lib_1.Stack.of(this.secretsManagementSettings.credentials ?? this).region);
            this.secretsManagementSettings.credentials.grantRead(installerGroup);
            installerArgs.push('-c', this.secretsManagementSettings.credentials.secretArn ?? '');
        }
        if (settings) {
            const repositorySettingsFilePath = installerGroup.userData.addS3DownloadCommand({
                bucket: settings.bucket,
                bucketKey: settings.s3ObjectKey,
            });
            installerArgs.push('-s', repositorySettingsFilePath);
        }
        // We can ignore this in test coverage because we always use Repository.REPOSITORY_OWNER
        /* istanbul ignore next */
        if (owner) {
            installerArgs.push('-o', `${owner.uid}:${owner.gid}`);
        }
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: installerArgs,
        });
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "aws-rfdk.deadline.Repository", version: "1.1.0" };
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = aws_cdk_lib_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = aws_cdk_lib_1.Duration.days(15);
/**
 * The Repository owner is 1000:1000 (ec2-user on AL2).
 */
Repository.REPOSITORY_OWNER = { uid: 1000, gid: 1000 };
/**
 * Default username for auto generated admin credentials in Secret Manager.
 */
Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME = 'admin';
//# sourceMappingURL=data:application/json;base64,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