"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const fsx = require("aws-cdk-lib/aws-fsx");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('MountableFsxLustre', () => {
    let app;
    let stack;
    let vpc;
    let fs;
    let fsSecurityGroup;
    let instance;
    let instanceSecurityGroup;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        fsSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'FSxLSecurityGroup', {
            vpc,
        });
        fs = new fsx.LustreFileSystem(stack, 'FSxL', {
            vpc,
            vpcSubnet: vpc.privateSubnets[0],
            lustreConfiguration: {
                deploymentType: fsx.LustreDeploymentType.SCRATCH_1,
            },
            storageCapacityGiB: 1200,
            securityGroup: fsSecurityGroup,
        });
        instanceSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'InstanceSecurityGroup', {
            vpc,
        });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            securityGroup: instanceSecurityGroup,
        });
    });
    test('mounts with defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the FSxL's security group
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 988,
            ToPort: 1023,
            SourceSecurityGroupId: stack.resolve(instanceSecurityGroup.securityGroupId),
            GroupId: stack.resolve(fsSecurityGroup.securityGroupId),
        });
        // Make sure we download the mountFsxLustre script asset bundle
        const s3Copy = `aws s3 cp 's3://\${Token[TOKEN.\\d+]}/${asset_constants_1.MOUNT_FSX_SCRIPT_LINUX.Key}.zip' '/tmp/${asset_constants_1.MOUNT_FSX_SCRIPT_LINUX.Key}.zip'`;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`unzip /tmp/${asset_constants_1.MOUNT_FSX_SCRIPT_LINUX.Key}.zip`)));
        // Make sure we install the Lustre client
        expect(userData).toMatch('bash ./installLustreClient.sh');
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/fsx/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} r')));
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} rw,option1,option2')));
    });
    test('asset is singleton', () => {
        // GIVEN
        const mount1 = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        const mount2 = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = `aws s3 cp 's3://\${Token[TOKEN.\\d+]}/${asset_constants_1.MOUNT_FSX_SCRIPT_LINUX.Key}.zip'`;
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = userData.match(regex) ?? [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    test('applies Lustre fileset', () => {
        // GIVEN
        const fileset = 'fileset';
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
            fileset,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`bash ./mountFsxLustre.sh \${Token[TOKEN.\\d+]} /mnt/fsx/fs1 \${Token[TOKEN.\\d+]}/${fileset} rw`)));
    });
    test('.usesUserPosixPermissions() is true', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        const usesUserPosixPermissions = mount.usesUserPosixPermissions();
        // THEN
        expect(usesUserPosixPermissions).toEqual(true);
    });
});
//# sourceMappingURL=data:application/json;base64,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