"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const efs = require("aws-cdk-lib/aws-efs");
const lib_1 = require("../lib");
const mount_permissions_helper_1 = require("../lib/mount-permissions-helper");
const asset_constants_1 = require("./asset-constants");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableEFS', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let instance;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new efs.FileSystem(stack, 'EFS', { vpc });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the EFS's security group
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 2049,
            ToPort: 2049,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'InstanceInstanceSecurityGroupF0E2D5BE',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'EFSEfsSecurityGroup56F189CE',
                    'GroupId',
                ],
            },
        });
        // Make sure we download the mountEfs script asset bundle
        const s3Copy = `aws s3 cp 's3://\${Token[TOKEN.\\d+]}/${asset_constants_1.MOUNT_EFS_SCRIPT_LINUX.Key}.zip' '/tmp/${asset_constants_1.MOUNT_EFS_SCRIPT_LINUX.Key}.zip'`;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`unzip /tmp/${asset_constants_1.MOUNT_EFS_SCRIPT_LINUX.Key}.zip`)));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/efs/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false r')));
    });
    describe.each([
        [undefined],
        [lib_1.MountPermissions.READONLY],
        [lib_1.MountPermissions.READWRITE],
    ])('access point with %s access permissions', (mountPermission) => {
        describe.each([
            [
                'unspecified POSIX user',
                {
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved non-root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '1000' },
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '0' },
                    expectedClientRootAccess: true,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '0', gid: '1000' },
                    expectedClientRootAccess: true,
                },
            ],
        ])('%s', (_name, testCase) => {
            // GIVEN
            const { posixUser, expectedClientRootAccess } = testCase;
            const expectedActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mountPermission);
            if (expectedClientRootAccess) {
                expectedActions.push('elasticfilesystem:ClientRootAccess');
            }
            const mountPath = '/mnt/efs/fs1';
            let userData;
            let accessPoint;
            let expectedMountMode;
            beforeEach(() => {
                // GIVEN
                accessPoint = new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser,
                });
                const mount = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                });
                expectedMountMode = (mountPermission === lib_1.MountPermissions.READONLY) ? 'ro' : 'rw';
                // WHEN
                mount.mountToLinuxInstance(instance, {
                    location: mountPath,
                    permissions: mountPermission,
                });
                userData = stack.resolve(instance.userData.render());
            });
            test('userdata specifies access point when mounting', () => {
                // THEN
                expect(userData).toEqual({
                    'Fn::Join': [
                        '',
                        expect.arrayContaining([
                            expect.stringMatching(new RegExp('(\\n|^)bash \\./mountEfs.sh $')),
                            stack.resolve(efsFS.fileSystemId),
                            ` ${mountPath} false ${expectedMountMode},iam,accesspoint=`,
                            stack.resolve(accessPoint.accessPointId),
                            expect.stringMatching(/^\n/),
                        ]),
                    ],
                });
            });
            test('grants IAM access point permissions', () => {
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: assertions_1.Match.objectLike({
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: expectedActions.length === 1 ? expectedActions[0] : expectedActions,
                                Condition: {
                                    StringEquals: {
                                        'elasticfilesystem:AccessPointArn': stack.resolve(accessPoint.accessPointArn),
                                    },
                                },
                                Effect: 'Allow',
                                Resource: stack.resolve(efsFS.node.defaultChild.attrArn),
                            },
                        ]),
                        Version: '2012-10-17',
                    }),
                    Roles: assertions_1.Match.arrayWith([
                        // The Policy construct micro-optimizes the reference to a role in the same stack using its logical ID
                        stack.resolve(instance.role.node.defaultChild.ref),
                    ]),
                });
            });
        });
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw,option1,option2')));
    });
    test('asset is singleton', () => {
        // GIVEN
        const mount1 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        const mount2 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = `aws s3 cp 's3://\${Token[TOKEN.\\d+]}/${asset_constants_1.MOUNT_EFS_SCRIPT_LINUX.Key}.zip'`;
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = userData.match(regex) ?? [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    describe('resolves mount target using API', () => {
        describe.each([
            ['with access point', () => {
                    return new efs.AccessPoint(stack, 'AccessPoint', {
                        fileSystem: efsFS,
                        posixUser: {
                            gid: '1',
                            uid: '1',
                        },
                    });
                }],
            ['without access point', () => undefined],
        ])('%s', (_, getAccessPoint) => {
            let accessPoint;
            beforeEach(() => {
                // GIVEN
                accessPoint = getAccessPoint();
                const mountable = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                    resolveMountTargetDnsWithApi: true,
                });
                // WHEN
                mountable.mountToLinuxInstance(instance, {
                    location: '/mnt/efs',
                });
            });
            test('grants DescribeMountTargets permission', () => {
                const expectedResources = [
                    stack.resolve(efsFS.node.defaultChild.attrArn),
                ];
                if (accessPoint) {
                    expectedResources.push(stack.resolve(accessPoint?.accessPointArn));
                }
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: assertions_1.Match.objectLike({
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: 'elasticfilesystem:DescribeMountTargets',
                                Effect: 'Allow',
                                Resource: expectedResources.length == 1 ? expectedResources[0] : expectedResources,
                            },
                        ]),
                    }),
                    Roles: assertions_1.Match.arrayWith([
                        stack.resolve(instance.role.node.defaultChild.ref),
                    ]),
                });
            });
        });
    });
    describe('.usesUserPosixPermissions()', () => {
        test('access point with POSIX user returns false', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint: new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser: {
                        uid: '1000',
                        gid: '1000',
                    },
                }),
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(false);
        });
        test('access point without POSIX user returns true', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint: new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                }),
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(true);
        });
        test.each([
            [
                'AccessPoint.fromAccessPointId(...)',
                (inputStack) => efs.AccessPoint.fromAccessPointId(inputStack, 'AccessPoint', 'accessPointId'),
            ],
            [
                'AccessPoint.fromAccessPointAttributes(...)',
                (inputStack) => {
                    return efs.AccessPoint.fromAccessPointAttributes(inputStack, 'AccessPoint', {
                        accessPointArn: aws_cdk_lib_1.Arn.format({
                            resource: 'AccessPoint',
                            service: 'efs',
                            resourceName: 'accessPointName',
                        }, inputStack),
                        fileSystem: efsFS,
                    });
                },
            ],
        ])('%s throws error', (_label, accessPointProvider) => {
            // GIVEN
            const accessPoint = accessPointProvider(stack);
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint,
            });
            // WHEN
            function when() {
                mount.usesUserPosixPermissions();
            }
            // THEN
            expect(when).toThrow(/^MountableEfs.usesUserPosixPermissions\(\) only supports efs.AccessPoint instances, got ".*"$/);
        });
        test('no access point returns true', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(true);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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