"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const lib_1 = require("../lib");
describe('MongoDbPostInstall', () => {
    let stack;
    let vpc;
    let mongoDb;
    let pwUser1Arn;
    let pwUser2Arn;
    let pwUser1;
    let pwUser2;
    let x509User1Arn;
    let x509User2Arn;
    let x509User1;
    let x509User2;
    beforeEach(() => {
        const hostname = 'mongodb';
        const zoneName = 'testZone.internal';
        const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
        const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        const serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        mongoDb = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        pwUser1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1-abcdef';
        pwUser2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2-abcdef';
        pwUser1 = aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'PwUser1', pwUser1Arn);
        pwUser2 = aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'PwUser2', pwUser2Arn);
        x509User1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User1-abcdef';
        x509User2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User2-abcdef';
        x509User1 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'x509User1', x509User1Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb1' }]),
        };
        x509User2 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'x509User2', x509User2Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb2' }]),
        };
    });
    test('created correctly: both user types', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs16.x',
            VpcConfig: {
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'MongoPostInstallLambdaSecurityGroup62729E3A',
                            'GroupId',
                        ],
                    },
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                    {
                        Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                    },
                ],
            },
        });
        // Lambda role can get the required secrets.
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assertions_1.Match.arrayWith([
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser1Arn,
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser2Arn,
                    },
                ]),
            },
        });
        assertions_1.Template.fromStack(stack).hasResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Properties: {
                Connection: {
                    Hostname: 'mongodb.testZone.internal',
                    Port: '27017',
                    CaCertificate: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                    Credentials: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                },
                PasswordAuthUsers: [
                    pwUser1Arn,
                    pwUser2Arn,
                ],
                X509AuthUsers: [
                    {
                        Certificate: x509User1Arn,
                        Roles: x509User1.roles,
                    },
                    {
                        Certificate: x509User2Arn,
                        Roles: x509User2.roles,
                    },
                ],
            },
            DependsOn: [
                'MongoDbInstanceServerAsgASG2643AD1D',
                'MongoPostInstallLambdaServiceRoleDefaultPolicy8B1C1CE8',
                'MongoPostInstallLambdaServiceRoleCD03B9B9',
            ],
        });
    });
    test('created correctly: only password users', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        // Lambda role can get the required secrets.
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser1Arn,
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser2Arn,
                    },
                ],
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            PasswordAuthUsers: [
                pwUser1Arn,
                pwUser2Arn,
            ],
        });
    });
    test('created correctly: only x509 users', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            X509AuthUsers: [
                {
                    Certificate: x509User1Arn,
                    Roles: x509User1.roles,
                },
                {
                    Certificate: x509User2Arn,
                    Roles: x509User2.roles,
                },
            ],
        });
    });
    test('use selected subnets', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            mongoDb,
            users,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            VpcConfig: {
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                ],
            },
        });
    });
    test('assert bad x509 role', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [
                {
                    certificate: x509User1.certificate,
                    roles: '}{',
                },
            ],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
                vpc,
                mongoDb,
                users,
            });
        }).toThrowError(/MongoDbPostInstallSetup: Could not parse JSON role for x509 user:/);
    });
});
//# sourceMappingURL=data:application/json;base64,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