"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('Test MongoDbInstaller', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
    });
    test('license rejection throws', () => {
        // GIVEN
        const errorString = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
        // Must throw when providing no SSPL option.
        expect(() => {
            new lib_1.MongoDbInstaller(stack, {
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            });
        }).toThrowError(errorString);
        // Must throw when explicitly rejecting the SSPL.
        expect(() => {
            new lib_1.MongoDbInstaller(stack, {
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_REJECTS_SSPL,
            });
        }).toThrowError(errorString);
    });
    test('linux installation', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        const installer = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        // WHEN
        installer.installOnLinuxInstance(instance);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            'Fn::Sub': asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            'Fn::Sub': asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
            },
        });
        // Make sure we download and run the mongo install script
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `#!/bin/bash\nmkdir -p $(dirname '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                            },
                            `/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh' '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh'\nbash /tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh`,
                        ],
                    ],
                },
            },
        });
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const installer = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        // THEN
        expect(() => {
            installer.installOnLinuxInstance(windowsInstance);
        }).toThrowError('Target instance must be Linux.');
    });
    test('asset is singleton', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        const installer1 = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        const installer2 = new lib_1.MongoDbInstaller(stack, {
            version: lib_1.MongoDbVersion.COMMUNITY_3_6,
            userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
        });
        // WHEN
        installer1.installOnLinuxInstance(instance);
        installer2.installOnLinuxInstance(instance);
        // THEN
        // The source of the asset copy should be identical from installer1 & installer2
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        assertions_1.Match.arrayWith([
                            `#!/bin/bash\nmkdir -p $(dirname '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                            },
                            `/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh' '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh'\nbash /tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh\nmkdir -p $(dirname '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                            },
                            `/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh' '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh'\nbash /tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh`,
                        ]),
                    ],
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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