"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const exporting_log_group_1 = require("../lib/exporting-log-group");
test('default exporting log group is created correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const bucket = new aws_s3_1.Bucket(stack, 'DestinationBucket', {
        bucketName: 'test-bucket',
    });
    // WHEN
    new exporting_log_group_1.ExportingLogGroup(stack, 'ExportingLogGroup', {
        bucketName: bucket.bucketName,
        logGroupName: 'logGroup',
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        ServiceToken: {
            'Fn::GetAtt': [
                'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                'Arn',
            ],
        },
        LogGroupName: 'logGroup',
        SdkRetry: {
            maxRetries: 7,
            base: 200,
        },
        RetentionInDays: 3,
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'logs:DescribeExportTasks',
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'logs:CreateExportTask',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'ExportingLogGroupLogRetention80FFADE8',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                },
            ],
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanOrEqualToThreshold',
        EvaluationPeriods: 1,
        Dimensions: [
            {
                Name: 'FunctionName',
                Value: {
                    Ref: 'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eC741F553',
                },
            },
        ],
        MetricName: 'Errors',
        Namespace: 'AWS/Lambda',
        Period: 300,
        Statistic: 'Sum',
        Threshold: 1,
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Events::Rule', {
        ScheduleExpression: 'rate(1 hour)',
        State: 'ENABLED',
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eC741F553',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                Input: {
                    'Fn::Join': [
                        '',
                        [
                            '{\"BucketName\":\"',
                            {
                                Ref: 'DestinationBucket4BECDB47',
                            },
                            '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"logGroup\",\"RetentionInHours\":72}',
                        ],
                    ],
                },
            },
        ],
    });
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Lambda::Function', 2);
});
test('custom set retention is created correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const bucket = new aws_s3_1.Bucket(stack, 'DestinationBucket', {
        bucketName: 'test-bucket',
    });
    // WHEN
    new exporting_log_group_1.ExportingLogGroup(stack, 'ExportingLogGroup', {
        bucketName: bucket.bucketName,
        logGroupName: 'logGroup',
        retention: aws_logs_1.RetentionDays.ONE_WEEK,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        ServiceToken: {
            'Fn::GetAtt': [
                'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                'Arn',
            ],
        },
        LogGroupName: 'logGroup',
        SdkRetry: {
            maxRetries: 7,
            base: 200,
        },
        RetentionInDays: 7,
    });
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Lambda::Function', 2);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::CloudWatch::Alarm', 1);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Events::Rule', 1);
});
//# sourceMappingURL=data:application/json;base64,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