"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const deployment_instance_1 = require("../lib/deployment-instance");
const tag_helpers_1 = require("./tag-helpers");
const test_helper_1 = require("./test-helper");
const DEFAULT_CONSTRUCT_ID = 'DeploymentInstance';
/**
 * Machine image that spies on the following user data methods:
 *
 * * `.addOnExitCommands`
 * * `.addExecuteFileCommand`
 */
class AmazonLinuxWithUserDataSpy extends aws_ec2_1.AmazonLinuxImage {
    getImage(scope) {
        const result = super.getImage(scope);
        jest.spyOn(result.userData, 'addOnExitCommands');
        jest.spyOn(result.userData, 'addExecuteFileCommand');
        return result;
    }
}
describe('DeploymentInstance', () => {
    describe('defaults', () => {
        let app;
        let depStack;
        let vpc;
        let stack;
        let target;
        beforeAll(() => {
            // GIVEN
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'DefaultsStack');
            target = new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
                vpc,
            });
        });
        describe('Auto-Scaling Group', () => {
            // Only one ASG is deployed. This is an anchor for the tests that follow. Each test is independent and not
            // guaranteed to match on the same resource in the CloudFormation template. Having a test that asserts a single
            // ASG makes these assertions linked
            test('deploys a single Auto-Scaling Group', () => {
                // THEN
                assertions_1.Template.fromStack(stack).resourceCountIs('AWS::AutoScaling::AutoScalingGroup', 1);
            });
            test('MaxSize is 1', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    MaxSize: '1',
                });
            });
            test('MinSize is 1', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                });
            });
            test('uses private subnets', () => {
                // GIVEN
                const privateSubnetIDs = vpc.selectSubnets({ subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS }).subnetIds;
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    VPCZoneIdentifier: assertions_1.Match.arrayWith([
                        ...stack.resolve(privateSubnetIDs),
                    ]),
                });
            });
            test('waits 15 minutes for one signal', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
                    CreationPolicy: {
                        ResourceSignal: {
                            Count: 1,
                            Timeout: 'PT15M',
                        },
                    },
                });
            });
            test('sets replacing update policy', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
                    UpdatePolicy: {
                        AutoScalingReplacingUpdate: {
                            WillReplace: true,
                        },
                        AutoScalingScheduledAction: {
                            IgnoreUnmodifiedGroupSizeProperties: true,
                        },
                    },
                });
            });
            test('uses Launch Configuration', () => {
                // GIVEN
                const launchConfig = target.node.findChild('ASG').node.findChild('LaunchConfig');
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
                    LaunchConfigurationName: stack.resolve(launchConfig.ref),
                });
            });
        });
        describe('Launch Configuration', () => {
            // Only one ASG is deployed. This is an anchor for the tests that follow. Each test is independent and not
            // guaranteed to match on the same resource in the CloudFormation template. Having a test that asserts a single
            // ASG makes these assertions linked
            test('deploys a single Launch Configuration', () => {
                // THEN
                assertions_1.Template.fromStack(stack).resourceCountIs('AWS::AutoScaling::LaunchConfiguration', 1);
            });
            test('uses latest Amazon Linux machine image', () => {
                // GIVEN
                const amazonLinux = aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 });
                const imageId = stack.resolve(amazonLinux.getImage(stack)).imageId;
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                    ImageId: imageId,
                });
            });
            test('uses t3.small', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                    InstanceType: 't3.small',
                });
            });
            test('Uses created Security Group', () => {
                // GIVEN
                const securityGroup = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceSecurityGroup'));
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: [
                        stack.resolve(securityGroup.securityGroupId),
                    ],
                });
            });
            test('depends on policy', () => {
                // GIVEN
                const policy = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceRole')
                    .node.findChild('DefaultPolicy')
                    .node.defaultChild);
                // THEN
                assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::LaunchConfiguration', {
                    DependsOn: assertions_1.Match.arrayWith([
                        stack.resolve(policy.logicalId),
                    ]),
                });
            });
        });
        describe('Security Group', () => {
            test('creates Security Group in the desired VPC', () => {
                // THEN
                assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EC2::SecurityGroup', 1);
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroup', {
                    VpcId: stack.resolve(vpc.vpcId),
                });
            });
        });
        describe('ASG IAM role', () => {
            let instanceRole;
            beforeAll(() => {
                // GIVEN
                instanceRole = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceRole')
                    .node.defaultChild);
            });
            test('creates an instance profile', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::InstanceProfile', {
                    Roles: [
                        { Ref: stack.getLogicalId(instanceRole) },
                    ],
                });
            });
            test('creates a role that can be assumed by EC2', () => {
                // GIVEN
                const servicePrincipal = new iam.ServicePrincipal('ec2.amazonaws.com');
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Role', {
                    AssumeRolePolicyDocument: {
                        Statement: [
                            {
                                Action: 'sts:AssumeRole',
                                Effect: 'Allow',
                                Principal: {
                                    Service: stack.resolve(servicePrincipal.policyFragment.principalJson).Service[0],
                                },
                            },
                        ],
                    },
                });
            });
            test('can signal to CloudFormation', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: 'cloudformation:SignalResource',
                                Effect: 'Allow',
                                Resource: { Ref: 'AWS::StackId' },
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
            test('can write to the log group', () => {
                // GIVEN
                const logGroup = target.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    'logs:CreateLogStream',
                                    'logs:PutLogEvents',
                                ],
                                Effect: 'Allow',
                                Resource: stack.resolve(logGroup.logGroupArn),
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
            test('can fetch the CloudWatch Agent install script', () => {
                // GIVEN
                const cloudWatchAgentScriptAsset = (target
                    .node.findChild('CloudWatchConfigurationScriptAsset'));
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    's3:GetObject*',
                                    's3:GetBucket*',
                                    's3:List*',
                                ],
                                Effect: 'Allow',
                                Resource: stack.resolve([
                                    cloudWatchAgentScriptAsset.bucket.bucketArn,
                                    cloudWatchAgentScriptAsset.bucket.arnForObjects('*'),
                                ]),
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
            test('can fetch the CloudWatch Agent configuration file SSM parameter', () => {
                // GIVEN
                const cloudWatchConfigSsmParam = (target
                    .node.findChild('StringParameter'));
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    'ssm:DescribeParameters',
                                    'ssm:GetParameters',
                                    'ssm:GetParameter',
                                    'ssm:GetParameterHistory',
                                ],
                                Effect: 'Allow',
                                Resource: stack.resolve(cloudWatchConfigSsmParam.parameterArn),
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
            test('can fetch the CloudWatch Agent installer from S3', () => {
                // GIVEN
                const cloudWatchAgentInstallerBucket = aws_s3_1.Bucket.fromBucketArn(depStack, 'CloudWatchAgentInstallerBucket', `arn:aws:s3:::amazoncloudwatch-agent-${stack.region}`);
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    's3:GetObject*',
                                    's3:GetBucket*',
                                    's3:List*',
                                ],
                                Effect: 'Allow',
                                Resource: stack.resolve([
                                    cloudWatchAgentInstallerBucket.bucketArn,
                                    cloudWatchAgentInstallerBucket.arnForObjects('*'),
                                ]),
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
            test('can fetch GPG installer from RFDK dependencies S3 bucket', () => {
                // GIVEN
                const rfdkExternalDepsBucket = aws_s3_1.Bucket.fromBucketArn(depStack, 'RfdkExternalDependenciesBucket', `arn:aws:s3:::rfdk-external-dependencies-${stack.region}`);
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: [
                                    's3:GetObject*',
                                    's3:GetBucket*',
                                    's3:List*',
                                ],
                                Effect: 'Allow',
                                Resource: stack.resolve([
                                    rfdkExternalDepsBucket.bucketArn,
                                    rfdkExternalDepsBucket.arnForObjects('*'),
                                ]),
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
            test('can scale the Auto-Scaling Group', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: 'autoscaling:UpdateAutoScalingGroup',
                                Condition: {
                                    // This tag is added by RFDK to scope down the permissions of the policy for least-privilege
                                    StringEquals: { 'autoscaling:ResourceTag/resourceLogicalId': cdk.Names.uniqueId(target) },
                                },
                                Effect: 'Allow',
                                Resource: '*',
                            },
                            // The instance determines its Auto-Scaling Group by reading the tag created on the instance by the EC2
                            // Auto-Scaling service
                            {
                                Action: 'ec2:DescribeTags',
                                Effect: 'Allow',
                                Resource: '*',
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                });
            });
        });
        describe('CloudWatch Agent config SSM parameter', () => {
            test('configures log group', () => {
                // GIVEN
                const logGroup = target.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
                    Type: 'String',
                    Value: {
                        'Fn::Join': [
                            '',
                            assertions_1.Match.arrayWith([
                                '{"logs":{"logs_collected":{"files":{"collect_list":[{"log_group_name":"',
                                stack.resolve(logGroup.logGroupName),
                            ]),
                        ],
                    },
                });
            });
            test('configures cloud-init log', () => {
                // THEN
                assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
                    Type: 'String',
                    Value: {
                        'Fn::Join': [
                            '',
                            assertions_1.Match.arrayWith([
                                assertions_1.Match.stringLikeRegexp('.*"log_stream_name":"cloud-init-output-{instance_id}","file_path":"/var/log/cloud-init-output.log",.*'),
                            ]),
                        ],
                    },
                });
            });
        });
        describe('Tags resources with RFDK meta-data', () => {
            tag_helpers_1.testConstructTags({
                constructName: 'DeploymentInstance',
                createConstruct: () => {
                    return stack;
                },
                resourceTypeCounts: {
                    'AWS::EC2::SecurityGroup': 1,
                    'AWS::IAM::Role': 1,
                    'AWS::AutoScaling::AutoScalingGroup': 1,
                    'AWS::SSM::Parameter': 1,
                },
            });
        });
        // RFDK adds the resourceLogicalId tag to the Auto-Scaling Group in order to scope down the permissions of the
        // IAM policy given to the EC2 instance profile so that only that ASG can be scaled by the instance.
        test('Tagging for self-termination', () => {
            // THEN
            const matcher = tag_helpers_1.resourceTagMatcher('AWS::AutoScaling::AutoScalingGroup', 'resourceLogicalId', cdk.Names.uniqueId(target));
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', matcher);
        });
    });
    describe('User Data', () => {
        let app;
        let depStack;
        let vpc;
        let stack;
        let target;
        beforeAll(() => {
            // GIVEN
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'UserDataStack');
            // WHEN
            target = new deployment_instance_1.DeploymentInstance(stack, 'DeploymentInstanceNew', {
                vpc,
                // a hack to be able to spy on the user data's "addOnExitCommand" and "addExecuteFileCommand" methods.
                machineImage: new AmazonLinuxWithUserDataSpy(),
            });
        });
        test('configures self-termination', () => {
            // THEN
            expect(target.userData.addOnExitCommands).toHaveBeenCalledWith('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)', 'INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"', `ASG="$(aws --region ${stack.region} ec2 describe-tags --filters "Name=resource-id,Values=\${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"`, `aws --region ${stack.region} autoscaling update-auto-scaling-group --auto-scaling-group-name \${ASG} --min-size 0 --max-size 0 --desired-capacity 0`);
        });
        test('configures CloudWatch Agent', () => {
            // GIVEN
            const spy = target.userData.addExecuteFileCommand;
            const cloudWatchConfigSsmParam = (target
                .node.findChild('StringParameter'));
            // THEN
            // Should have been called
            expect(spy.mock.calls.length).toBeGreaterThanOrEqual(1);
            // The first call...
            const executeFileOptions = spy.mock.calls[0][0];
            // Should have been called with arguments
            const args = executeFileOptions.arguments;
            expect(args).not.toBeUndefined();
            const splitArgs = args.split(' ');
            // Should have three arguments
            expect(splitArgs).toHaveLength(3);
            // Specify the flag to install the CloudWatch Agent
            expect(splitArgs[0]).toEqual('-i');
            // Should pass the region
            expect(stack.resolve(splitArgs[1])).toEqual(stack.resolve(stack.region));
            // Should pass the SSM parameter containing the CloudWatch Agent configuration
            expect(stack.resolve(splitArgs[2])).toEqual(stack.resolve(cloudWatchConfigSsmParam.parameterName));
        });
    });
    describe('Custom::LogRetention.LogGroupName', () => {
        let app;
        let depStack;
        let vpc;
        let stack;
        beforeEach(() => {
            // We need a clean construct tree, because the tests use the same construct ID
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'Stack');
        });
        // GIVEN
        test.each([
            [
                {},
                // defaults expected final log group name
                `/renderfarm/${DEFAULT_CONSTRUCT_ID}`,
            ],
            [
                { logGroupName: 'foo' },
                // expected final log group name
                '/renderfarm/foo',
            ],
            [
                {
                    logGroupPrefix: 'logGroupPrefix',
                },
                // expected final log group name
                `logGroupPrefix${DEFAULT_CONSTRUCT_ID}`,
            ],
            [
                {
                    logGroupName: 'logGroupName',
                    logGroupPrefix: 'logGroupPrefix',
                },
                // expected final log group name
                'logGroupPrefixlogGroupName',
            ],
        ])('%s => %s', ({ logGroupName, logGroupPrefix }, expectedLogGroupName) => {
            // WHEN
            new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
                vpc,
                logGroupName,
                logGroupProps: logGroupPrefix ? { logGroupPrefix } : undefined,
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
                LogGroupName: expectedLogGroupName,
            });
        });
    });
    test('uses specified instance type', () => {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'TestStack');
        const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const instanceType = new aws_ec2_1.InstanceType('c5.large');
        // WHEN
        new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
            vpc,
            instanceType,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: instanceType.toString(),
        });
    });
    test('uses specified security group', () => {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'TestStack');
        const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const securityGroupId = 'securityGroupId';
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SecurityGroup', securityGroupId);
        // WHEN
        new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
            vpc,
            securityGroup,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            SecurityGroups: assertions_1.Match.arrayWith([
                securityGroupId,
            ]),
        });
    });
    describe('.selfTermination = false', () => {
        let app;
        let depStack;
        let vpc;
        let stack;
        let target;
        beforeAll(() => {
            // GIVEN
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'DefaultsStack');
            // Spy on user data method calls
            const machineImage = new AmazonLinuxWithUserDataSpy();
            const deploymentInstanceProps = {
                vpc,
                selfTerminate: false,
                machineImage,
            };
            // WHEN
            target = new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, deploymentInstanceProps);
        });
        test('does not add on-exit commands', () => {
            // THEN
            expect(target.userData.addOnExitCommands).not.toHaveBeenCalledWith(expect.arrayContaining([
                expect.stringMatching(/\baws\s+.*\bautoscaling\s+update-auto-scaling-group/),
            ]));
        });
        test('is not granted IAM permissions to scale the Auto-Scaling Group', () => {
            // GIVEN
            const instanceRole = (target
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
            const matcher = assertions_1.Match.objectLike({
                Properties: {
                    PolicyDocument: {
                        Statement: assertions_1.Match.arrayWith([
                            {
                                Action: 'autoscaling:UpdateAutoScalingGroup',
                                Condition: {
                                    // This tag is added by RFDK to scope down the permissions of the policy for least-privilege
                                    StringEquals: { 'autoscaling:ResourceTag/resourceLogicalId': cdk.Names.uniqueId(target) },
                                },
                                Effect: 'Allow',
                                Resource: '*',
                            },
                            // The instance determines its Auto-Scaling Group by reading the tag created on the instance by the EC2
                            // Auto-Scaling service
                            {
                                Action: 'ec2:DescribeTags',
                                Effect: 'Allow',
                                Resource: '*',
                            },
                        ]),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                },
            });
            // THEN
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Policy', matcher, 0);
        });
        test('does not tag for self-termination', () => {
            // THEN
            const matcher = tag_helpers_1.resourceTagMatcher('AWS::AutoScaling::AutoScalingGroup', 'resourceLogicalId', cdk.Names.uniqueId(target));
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', assertions_1.Match.not(matcher));
        });
    });
    // GIVEN
    describe('.executionTimeout is specified', () => {
        let app;
        let depStack;
        let vpc;
        let stack;
        const executionTimeout = cdk.Duration.minutes(30);
        beforeAll(() => {
            // GIVEN
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'DefaultsStack');
            const deploymentInstanceProps = {
                vpc,
                executionTimeout,
            };
            // WHEN
            new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, deploymentInstanceProps);
        });
        // THEN
        test('AWS::AutoScaling::AutoScalingGroup creation policy signal timeout is set accordingly', () => {
            assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
                CreationPolicy: {
                    ResourceSignal: {
                        Count: 1,
                        Timeout: executionTimeout.toIsoString(),
                    },
                },
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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