"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            pointInTimeRecovery: true,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${aws_cdk_lib_1.Names.uniqueId(this)}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${aws_cdk_lib_1.Names.uniqueId(this)} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: aws_cdk_lib_1.Duration.seconds(90),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        props.encryptionKey?.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(aws_cdk_lib_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new aws_cdk_lib_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into
 * Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated.
 * When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is
 * comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key
 * that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePem.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * - Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *   to only the principal(s) that require access.
 */
class X509CertificatePem extends X509CertificateBase {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        if ((props.validFor ?? 1) < 1 && !aws_cdk_lib_1.Token.isUnresolved(props.validFor)) {
            aws_cdk_lib_1.Annotations.of(this).addError('Certificates must be valid for at least one day.');
        }
        props.signingCertificate?.cert.grantRead(this.lambdaFunc);
        props.signingCertificate?.key.grantRead(this.lambdaFunc);
        props.signingCertificate?.passphrase.grantRead(this.lambdaFunc);
        props.signingCertificate?.certChain?.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: props.subject.o ?? 'AWS',
                OU: props.subject.ou ?? 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: props.encryptionKey?.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
            CertificateValidFor: props.validFor?.toString(),
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: aws_cdk_lib_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretCompleteArn: aws_cdk_lib_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretCompleteArn: aws_cdk_lib_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate
     */
    grantCertRead(grantee) {
        const result = this.cert.grantRead(grantee);
        this.certChain?.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase
     */
    grantFullRead(grantee) {
        const result = this.cert.grantRead(grantee);
        this.certChain?.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
_a = JSII_RTTI_SYMBOL_1;
X509CertificatePem[_a] = { fqn: "aws-rfdk.X509CertificatePem", version: "1.1.0" };
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be
 * provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and
 * the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly
 * generated and stored in a Secret.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePkcs12.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the CustomResource.
 * - Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * - Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *   to only the principal(s) that require access.
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: props.encryptionKey?.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: aws_cdk_lib_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
_b = JSII_RTTI_SYMBOL_1;
X509CertificatePkcs12[_b] = { fqn: "aws-rfdk.X509CertificatePkcs12", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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