"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const constructs_1 = require("constructs");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that
 * has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Elastic Network Interface (ENI).
 * - Security Group for the ASG.
 * - Instance Role and corresponding IAM Policy.
 * - SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack.
 * - Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda that is deployed through this construct has broad IAM permissions to attach any Elastic
 *   Network Interface (ENI) to any instance. You should not grant any additional actors/principals the ability
 *   to modify or execute this Lambda.
 * - The SNS Topic that is deployed through this construct controls the execution of the Lambda discussed above.
 *   Principals that can publish messages to this SNS Topic will be able to trigger the Lambda to run. You should
 *   not allow any additional principals to publish messages to this SNS Topic.
 */
class StaticPrivateIpServer extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            signals: props.resourceSignalTimeout ? aws_autoscaling_1.Signals.waitForCount(1, { timeout: props.resourceSignalTimeout }) : undefined,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: aws_cdk_lib_1.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance
     * in this construct's AutoScalingGroup when a new instance is launched.
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     */
    setupLifecycleEventHandlerFunction() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_16_X,
                description: `Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = aws_cdk_lib_1.Names.uniqueId(eventHandler);
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        aws_cdk_lib_1.Tags.of(this.autoscalingGroup).add(tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPrincipalPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPrincipalPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events. Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns { topic: Topic, role: Role }
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = aws_cdk_lib_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For RFDK instance-launch notifications for stack '${stack.stackName}'`,
            });
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
_a = JSII_RTTI_SYMBOL_1;
StaticPrivateIpServer[_a] = { fqn: "aws-rfdk.StaticPrivateIpServer", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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