"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstaller = exports.MongoDbSsplLicenseAcceptance = exports.MongoDbVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
/**
 * Versions of MongoDB Community Edition that the {@link MongoDbInstaller} is
 * able to install.
 */
var MongoDbVersion;
(function (MongoDbVersion) {
    /**
     * MongoDB 3.6 Community Edition.
     * See: https://docs.mongodb.com/v3.6/introduction/
     */
    MongoDbVersion["COMMUNITY_3_6"] = "3.6";
})(MongoDbVersion = exports.MongoDbVersion || (exports.MongoDbVersion = {}));
/**
 * Choices for signifying the user's stance on the terms of the SSPL.
 * See: https://www.mongodb.com/licensing/server-side-public-license
 */
var MongoDbSsplLicenseAcceptance;
(function (MongoDbSsplLicenseAcceptance) {
    /**
     * The user signifies their explicit rejection of the tems of the SSPL.
     */
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_REJECTS_SSPL"] = 0] = "USER_REJECTS_SSPL";
    /**
     * The user signifies their explicit acceptance of the terms of the SSPL.
     */
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_ACCEPTS_SSPL"] = 1] = "USER_ACCEPTS_SSPL";
})(MongoDbSsplLicenseAcceptance = exports.MongoDbSsplLicenseAcceptance || (exports.MongoDbSsplLicenseAcceptance = {}));
/**
 * This class provides a mechanism to install a version of MongoDB Community Edition during the
 * initial launch of an instance. MongoDB is installed from the official sources using the system
 * package manger (yum). It installs the mongodb-org metapackage which will install the following packages:
 * 1) mongodb-org-mongos;
 * 2) mongodb-org-server;
 * 3) mongodb-org-shell; and
 * 4) mongodb-org-tools.
 *
 * Successful installation of MongoDB with this class requires:
 * 1) Explicit acceptance of the terms of the SSPL license, under which MongoDB is distributed; and
 * 2) The instance on which the installation is being performed is in a subnet that can access
 * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org
 *
 * Resources Deployed
 * ------------------------
 * - A CDK Asset package containing the installation scripts is deployed to your CDK staging bucket.
 *
 * Security Considerations
 * ------------------------
 * - Since this class installs MongoDB from official sources dynamically during instance start-up, it is succeptable
 *   to an attacker compromising the official MongoDB Inc. distribution channel for MongoDB. Such a compromise may
 *   result in the installation of unauthorized MongoDB binaries. Executing this attack would require an attacker
 *   compromise both the official installation packages and the MongoDB Inc. gpg key with which they are signed.
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class MongoDbInstaller {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        // Legal requirement: Users of this class must agree to the terms of the SSPL, without exception.
        // Developer note: It is a legal requirement that the default be USER_REJECTS_SSPL, so this check
        // must throw an error for every value except USER_ACCEPTS_SSPL.
        if (props.userSsplAcceptance !== MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL) {
            throw new Error(MongoDbInstaller.SSPL_ACCEPT_MESSAGE);
        }
    }
    /**
     * Install MongoDB to the given instance at instance startup. This is accomplished by
     * adding scripting to the instance's UserData to install MongoDB.
     *
     * Notes:
     * 1) The instance on which the installation is being performed must be in a subnet that can access
     * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org; and
     * 2) At this time, this method only supports installation onto instances that are running an operating system
     * that is compatible with x86-64 RedHat 7 -- this includes Amazon Linux 2, RedHat 7, and CentOS 7.
     *
     * @param target The target instance onto which to install MongoDB.
     */
    installOnLinuxInstance(target) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        const installerScriptAsset = this.installerAssetSingleton();
        installerScriptAsset.grantRead(target.grantPrincipal);
        const installerScript = target.userData.addS3DownloadCommand({
            bucket: installerScriptAsset.bucket,
            bucketKey: installerScriptAsset.s3ObjectKey,
        });
        target.userData.addCommands(`bash ${installerScript}`);
    }
    /**
     * Fetch the Asset singleton for the installation script, or generate it if needed.
     */
    installerAssetSingleton() {
        const stack = aws_cdk_lib_1.Stack.of(this.scope);
        const uuid = '5b141ac9-fde5-45d8-9961-b7108fb3b18a';
        const uniqueId = 'MongoDbInstallerAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', this.props.version, 'installMongoDb.sh'),
        });
    }
}
exports.MongoDbInstaller = MongoDbInstaller;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstaller[_a] = { fqn: "aws-rfdk.MongoDbInstaller", version: "1.1.0" };
/**
 * The SSPL licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the SSPL.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
MongoDbInstaller.SSPL_ACCEPT_MESSAGE = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
//# sourceMappingURL=data:application/json;base64,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