"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExportingLogGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * This construct takes the name of a CloudWatch LogGroup and will either create it if it doesn't already exist,
 * or reuse the existing one. It also creates a regularly scheduled lambda that will export LogEvents to S3
 * before they expire in CloudWatch.
 *
 * It's used for cost-reduction, as it is more economical to archive logs in S3 than CloudWatch when
 * retaining them for more than a week.
 * Note, that it isn't economical to export logs to S3 if you plan on storing them for less than
 * 7 days total (CloudWatch and S3 combined).
 *
 * Resources Deployed
 * ------------------------
 * - The Lambda SingletonFunction that checks for the existence of the LogGroup.
 * - The CloudWatch LogGroup (if it didn't exist already).
 * - The CloudWatch Alarm watching log exportation failures.
 * - The CloudWatch Event Rule to schedule log exportation.
 * - The Lambda SingletonFunction, with role, to export log groups to S3 by schedule.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
class ExportingLogGroup extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * UUID needed to identify the SingletonFunction for the log exporter.
         */
        this.LOG_EXPORTER_UUID = '6382448c-e4b2-42e9-b14f-a0a9ccdb198e';
        /**
         * Duration of time between export task Lambda runs.
         */
        this.EXPORT_TASK_FREQUENCY = aws_cdk_lib_1.Duration.hours(1);
        /**
         * Default value for the number of days to retain logs in CloudWatch for.
         */
        this.RETENTION_DEFAULT = aws_logs_1.RetentionDays.THREE_DAYS;
        const retentionInDays = props.retention ? props.retention : this.RETENTION_DEFAULT;
        const exportLogsFunction = this.setupLogsExporter();
        this.exportErrorAlarm = exportLogsFunction.metricErrors().createAlarm(this, 'LogExporterFailureAlarm', {
            evaluationPeriods: 1,
            threshold: 1,
        });
        // Define log retention retry options to reduce the risk of the rate exceed error
        // as the default create log group TPS is only 5. Make sure to set the timeout of log retention function
        // to be greater than total retry time. That's because if the function that is used for a custom resource
        // doesn't exit properly, it'd end up in retries and may take cloud formation an hour to realize that
        // the custom resource failed.
        const logRetention = new aws_logs_1.LogRetention(this, 'LogRetention', {
            logGroupName: props.logGroupName,
            retention: retentionInDays,
            logRetentionRetryOptions: {
                base: aws_cdk_lib_1.Duration.millis(200),
                maxRetries: 7,
            },
        });
        // referenced from cdk code: https://github.com/aws/aws-cdk/blob/v2.33.0/packages/@aws-cdk/aws-logs/lib/log-retention.ts#L116
        const logRetentionFunctionConstructId = 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8a';
        const logRetentionFunction = aws_cdk_lib_1.Stack.of(this).node.findChild(logRetentionFunctionConstructId);
        const cfnFunction = logRetentionFunction.node.defaultChild;
        cfnFunction.addPropertyOverride('Timeout', 30);
        this.logGroup = aws_logs_1.LogGroup.fromLogGroupArn(scope, `${props.logGroupName}LogGroup`, logRetention.logGroupArn);
        this.logGroup.grant(exportLogsFunction, 'logs:CreateExportTask');
        const scheduledLogExportation = new aws_events_1.Rule(this, 'LogExporterRule', {
            schedule: aws_events_1.Schedule.rate(this.EXPORT_TASK_FREQUENCY),
        });
        scheduledLogExportation.addTarget(new aws_events_targets_1.LambdaFunction(exportLogsFunction, {
            event: aws_events_1.RuleTargetInput.fromObject({
                BucketName: props.bucketName,
                ExportFrequencyInHours: this.EXPORT_TASK_FREQUENCY.toHours(),
                LogGroupName: props.logGroupName,
                RetentionInHours: retentionInDays.valueOf() * 24,
            }),
        }));
    }
    setupLogsExporter() {
        const exportLogsFunction = new aws_lambda_1.SingletonFunction(this, 'LogExporterFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'export-logs')),
            handler: 'index.handler',
            lambdaPurpose: 'LogGroupExporter',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            uuid: this.LOG_EXPORTER_UUID,
        });
        exportLogsFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['logs:DescribeExportTasks'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: ['*'],
        }));
        return exportLogsFunction;
    }
}
exports.ExportingLogGroup = ExportingLogGroup;
_a = JSII_RTTI_SYMBOL_1;
ExportingLogGroup[_a] = { fqn: "aws-rfdk.ExportingLogGroup", version: "1.1.0" };
//# sourceMappingURL=data:application/json;base64,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