"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const common_functions_1 = require("./common/common-functions");
/**
 * @stability stable
 */
class BaseFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        /**
         * @stability stable
         */
        this.zoneName = '';
        /**
         * @stability stable
         */
        this.hasExternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.hasInternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        this.hasSpotCapacity = false;
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.enableLoadBalancerAlias = ((_b = props.route53Ops) === null || _b === void 0 ? void 0 : _b.enableLoadBalancerAlias) != false;
        if (props.vpc && props.cluster) {
            throw new Error('Cannot specify vpc and cluster at the same time');
        }
        if (props.clusterProps && props.cluster) {
            throw new Error('Cannot specify clusterProps and cluster at the same time');
        }
        this.vpc = props.cluster ? props.cluster.vpc : (_c = props.vpc) !== null && _c !== void 0 ? _c : common_functions_1.getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.external) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.internal) {
                this.hasInternalLoadBalancer = true;
            }
        });
        const cluster = (_d = props.cluster) !== null && _d !== void 0 ? _d : new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
            ...props.clusterProps,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        if (props.spot == true)
            this.hasSpotCapacity = true;
        props.tasks.forEach(t => {
            var _b, _c, _d, _e;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                serviceName: t.serviceName,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
                circuitBreaker: props.circuitBreaker != false ? {
                    rollback: true,
                } : undefined,
                maxHealthyPercent: t.maxHealthyPercent,
                minHealthyPercent: t.minHealthyPercent,
                healthCheckGracePeriod: t.healthCheckGracePeriod,
            });
            this.service.push(svc);
            /**
             * determine if we have spot capacity in this cluster
             * scenario 1: FARGATE_SPOT with weight > 0
             * scenario 2: FARGATE_SPOT with base > 0
             * scenario 3: props.spot = true
             */
            (_e = t.capacityProviderStrategy) === null || _e === void 0 ? void 0 : _e.forEach(s => {
                if (s.capacityProvider == 'FARGATE_SPOT' && ((s.weight && s.weight > 0)
                    || (s.base && s.base > 0))) {
                    this.hasSpotCapacity = true;
                }
            });
        });
        // ensure the dependency
        if (!props.cluster) {
            const cp = this.node.tryFindChild('Cluster');
            this.service.forEach(s => {
                s.node.addDependency(cp);
            });
        }
        ;
        // Route53
        this.zoneName = (_f = (_e = props.route53Ops) === null || _e === void 0 ? void 0 : _e.zoneName) !== null && _f !== void 0 ? _f : 'svc.local';
        // spot termination handler by default
        if (this.hasSpotCapacity && props.spotTerminationHandler !== false) {
            this.createSpotTerminationHandler(cluster);
        }
        // add solution ID for the stack
        if (!cdk.Stack.of(this).templateOptions.description) {
            cdk.Stack.of(this).templateOptions.description = '(SO8030) - AWS CDK stack with cdk-fargate-patterns';
        }
    }
    createSpotTerminationHandler(cluster) {
        // create the handler
        const handler = new lambda.DockerImageFunction(this, 'SpotTermHandler', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/spot-term-handler')),
            timeout: cdk.Duration.seconds(20),
        });
        // create event rule
        const rule = new events.Rule(this, 'OnTaskStateChangeEvent', {
            eventPattern: {
                source: ['aws.ecs'],
                detailType: ['ECS Task State Change'],
                detail: {
                    clusterArn: [cluster.clusterArn],
                },
            },
        });
        rule.addTarget(new event_targets.LambdaFunction(handler));
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'elasticloadbalancing:DeregisterTargets',
                'ec2:DescribeSubnets',
            ],
            resources: ['*'],
        }));
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.BaseFargateService = BaseFargateService;
_a = JSII_RTTI_SYMBOL_1;
BaseFargateService[_a] = { fqn: "cdk-fargate-patterns.BaseFargateService", version: "0.3.68" };
//# sourceMappingURL=data:application/json;base64,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