"""
This module analyzes and manipulates macrostate probability distributions generated by FEASST
flat-histogram simulations.
See more information on FEASST at https://doi.org/10.18434/M3S095

"""

import numpy as np
import matplotlib.pyplot as plt
import pandas as pd
from scipy.optimize import minimize_scalar

class MacrostateDistribution:
    """
    The natural logarithm of the macrostate probability from a flat histogram simulation.
    """
    def __init__(self, macrostates, ln_prob, minimum_smoothing=10):
        """
        Constructs all the necessary attributes for a MacrostateDistribution.

        :param pandas.Series macrostates:
            In the grand canonical ensemble, the macrostates are the number of particles.
        :param pandas.Series ln_prob:
            The natural logarithm of the probability of observing each macrostate.
        :param int minimum_smoothing:
            During the search for the minimum, the number of macrostates that the minimum
            must be smaller than in both directions.
            Increase this if the data is noisey or if there are undulations in the ln_prob.
        """
        self.set_macrostates(macrostates)
        self.set_ln_prob(ln_prob)
        self.set_minimum_smoothing(minimum_smoothing)

    def set_macrostates(self, macrostates):
        """Set the macrostates."""
        self.macrostates = pd.Series(macrostates)

    def normalize(self):
        """Set the sum of the probabilities to one."""
        self.ln_prob -= np.log(sum(np.exp(self.ln_prob)))

    def set_ln_prob(self, ln_prob):
        """
        Set the natural logrithm of the probabilities, and always normalize them.
        If the macrostates have not been set, assume macrostates [0, 1, 2, ... len(ln_prob) - 1].
        If the macrostates have been set, assert that the ln_prob is of the same size.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> lnpi.set_ln_prob([0])
        Traceback (most recent call last):
            ...
        AssertionError
        """
        try:
            self.macrostates
        except AttributeError:
            self.macrostates = range(0, len(ln_prob) - 1)
        assert len(self.macrostates) == len(ln_prob)
        self.ln_prob = pd.Series(ln_prob)
        self.normalize()

    def to_csv(self, file_name):
        """
        Write MacrostateDistribution to a csv file.

        >>> import pandas as pd
        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> lnpi.ln_prob[0]
        -137.1742726463091
        >>> lnpi.minimum_smoothing
        10
        >>> lnpi.set_minimum_smoothing(20)
        >>> lnpi.to_csv('tmp.csv')
        >>> lnpi2 = macrostate_distribution.read_csv('tmp.csv')
        >>> assert lnpi2.ln_prob[0] == lnpi.ln_prob[0]
        >>> lnpi.minimum_smoothing
        20
        """
        dataframe = pd.DataFrame({'state': self.macrostates,
                                  'ln_prob': self.ln_prob})
        dataframe.to_csv(file_name)
        with open(file_name, 'r+') as file:
            content = file.read()
            new_line = "#\"minimum_smoothing\":"+str(self.minimum_smoothing)+"\n"
            file.seek(0)
            file.write(new_line + content)

    def average_macrostate(self):
        """
        Return the average macrostate.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> lnpi.average_macrostate()
        437.6820787231706
        """
        return (np.exp(self.ln_prob) * self.macrostates).sum()

    def set_minimum_smoothing(self, minimum_smoothing):
        '''Set the number of macrostates to smooth the minimum search.'''
        self.minimum_smoothing = minimum_smoothing

    def plot(self):
        '''Create a matplotlib.pyplot plot.'''
        plt.plot(self.macrostates, self.ln_prob)
        plt.xlabel('macrostate')
        plt.ylabel(r'$\ln\Pi$')
        #plt.show()

    def minimums(self):
        """
        Return the minimums of the ln_prob.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> lnpi.ln_prob[0]
        -137.1742726463091
        >>> lnpi.ln_prob.values[-1]
        -48.68048289563584
        >>> mins = lnpi.minimums()
        >>> len(mins)
        1
        >>> mins.values[0]
        150
        """
        mns = self.ln_prob[(self.ln_prob.shift(1) > self.ln_prob) &
                           (self.ln_prob.shift(-1) > self.ln_prob)]
        for shift in range(2, self.minimum_smoothing+1):
            mns = self.ln_prob[mns.notna() &
                               (self.ln_prob.shift(shift) > self.ln_prob) &
                               (self.ln_prob.shift(-shift) > self.ln_prob)]
        return self.macrostates[mns.notna().index]

    def split(self, minimum=-1):
        """
        Return two MacrostateDistribution by splitting the current object at a minimum.

        :param int minimum: If -1, find the last minimum. Otherwise, supply it.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> vapor, liquid = lnpi.split()
        >>> vapor.average_macrostate()
        1.42126470419546
        >>> liquid.average_macrostate()
        437.6820787231706
        """
        if minimum == -1:
            minimum = self.minimums().values[-1]
        return [MacrostateDistribution(macrostates=self.macrostates[:minimum],
                       ln_prob=self.ln_prob[:minimum]),
                MacrostateDistribution(macrostates=self.macrostates[minimum+1:],
                       ln_prob=self.ln_prob[minimum+1:])]


    def reweight(self, delta_beta_mu, inplace=False):
        """
        Reweight the ln_prob by delta_beta_mu.
        If inplace, replace current ln_prob with the reweighted ones.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> reweight = lnpi.reweight(-1)
        >>> reweight.ln_prob[0]
        -0.5010868690523107
        >>> reweight = lnpi.reweight(1.5)
        >>> reweight.ln_prob[0]
        -811.7401044444931
        """
        # avoid negative log by subtracting min
        lnpi_rw = self.ln_prob + self.macrostates*delta_beta_mu - self.ln_prob.min()
        # shift to small exp before normalizin# shift to small exp before normalizingg
        if lnpi_rw.max() > 0:
            lnpi_rw -= lnpi_rw.max()
        # normalize
        lnpi_rw -= np.log(sum(np.exp(lnpi_rw)))
        if inplace:
            self.set_ln_prob(lnpi_rw)
        return MacrostateDistribution(macrostates=self.macrostates,
                      ln_prob=lnpi_rw,
                      minimum_smoothing=self.minimum_smoothing)

    def equilibrium_objective(self, delta_beta_mu):
        """
        Return an objective function to minimize in order to find equilibrium.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> lnpi.equilibrium_objective(-1)
        3.818281828459045
        """
        #print('delta_beta_mu', delta_beta_mu)
        lnpi_rw = self.reweight(delta_beta_mu)
        mins = lnpi_rw.minimums()
        #print('num mins', len(mins))
        if len(mins) == 0:
            return_val = 2.+(lnpi_rw.ln_prob[0] - lnpi_rw.ln_prob.values[-1])**2
        elif len(mins) == 1:
            mns = mins.values[0]
            p_lo = np.exp(lnpi_rw.ln_prob[:mns]).sum()
            p_hi = np.exp(lnpi_rw.ln_prob[mns+1:]).sum()
            # if one side is greatly favored over the other, numerical pecision
            # becomes an issue. Instead, nudge toward conjugate that alters this disaparity
            if abs(p_lo - 1) < 1e-6:
                return_val = 1.1+np.exp(-delta_beta_mu)
            elif abs(p_hi - 1) < 1e-6:
                return_val = 1.1+np.exp(delta_beta_mu)
            else:
                return_val = abs(p_lo-p_hi)
        else:
            #lnpi_rw.plot()
            #plt.show()
            assert False # more than one minima
        return return_val

    def equilibrium(self, delta_beta_mu_guess=-0.1, tol=1e-6):
        """
        Reweight ln_prob to equilibrium and return the delta_beta_mu of the equilibrium condition.

        :param float delta_beta_mu_guess: Provide a guess for delta_beta_mu which brings the ln_prob
            closer to equilibrium.
        :param float tol: Tolerance of minimization algorithm.

        >>> from pyfeasst import macrostate_distribution
        >>> lnpi = macrostate_distribution.read_csv('../../tests/lnpi.csv')
        >>> lnpi.equilibrium(delta_beta_mu_guess=-1, tol=1e-4)
        -0.31323799472042363
        >>> vapor, liquid = lnpi.split()
        >>> vapor.average_macrostate()/8**3
        0.0020003471025686094
        >>> liquid.average_macrostate()/8**3
        0.8430480261823022
        """
        res = minimize_scalar(self.equilibrium_objective, tol=tol, bracket=(0, delta_beta_mu_guess))
        delta_beta_mu_equilibrium = res["x"]
        self.reweight(delta_beta_mu_equilibrium, inplace=True)
        return delta_beta_mu_equilibrium

def read_csv(file_name, macrostate_name='state', ln_prob_name='ln_prob'):
    '''
    Read a MacrostateDistribution from a csv file.
    Obtain minimum_smoothing as commented dictionary on the first line in the following format:
    #"minimum_smoothing":10

    :param str macrostate_name: The name of the column for the macrostates.
    :param str ln_prob_name: The name of the column for the natural logarithm of the probability.
    '''
    dataframe = pd.read_csv(file_name, comment="#")
    lnpi = MacrostateDistribution(macrostates=dataframe[macrostate_name], ln_prob=dataframe[ln_prob_name])
    with open(file_name, 'r') as file:
        lines = file.readlines()
        exec('params={'+lines[0][1:].strip()+'}', None, globals())
        lnpi.set_minimum_smoothing(params['minimum_smoothing'])
    return lnpi

def splice(ln_probs, extra_overlap=0):
    """
    Return a MacrostateDistribution that is spliced together from multiple MacrostateDistribution.
    Each MacrostateDistribution must overlap by atleast one macrostate with its neighbor.
    If macrostates overlap by more than one (extra_overlap), simply drop the
    macrostates in the upper MacrostateDistribution.

    >>> from pyfeasst import macrostate_distribution
    >>> lnpis=list()
    >>> for i in range(4):
    ...     dat = pd.read_csv('../../tests/lnpi'+str(i)+'.csv')
    ...     lnpis.append(macrostate_distribution.MacrostateDistribution(macrostates=dat['state'], ln_prob=dat['ln_prob']))
    >>> lnpi = macrostate_distribution.splice(lnpis)
    >>> lnpi.set_minimum_smoothing(50)
    >>> len(lnpi.ln_prob)
    631
    >>> lnpi.average_macrostate()
    596.292899018217
    >>> len(lnpi.minimums())
    0
    >>> lnpi.equilibrium(delta_beta_mu_guess=-1.5)
    -1.2768125968551267
    >>> vapor, liquid = lnpi.split()
    >>> vapor.average_macrostate()
    0.03172081147590359
    >>> liquid.average_macrostate()
    523.3553081498678
    """
    max_minimum_smoothing = 0
    for index, lnpi in enumerate(ln_probs):
        if max_minimum_smoothing < lnpi.minimum_smoothing:
            max_minimum_smoothing = lnpi.minimum_smoothing
        vals = lnpi.ln_prob
        if index == 0:
            lnpi_tot = vals.values
        else:
            prev_last_index = len(ln_probs[index-1].ln_prob) - 1
            shift = ln_probs[index-1].ln_prob.values[prev_last_index-extra_overlap] - vals.values[0]
            vals += shift
            for _ in range(1 + extra_overlap):
                vals.pop(0)
            lnpi_tot = np.concatenate([lnpi_tot, vals.values])
    return MacrostateDistribution(macrostates=list(range(0, len(lnpi_tot))),
                  ln_prob=lnpi_tot.tolist(),
                  minimum_smoothing=max_minimum_smoothing)

if __name__ == "__main__":
    import doctest
    doctest.testmod()
