"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const domainNames = props.recordNames || [props.zone.zoneName];
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            });
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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