"""Get Report of a workflow from the project"""

import json
from collections.abc import Sequence
from pathlib import Path
from tempfile import TemporaryDirectory
from typing import Any

from cmem.cmempy.api import get_json
from cmem.cmempy.config import get_di_api_endpoint
from cmem_plugin_base.dataintegration.context import ExecutionContext
from cmem_plugin_base.dataintegration.description import Icon, Plugin, PluginParameter
from cmem_plugin_base.dataintegration.entity import (
    Entities,
)
from cmem_plugin_base.dataintegration.plugins import WorkflowPlugin
from cmem_plugin_base.dataintegration.ports import FixedNumberOfInputs, FixedSchemaPort
from cmem_plugin_base.dataintegration.typed_entities.file import FileEntitySchema, LocalFile
from cmem_plugin_base.dataintegration.utils import setup_cmempy_user_access

from cmem_plugin_wfreports.workflow_type import WorkflowParameterType


@Plugin(
    label="Get workflow report",
    plugin_id="cmem_plugin_workflows_get_report",
    description="Output the last report of a workflow as a JSON file.",
    documentation="""
This workflow operator retrieves the most recent execution report of a specified workflow
and outputs it as a JSON file.

The plugin queries the backend API to list all available reports for the given workflow,
sorts them by timestamp, and downloads the latest report. The report contains detailed information
about the workflow execution, including task results, execution times, and any errors or warnings.

## Output

The plugin outputs a single JSON file entity containing the complete workflow execution report.

## Usage

This operator is useful for:
- Monitoring workflow execution results
- Creating audit trails of workflow runs
- Archiving execution reports for compliance purposes
- Feeding execution data into downstream analysis tasks
""",
    icon=Icon(package=__package__, file_name="get_report.svg"),
    parameters=[
        PluginParameter(
            name="workflow_id",
            label="Workflow",
            param_type=WorkflowParameterType(),
        ),
    ],
)
class GetReport(WorkflowPlugin):
    """Get Report Workflow Plugin"""

    def __init__(self, workflow_id: str) -> None:
        self.workflow_id = workflow_id
        self.download_dir = Path(TemporaryDirectory().name)
        self.output_schema = FileEntitySchema()
        self.input_ports = FixedNumberOfInputs([])
        self.output_port = FixedSchemaPort(schema=self.output_schema)

    def list_reports(self, context: ExecutionContext, workflow_id: str) -> list[dict[str, str]]:
        """List the workflow reports."""
        setup_cmempy_user_access(context=context.user)
        # list?projectId=CMEM7071_b0f29341d3fdb8c0&taskId=DemofailedRESTrequests_67dcee4fe1696181
        api_base = get_di_api_endpoint() + "/api/workspace/reports/list"
        response: list[dict[str, str]] = get_json(
            api_base,
            params={"projectId": context.task.project_id(), "taskId": workflow_id},
        )
        return response

    def get_report(self, context: ExecutionContext, workflow_id: str, time: str) -> dict[str, Any]:
        """Get Workflow Report"""
        setup_cmempy_user_access(context=context.user)
        api_base = get_di_api_endpoint() + "/api/workspace/reports/report"
        report: dict[str, Any] = get_json(
            api_base,
            params={"projectId": context.task.project_id(), "taskId": workflow_id, "time": time},
        )
        return report

    def get_last_report(self, context: ExecutionContext) -> list[LocalFile]:
        """Get the last workflow report."""
        sorted_reports = sorted(
            self.list_reports(context=context, workflow_id=self.workflow_id),
            key=lambda report: report["time"],
        )
        last_report = sorted_reports[-1]
        report = self.get_report(
            context=context, workflow_id=self.workflow_id, time=last_report["time"]
        )
        self.download_dir.mkdir(parents=True, exist_ok=True)
        report_file = self.download_dir / "report.json"
        with report_file.open("w") as file:
            json.dump(report, file)
        return [LocalFile(str(report_file), "application/json")]

    def execute(
        self,
        inputs: Sequence[Entities],  # noqa: ARG002
        context: ExecutionContext,
    ) -> Entities:
        """Run the workflow operator."""
        entities = [self.output_schema.to_entity(file) for file in self.get_last_report(context)]
        return Entities(entities=iter(entities), schema=self.output_schema)
