# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/03_utils.ipynb (unless otherwise specified).

__all__ = ['get_name', 'find_contours', 'download_file_from_google_drive', 'mkdir', 'put_text', 'images_to_video',
           'get_paths', 'video_to_images', 'av_v2i', 'TimeLoger', 'identify', 'memoize']

# Cell
import os
from glob import glob
import os
import cv2
import os.path as osp
from tqdm import tqdm
import mmcv
from fastcore.script import call_parse, Param
from .process import multi_thread

def get_name(path):
    path = osp.basename(path).split('.')[:-1]
    return '.'.join(path)


def find_contours(thresh):
    """
        Get contour of a binary image
            Arguments:
                thresh: binary image
            Returns:
                Contours: a list of contour
                Hierarchy:

    """
    try:
        contours, hierarchy = cv2.findContours(thresh, cv2.RETR_TREE,
                                            cv2.CHAIN_APPROX_SIMPLE)
        return contours, hierarchy[0]
    except:
        return None, None


@call_parse
def download_file_from_google_drive(id_or_link: Param("Link or file id"), destination: Param("Path to the save file")):
    if "https" in id_or_link:
        x = id_or_link
        id = x.split("/")[x.split("/").index("d")+1]
    else:
        id = id_or_link
    print("Download from id:", id)
    import requests

    def get_confirm_token(response):
        for key, value in response.cookies.items():
            if key.startswith('download_warning'):
                return value

        return None

    def save_response_content(response, destination):
        CHUNK_SIZE = 32768

        with open(destination, "wb") as f:
            for chunk in response.iter_content(CHUNK_SIZE):
                if chunk:  # filter out keep-alive new chunks
                    f.write(chunk)

    URL = "https://docs.google.com/uc?export=download"

    session = requests.Session()

    response = session.get(URL, params={'id': id}, stream=True)
    token = get_confirm_token(response)

    if token:
        params = {'id': id, 'confirm': token}
        response = session.get(URL, params=params, stream=True)

    save_response_content(response, destination)
    print("Done ->", destination)
    return osp.abspath(destination)


def mkdir(path):
    os.makedirs(path, exist_ok=True)


def put_text(image, pos, text, color=(255, 255, 255)):
    return cv2.putText(image, text, pos, cv2.FONT_HERSHEY_SIMPLEX, 1.0,
                       color, 2)


def images_to_video(
        images,
        out_path=None,
        fps:int= 30,
        no_sort= False,
        max_num_frame= 10e12,
        resize_rate = 1,
        with_text = False,
        text_is_date= False,
        verbose=True,
        ):

    if out_path is None:
        assert isinstance(images, str), "No out_path specify, you need to input a string to a directory"
        out_path = images+'.mp4'
    if isinstance(images, str) and os.path.isdir(images):
        from glob import glob
        images = glob(os.path.join(images, "*.jpg")) + \
            glob(os.path.join(images, "*.png"))+glob(os.path.join(images, "*.jpeg"))

    imgs = []

    def get_num(s):
        try:
            s = os.path.basename(s)
            num = int(''.join([c for c in s if c.isdigit()]))
        except:
            num = s
        return num

    def f(img_or_path):
        if isinstance(img_or_path, str):
            name = os.path.basename(img_or_path)
            img = mmcv.imread(img_or_path)
            img = cv2.resize(img, output_size)
            assert img is not None, img_or_path
            if with_text:
                if text_is_date:
                    from datetime import datetime
                    name = name.split('.')[0].split('_')
                    f = float('{}.{}'.format(*name))
                    name = str(datetime.fromtimestamp(f))
                img = put_text(img, (20, 20), name)
        else:
            img = img_or_path
        return img


    if not no_sort and isinstance(images[0], str):
        images = list(sorted(images, key=get_num))

    max_num_frame = int(max_num_frame)
    max_num_frame = min(len(images), max_num_frame)

    h, w = mmcv.imread(images[0]).shape[:2]
    output_size = (int(w*resize_rate), int(h*resize_rate))
    if out_path.endswith('.mp4'):
        fourcc = cv2.VideoWriter_fourcc(*'MP4V')
        out = cv2.VideoWriter(out_path, fourcc, fps, output_size)
    elif out_path.endswith('.avi'):
        out = cv2.VideoWriter(out_path, cv2.VideoWriter_fourcc(*'DIVX'), fps, output_size)
    else:
        raise NotImplementedError
    images = images[:max_num_frame]

    images = multi_thread(f, images, desc='Reading images', verbose=verbose)
    if verbose:
        print("Write video, output_size:", output_size)
        pbar = mmcv.ProgressBar(len(images))
    for img in images:
        img = cv2.resize(img, output_size)
        out.write(img)
        if verbose:
            pbar.update()
    out.release()

def get_paths(directory, input_type='png', sort=True):
    """
        Get a list of input_type paths
        params args:
        return: a list of paths
    """
    paths = glob(os.path.join(directory, '*.{}'.format(input_type)))
    if sort:
        paths = list(sorted(paths))
    print('Found and sorted {} files {}'.format(len(paths), input_type))
    return paths

# Cell
def video_to_images(input_video, output_dir=None, skip=1):
    """
        Extract video to image:
            inputs:
                input_video: path to video
                output_dir: default is set to video name
    """
    import cv2
    import os
    import concurrent
    from imutils.video import count_frames
    if output_dir is None:
        output_dir = input_video.split('.')[0]
        print('Set output_dir =',output_dir)

    skip = int(skip)
    # Read the video from specified path
    cam = cv2.VideoCapture(input_video)
    total_frames = count_frames(input_video)
    os.makedirs(output_dir, exist_ok=True)
    # frame
    currentframe = 0
    # while(True):
    with concurrent.futures.ProcessPoolExecutor() as e:
        f_results = []
        for current_frame in range(0, total_frames, skip):
            # reading from frame
            ret,frame = cam.read()

            if ret:
                # if video is still left continue creating images
                name =  os.path.join(output_dir,f'{current_frame:05d}' + '.jpg')
                if currentframe % skip == 0:
                    f_results.append(e.submit(cv2.imwrite, name, frame))
            else:
                break
        for result in tqdm(concurrent.futures.as_completed(f_results)):
            pass
    cam.release()
    cv2.destroyAllWindows()

#export
@call_parse
def av_v2i(input_video:Param("", str), output_dir:Param("", str)=None, skip:Param("", int)=1):
    return video_to_images(input_video, output_dir, skip)

# Cell
from mmcv import Timer
import pandas as pd
import numpy as np
class TimeLoger:
    def __init__(self):
        self.timer = Timer()
        self.time_dict = dict()

    def start(self):
        self.timer.start()

    def update(self, name):
        # assert not name in self.time_dict
        duration = self.timer.since_last_check()
        if name in self.time_dict:
            self.time_dict[name].append(duration)
        else:
            self.time_dict[name] = [duration]

    def __str__(self):
        total_time = np.sum([np.sum(v) for v in self.time_dict.values()])
        s = f"------------------Time Loger Summary : Total {total_time:0.2f} ---------------------:\n"
        for k, v in self.time_dict.items():
            average = np.mean(v)
            times = len(v)
            percent = np.sum(v)*100/total_time
            # print()
            s += f'\t\t{k}:  \t\t{percent:0.2f}% ({average:0.4f}s) | Times: {times} \n'
        # print('-----------------------------------------------------------')
        return s

# Cell
import xxhash
import pickle

def identify(x):
    '''Return an hex digest of the input'''
    return xxhash.xxh64(pickle.dumps(x), seed=0).hexdigest()


def memoize(func):
    import os
    import pickle
    from functools import wraps

    import xxhash

    '''Cache result of function call on disk
    Support multiple positional and keyword arguments'''
    @wraps(func)
    def memoized_func(*args, **kwargs):
        cache_dir = 'cache'
        try:
            import inspect
            func_id = identify((inspect.getsource(func), args, kwargs))
            cache_path = os.path.join(cache_dir, func.__name__+'_'+func_id)

            if (os.path.exists(cache_path) and
                    not func.__name__ in os.environ and
                    not 'BUST_CACHE' in os.environ):
                result = pickle.load(open(cache_path, 'rb'))
            else:
                result = func(*args, **kwargs)
                os.makedirs(cache_dir, exist_ok=True)
                pickle.dump(result, open(cache_path, 'wb'))
            return result
        except (KeyError, AttributeError, TypeError):
            return func(*args, **kwargs)
    return memoized_func