"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('custom domain name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            customDomain: {
                domainName: 'test-domain.example.com',
                certificate,
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'test-domain.example.com',
            CustomDomainConfig: {
                CertificateArn: 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8',
            },
        });
    });
    test('cognito domain prefix', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'cognito-domain-prefix',
        });
    });
    test('fails when neither cognitoDomain nor customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
            customDomain: {
                domainName: 'mydomain.com',
                certificate,
            },
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when both cognitoDomain and customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when domainPrefix has characters outside the allowed charset', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addDomain('Domain1', {
            cognitoDomain: { domainPrefix: 'domain.prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain2', {
            cognitoDomain: { domainPrefix: 'Domain-Prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain3', {
            cognitoDomain: { domainPrefix: 'dómäin-prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
    });
    test('does not fail when domainPrefix is a token', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const parameter = new core_1.CfnParameter(stack, 'Paraeter');
        expect(() => pool.addDomain('Domain', {
            cognitoDomain: { domainPrefix: parameter.valueAsString },
        })).not.toThrow();
    });
    test('custom resource is added when cloudFrontDomainName property is used', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const domain = pool.addDomain('Domain', {
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // WHEN
        const cfDomainName = domain.cloudFrontDomainName;
        // THEN
        expect(stack.resolve(cfDomainName)).toEqual({
            'Fn::GetAtt': [
                'PoolDomainCloudFrontDomainName340BF87E',
                'DomainDescription.CloudFrontDistribution',
            ],
        });
        expect(stack).toHaveResource('Custom::UserPoolCloudFrontDomainName');
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [{
                        Action: 'cognito-idp:DescribeUserPoolDomain',
                        Effect: 'Allow',
                        Resource: '*',
                    }],
                Version: '2012-10-17',
            },
        });
    });
    test('cloudFrontDomainName property can be called multiple times', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const domain = pool.addDomain('Domain', {
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        const cfDomainNameFirst = domain.cloudFrontDomainName;
        const cfDomainNameSecond = domain.cloudFrontDomainName;
        expect(cfDomainNameSecond).toEqual(cfDomainNameFirst);
    });
    test('import', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const client = lib_1.UserPoolDomain.fromDomainName(stack, 'Domain', 'domain-name-1');
        // THEN
        expect(client.domainName).toEqual('domain-name-1');
        expect(stack).not.toHaveResource('AWS::Cognito::UserPoolDomain');
    });
    describe('signInUrl', () => {
        test('returns the expected URL', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'Pool');
            const domain = pool.addDomain('Domain', {
                cognitoDomain: {
                    domainPrefix: 'cognito-domain-prefix',
                },
            });
            const client = pool.addClient('Client', {
                oAuth: {
                    callbackUrls: ['https://example.com'],
                },
            });
            // WHEN
            const signInUrl = domain.signInUrl(client, {
                redirectUri: 'https://example.com',
            });
            // THEN
            expect(stack.resolve(signInUrl)).toEqual({
                'Fn::Join': [
                    '', [
                        'https://',
                        { Ref: 'PoolDomainCFC71F56' },
                        '.auth.',
                        { Ref: 'AWS::Region' },
                        '.amazoncognito.com/login?client_id=',
                        { Ref: 'PoolClient8A3E5EB7' },
                        '&response_type=code&redirect_uri=https://example.com',
                    ],
                ],
            });
        });
        test('correctly uses the signInPath', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'Pool');
            const domain = pool.addDomain('Domain', {
                cognitoDomain: {
                    domainPrefix: 'cognito-domain-prefix',
                },
            });
            const client = pool.addClient('Client', {
                oAuth: {
                    callbackUrls: ['https://example.com'],
                },
            });
            // WHEN
            const signInUrl = domain.signInUrl(client, {
                redirectUri: 'https://example.com',
                signInPath: '/testsignin',
            });
            // THEN
            expect(signInUrl).toMatch(/amazoncognito\.com\/testsignin\?/);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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